/**
 * $Id: CopyPropertyTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.Property;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ExportScope;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.go.Unless;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.parameters.TransformEnabled;
import  com.idaremedia.antx.parameters.TransformHelper;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.ValueTransform;

/**
 * Brain-dead Property extension to support flexible value-source attributes
 * while allowing all those "only-allow-nested-Properties" tasks to work properly. Unlike
 * a regular property a <i>CopyPropertyTask</i> can initialize its property from an literal
 * value (like standard), another property's value, a variable property's value, or
 * a stringifiable project reference. Usually defined &lt;copyproperty&gt;.
 * <p>
 * This task pre-dates the &lt;assignimport&gt; task, so some of its variable-support
 * functionality now overlaps with that task.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;copyproperty name="sourcefile" value="${ant.file}"/&gt;
 *    &lt;copyproperty name="someProperty" property="delayed.created.property"/&gt;
 *    &lt;copyproperty name="outdir" value="${tempdir}/${user.name}" transform="ospath"/&gt;
 *
 *    &lt;assign var="theVariable" value="100"/&gt;...
 *    &lt;copyproperty name="regularProperty" variable="theVariable"/&gt;
 *
 *    &lt;copyproperty name="val.lowercase" value="val.MixedCase" transform="lowercase"/&gt;
 *    &lt;copyproperty name="system.property" variable="theVariable" scope="all"/>
 *
 *    &lt;call targets="a,b,c"&gt;
 *       &lt;property name="property1" variable="variable1"/&gt;
 *       &lt;property name="property2" property="property2"/&gt;
 *       &lt;property name="property3" reference="source.file"/&gt;
 *    &lt;/call&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1 (moved from antx since AntX 0.2)
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      CopyReferenceTask
 **/

public final class CopyPropertyTask extends Property
    implements FlexValueSupport, TrimEnabled, TransformEnabled, AntLibFriendly
{
    /**
     * Initializes new blank CopyPropertyTask.
     **/
    public CopyPropertyTask()
    {
        super();
    }


    /**
     * Update this task's project and any dependent project component.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_XValue.setProject(P);
    }


    /**
     * Adds an unless-condition to this property task.
     * @since JWare/AntX 0.3
     **/
    public void setUnless(String property)
    {
        m_unlessProperty = (property==null) ? "" : property;
    }


    /**
     * Returns this task's (raw) unless-condition if any.
     * Returns empty string if condition never set.
     * @since JWare/AntX 0.3
     **/
    public final String getUnlessProperty()
    {
        return m_unlessProperty;
    }



    /**
     * Tells this task to trim the source value before
     * saving to the target.
     * @since JWare/AntX 0.3
     **/
    public void setTrim(boolean b)
    {
        m_XValue.setTrim(b);
    }


    /**
     * Returns <i>true</i> if this task will trim the value
     * before saving to the target.
     * @since JWare/AntX 0.3
     **/
    public boolean willTrim()
    {
        return m_XValue.isTrimmed();
    }


    /**
     * Sets the value transform for the value copied to
     * the target.
     * @since JWare/AntX 0.3
     **/
    public final void setTransform(ValueTransform t)
    {
        m_T = t==null ? ValueTransform.NONE
            : ValueTransform.from(t.getValue())/*normalize*/;
    }


    /**
     * Returns the value transformation this task will
     * perform on the to-be-copied value. Will return
     * {@linkplain ValueTransform#NONE} if never set.
     * @since JWare/AntX 0.3
     **/
    public final ValueTransform getTransform()
    {
        return m_T;
    }




    /**
     * Sets a literal value for this task to copy to a
     * local property. This is the inherited behavior.
     * @param value value of property (substitution done)
     **/
    public void setValue(String value)
    {
        super.setValue(value);
        m_XValue.set(value);
        m_XValue.setIsLiteral();//NB:clears all is-a flags
    }


    /**
     * Sets name of variable property to copy.
     * @param variable name of variable property (non-null)
     * @since JWare/AntX 0.2
     **/
    public void setVariable(String variable)
    {
        setValue(variable);
        m_XValue.setIsExported(true);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public final void setVar(String variable)
    {
        setVariable(variable);
    }


    /**
     * Returns <i>true</i> if this task will copy the contents
     * of a variable to a local property.
     * @since JWare/AntX 0.2
     **/
    public final boolean isVariable()
    {
        return m_XValue.isExported();
    }


    /**
     * Sets name of local property to copy.
     * @param property name of property (non-null)
     * @since JWare/AntX 0.2
     **/
    public void setProperty(String property)
    {
        setValue(property);
        m_XValue.setIsProperty(true);
    }


    /**
     * Returns <i>true</i> if this task will copy the contents
     * of one local property to another local property.
     * @since JWare/AntX 0.2
     **/
    public final boolean isProperty()
    {
        return m_XValue.isProperty();
    }


    /**
     * Sets name of local reference to copy.
     * @param refid identity of reference object (non-null)
     * @since JWare/AntX 0.2
     **/
    public void setReference(String refid)
    {
        setValue(refid);
        m_XValue.setIsReference(true);
    }


    /**
     * Returns <i>true</i> if this task will copy the stringified
     * version of a project reference to a local property.
     * @since JWare/AntX 0.2
     **/
    public final boolean isReference()
    {
        return m_XValue.isReference();
    }



    /**
     * Sets this new-property operation's scope to either the
     * project level (normal) or system level.
     * @param scope operation's scope ("system" or "project")
     * @since JWare/AntX 0.5
     **/
    public void setScope(String scope)
    {
        m_scope = ExportScope.from(scope,ExportScope.PROJECT);
    }



    /**
     * Returns <em>current</em> value of underlying thing being
     * copied (literal, property, variable, or reference) <em>or</em>
     * if source is undefined, returns the source's name. Will
     * return <i>null</i> if a source has never been defined.
     **/
    public final String getValue()
    {
        if (m_XValue.isUndefined()) {
            return null;
        }
        String xvalue = m_XValue.getValue(getProject());
        if (xvalue==null) {
            return m_XValue.get();//The *name* of the source as-is!
        }

        ValueTransform t= getTransform();
        if (t!=ValueTransform.NONE) {//done!
            return TransformHelper.apply(t,xvalue,getProject());
        }
        return xvalue;
    }



    /**
     * Returns <i>true</i> if this property is <em>just</em> a simple
     * a one(value)-to-one(property) definition.
     **/
    public final boolean isSimpleAssignment()
    {
        return (getFile()==null     &&
                getResource()==null &&
                getUrl()==null      &&  /*@since Ant1.6*/
                getRefid()==null    &&
                !m_XValue.isUndefined());
    }



    /**
     * Tries to initialize a project property based on this task's
     * specified value source. If multiple property sources have been
     * defined (like "file" <i>and</i> "resource") this method defaults
     * to the inherited behavior.
     * @throws BuildException if is a system property set and ant process
     *        does not have permission.
     **/
    public void execute()
    {
        /**
         * Heavy-handed: check if is a simple value->property
         * assignment. No combinations are supported.
         **/
        if (Unless.allowed(m_unlessProperty,getProject())) {
            if (getName()!=null && isSimpleAssignment()) {
                if (!ExportScope.ALL.equals(m_scope)) {
                    addProperty(getName(), getValue());
                }
                else {//@since AntX 0.5
                    String name1 = getName();
                    String valu = getValue();
                    try {
                        System.setProperty(name1, valu);
                    } catch(SecurityException secX) {
                        throw new BuildException(secX,getLocation());
                    }
                    AntXFixture.unsetProperty(name1);
                    getProject().setInheritedProperty(name1,valu);
                }
            } else {
                super.execute();
            }
        }
    }


    private FlexString m_XValue = new FlexString();
    private String m_unlessProperty="";
    private ValueTransform m_T= ValueTransform.NONE;
    private ExportScope m_scope = ExportScope.PROJECT;
}

/* end-of-CopyPropertyTask.java */
