/**
 * $Id: EchoErrorTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.IOException;
import  java.io.OutputStream;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ErrorSnapshot;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.EchoThingTask;

/**
 * Helper task to display contents of an {@linkplain ErrorSnapshot} or a BuildException.
 * Mostly used for debugging or during unit testing. Also useful if the AntX print
 * system is not being used.
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;<b>printerror</b> message="javadoc-error" snapshotid="javadoc-error"/&gt;
 *
 *   &lt;<b>printerror</b> message="cvs-error" thrownid="cvs-err"/&gt;
 *
 *   &lt;<b>printerror</b> msgid="err.javadoc" snapshotid="javadoc-error"
 *       tofile="${logs.root}/docs/javadoc.err" append="true" if="log.errors"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class EchoErrorTask extends EchoThingTask
{
    /**
     * Initializes new error echo task.
     **/
    public EchoErrorTask()
    {
        super(AntX.print);
    }


    /**
     * Initializes enclosed CV-labeled echo task.
     **/
    public EchoErrorTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets the error snapshot reference id for this task. This reference
     * is evaluated when the task is executed.
     **/
    public void setSnapshotId(String refId)
    {
        setThingRefId(refId);
    }


    /**
     * Returns the error snapshot's reference id; can be <i>null</i>
     * if never set. Required for task to execute properly.
     **/
    public final String getSnapshotRefId()
    {
        return getThingRefId();
    }


    /**
     * Sets the exception reference id for this task. This reference
     * is evaluated when the task is executed.
     * @since JWare/AntX 0.3
     **/
    public void setThrownId(String refId)
    {
        setThingRefId(refId);
    }


    /**
     * Returns the throwable's reference id; can be <i>null</i>
     * if never set. Required for task to execute properly.
     * @since JWare/AntX 0.3
     **/
    public final String getThrownRefId()
    {
        return getThingRefId();
    }


    /**
     * Returns the names of our refid parameters.
     * @since JWare/AntX 0.4
     **/
    protected String getThingRefParameterName()
    {
        return "thrownid|snapshotid";
    }


    /**
     * Verifies that refered-to thing is either an exception or
     * an {@linkplain ErrorSnapshot}. Returns the generic object
     * reference if ok. Never return <i>null</i>.
     * @throws BuildException if refid undefined or does not
     *         refer-to either an Exception or a ErrorSnapshot
     * @.sideeffect Also verifies this task is in a valid project
     **/
    public final Object getReferencedErrorThing()
        throws BuildException
    {
        verifyInProject_("getErrorRef");

        String refid = getThingRefId();
        String error = null;

        Object o = getProject().getReference(refid);

        if (o==null) {
            error = uistrs().get("task.missing.refid", refid);
        }
        else if (!(o instanceof Exception) && !(o instanceof ErrorSnapshot)) {
            error = uistrs().get("task.bad.refid", refid,
                                 Exception.class.getName()+"|"+ErrorSnapshot.class.getName(),
                                 o.getClass().getName());
        }
        if (error!=null) {
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        return o;
    }


    /**
     * Sets the list of captured properties to include in the output.
     * Use '<i>all</i>' to display all captured properties. Use '<i>none</i>'
     * to skip outputing properties altogether. By default will dump all
     * captured properties. Using the empty string is same as using '<i>none</i>.'
     * @param nameList comma-delimited list of property names (non-null)
     **/
    public void setProperties(String nameList)
    {
        require_(nameList!=null,"setprops- nonzro list");
        m_propertiesList= nameList;
    }


    /**
     * Returns list of captured properties to be output. Returns
     * "<i>all</i>" if all properties will be output. Returns
     * "<i>none</i>" or the empty string if no properties will be output.
     **/
    public final String getPropertiesNameList()
    {
        return m_propertiesList;
    }


    /**
     * Writes diagnostic contents of an ErrorSnapshot or Throwable
     * to this task's specified output stream (file|Ant-log).
     **/
    protected void echoThing()
    {
        OutputStream out= getOutputStream();
        try {
            Object thing = getReferencedErrorThing();

            String name;
            if (thing instanceof ErrorSnapshot) {
                name = ((ErrorSnapshot)thing).getName();
            } else {
                name = Tk.leafNameFrom(thing.getClass());
            }

            DisplayRequest dr= new DisplayRequest(getProject(),name,thing);
            dr.setFilter(getPropertiesNameList());

            m_displayWorker.print(dr,out);

            if (tryAntLog(out)) {
                log(getAntLogString(out),getMsgLevel().getNativeIndex());
            }

        } catch(IOException ioX) {
            String ermsg = uistrs().get("task.echo.unable");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());

        } finally {
            try { out.close(); } catch(Exception X){/*burp*/}
            out=null;
        }
    }


    /**
     * Ensures we're in a valid target/project and have a valid snapshot
     * reference id.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);
        getReferencedErrorThing();//NB: ensures refid -> ErrorSnapshot|Exception
    }


    private String m_propertiesList= Strings.ALL;
    private final ErrorPrinter m_displayWorker = new ErrorPrinter();
}

/* end-of-EchoErrorTask.java */
