/**
 * $Id: FixturePassthru.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the various fixture elements that an execution
 * context can pass-along to a subcontext. The following list explains what
 * the various settings <em>typically</em> mean (refer to a task's description to get
 * task-specific details):<ul>
 *    <li><span class="src">none</span>: No fixture information should be passed on.</li>
 *    <li><span class="src">all</span>: All relevant fixture information should be
 *        passed on.</li>
 *    <li><span class="src">properties</span>: Only project properties should be
 *        passed on. Similar to the common "<span class="src">inheritAll</span>"
 *        parameter.</li>
 *    <li><span class="src">references</span>: Only project references should be
 *        passed on. Similar to the common "<span class="src">inheritRefs</span>"
 *        parameter.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 **/

public final class FixturePassthru extends EnumSkeleton
{
    /** Index of {@linkplain #ALL ALL}. **/
    public static final int ALL_INDEX = 0;
    /** Index of {@linkplain #NONE NONE}. **/
    public static final int NONE_INDEX = ALL_INDEX+1;
    /** Index of {@linkplain #PROPERTIES PROPERTIES}. **/
    public static final int PROPERTIES_INDEX = NONE_INDEX+1;
    /** Index of {@linkplain #REFERENCES REFERENCES}. **/
    public static final int REFERENCES_INDEX = PROPERTIES_INDEX+1;


    /** Singleton "<span class="src">all</span>" choice. **/
    public static final FixturePassthru ALL=
        new FixturePassthru("all",ALL_INDEX);

    /** Singleton "<span class="src">none</span>" choice. **/
    public static final FixturePassthru NONE=
        new FixturePassthru("none",NONE_INDEX);

    /** Singleton "<span class="src">properties</span>" choice. **/
    public static final FixturePassthru PROPERTIES=
        new FixturePassthru("properties",PROPERTIES_INDEX);

    /** Singleton "<span class="src">references</span>" choice. **/
    public static final FixturePassthru REFERENCES=
        new FixturePassthru("references",REFERENCES_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public FixturePassthru()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private FixturePassthru(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"all", "none",
                             "properties", "references"};
    };



    /**
     * Helper that converts a scalar to a known FixturePassthru.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static FixturePassthru from(int i)
    {
        if (i==ALL.index)        { return ALL; }
        if (i==PROPERTIES.index) { return PROPERTIES; }
        if (i==REFERENCES.index) { return REFERENCES; }
        if (i==NONE.index)       { return NONE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * FixturePassthru's index.
     * @param i the index to be matched
     * @param dflt the default FixturePassthru if necessary
     **/
    public static FixturePassthru from(int i, FixturePassthru dflt)
    {
        FixturePassthru choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known FixturePassthru
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either FixturePassthru's symbolic name or its index.
     **/
    public static FixturePassthru from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (ALL.value.equals(s))        { return ALL; }
                if (PROPERTIES.value.equals(s)) { return PROPERTIES; }
                if (REFERENCES.value.equals(s)) { return REFERENCES; }
                if (Strings.DEFAULT.equals(s))  { return PROPERTIES; }/*std*/
                if (NONE.value.equals(s))       { return NONE; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * FixturePassthru's name.
     * @param s the symbolic name to be matched
     * @param dflt the default FixturePassthru if necessary
     **/
    public static FixturePassthru from(String s, FixturePassthru dflt)
    {
        FixturePassthru choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-FixturePassthru.java */
