/**
 * $Id: FeedbackLevel.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the level of script-facing feedback a task might
 * implement. Script-facing feedback may or may not include developer-targetted
 * implementation diagnostics. Similarly, telling a task container to be quiet
 * may or may not affect its nested tasks (this too must be task defined). The
 * following list explains what the various settings mean:<ul>
 *    <li><span class="src">none</span>: No script-facing feedback should be generated.
 *        Whether this turns off developer diagnostics (ant -debug) is task defined.</li>
 *    <li><span class="src">verbose</span>: As much feedback should be generated as
 *        implemented. Whether this is the same as verbose developer diagnostics 
 *        is task defined.</li>
 *    <li><span class="src">normal</span>: The normal feedback should be done. Usually
 *        this limits feedback to alerts, warnings, or worse conditions.</li>
 *    <li><span class="src">quiet</span>: Feedback should be limited to serious but not
 *        necessarily critical problems. Whether this setting applies to tasks launched
 *        by the primary task is task defined (the lauched task might, for instance, have
 *        its own settings).</li>
 *    <li><span class="src">veryquiet</span>: All feedback should be dropped except that
 *        which precedes a build error. This instruction should apply (when possible) to 
 *        subtasks or external processes run on the primary task's behalf.</li>
 * </ul>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

public final class FeedbackLevel extends EnumSkeleton
{
    //@.impl ORDERING of static declarations is important! ------

    //1) Indices (in order)

    public static int VERBOSE_INDEX = 0;
    public static int NORMAL_INDEX = VERBOSE_INDEX+1;
    public static int QUIET_INDEX = NORMAL_INDEX+1;
    public static int VERYQUIET_INDEX = QUIET_INDEX+1;
    public static int NONE_INDEX = VERYQUIET_INDEX+1;


    //2) Values (in order)

    /** Values in same order as public indices. **/
    private static final String[] VALUES_= new String[] {
        "verbose", "normal", "quiet", "veryquiet", "none"
    };


    //3) Singletons (depend on Indices and Values already existing!)

    /** Singleton "<span class="src">verbose</span>" level. **/
    public static final FeedbackLevel VERBOSE=
        new FeedbackLevel(VALUES_[VERBOSE_INDEX],VERBOSE_INDEX);

    /** Singleton "<span class="src">normal</span>" level. **/
    public static final FeedbackLevel NORMAL=
        new FeedbackLevel(VALUES_[NORMAL_INDEX],NORMAL_INDEX);

    /** Singleton "<span class="src">quiet</span>" level. **/
    public static final FeedbackLevel QUIET=
        new FeedbackLevel(VALUES_[QUIET_INDEX],QUIET_INDEX);

    /** Singleton "<span class="src">veryquiet</span>" level. **/
    public static final FeedbackLevel VERYQUIET=
        new FeedbackLevel(VALUES_[VERYQUIET_INDEX],VERYQUIET_INDEX);

    /** Singleton "<span class="src">none</span>" level. **/
    public static final FeedbackLevel NONE=
        new FeedbackLevel(VALUES_[NONE_INDEX],NONE_INDEX);



    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public FeedbackLevel()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private FeedbackLevel(String v, int i)
    {
        super(v);
    }



    /**
     * Returns a <em>copy</em> of the standard feedback levels
     * in order.
     * @param fillin [optional] array of strings to update with values.
     **/
    public static String[] copyOfDefaultValues(String[] fillin)
    {
        if (fillin==null) {
            fillin = new String[VALUES_.length];
        }
        System.arraycopy(VALUES_,0,fillin,0,VALUES_.length);
        return fillin;
    }



    /**
     * Returns copy of all possible feedback levels as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return FeedbackLevel.copyOfDefaultValues(null);
    };



    /**
     * Helper that converts a scalar to a known FeedbackLevel.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static FeedbackLevel from(int i)
    {
        if (i==NORMAL.index)    { return NORMAL; }
        if (i==QUIET.index)     { return QUIET; }
        if (i==VERYQUIET.index) { return VERYQUIET; }
        if (i==VERBOSE.index)   { return VERBOSE; }
        if (i==NONE.index)      { return NONE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * FeedbackLevel's index.
     * @param i the index to be matched
     * @param dflt the default FeedbackLevel if necessary
     **/
    public static FeedbackLevel from(int i, FeedbackLevel dflt)
    {
        FeedbackLevel choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known FeedbackLevel
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either FeedbackLevel's symbolic name or its index.
     **/
    public static FeedbackLevel from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (Strings.DEFAULT.equals(s))   { return NORMAL; }
                if (NORMAL.value.equals(s))      { return NORMAL; }
                if (QUIET.value.equals(s))       { return QUIET; }
                if (VERYQUIET.value.equals(s))   { return VERYQUIET; }
                if (VERBOSE.value.equals(s))     { return VERBOSE; }
                if (NONE.equals(s))              { return NONE; }
                if (Tk.string2NegBool(s)==Boolean.FALSE) {
                    return NONE;
                }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * FeedbackLevel's name.
     * @param s the symbolic name to be matched
     * @param dflt the default FeedbackLevel if necessary
     **/
    public static FeedbackLevel from(String s, FeedbackLevel dflt)
    {
        FeedbackLevel choice= from(s);
        return (choice==null) ? dflt : choice;
    }
    


    /**
     * Separates the loud feedback levels from the non-loud ones.
     * @param fbl feedback level under test (can be <i>null</i>)
     * @param nullIsQuiet <i>true</i> if a null level is considered quiet.
     * @return true if quiet-ish
     **/
    public static boolean isQuietish(FeedbackLevel fbl, boolean nullIsQuiet)
    {
        if (fbl==null) {
            return nullIsQuiet;
        }
        return fbl.getIndex()>=QUIET_INDEX;
    }
    


    /**
     * Separates the loud feedback levels from the non-loud ones.
     * Lets you specify whether <i>null</i> and NORMAL mean quiet.
     * @param fbl feedback level under test (can be <i>null</i>)
     * @param normalIsQuiet <i>true</i> if the normal level is considered quiet.
     * @param nullIsQuiet <i>true</i> if a null level is considered quiet.
     * @return true if quiet-ish
     **/
    public static boolean isQuietish(FeedbackLevel fbl, boolean normalIsQuiet,
        boolean nullIsQuiet)
    {
        if (fbl==null) {
            return nullIsQuiet;
        }
        if (fbl.getIndex()>=QUIET_INDEX) {
            return true;
        }
        return NORMAL_INDEX==fbl.getIndex() ? normalIsQuiet : false;
    }

}

/* end-of-FeedbackLevel.java */