/**
 * $Id: ScopedProperties.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Hashtable;
import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.ProjectDependent;
import  com.idaremedia.antx.helpers.GenericParameters;

/**
 * Common implementation of the {@linkplain org.apache.tools.ant.DynamicConfigurator
 * DynamicConfigurator} interface.
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   guarded
 * @.group    impl,helper
 * @see       com.idaremedia.antx.solo.LocalFixtureTaskSet LocalFixtureTaskSet
 **/

public final class ScopedProperties extends Hashtable
    implements ProjectDependent, Nameable
{
    private static final String IAM_= AntX.utilities+"ScopedProperties:";


    /**
     * Initializes a new supplemental properties dictionary. This
     * dictionary's project must be defined before it is installed.
     * @param commandProperties <i>true</i> if this helper's properties
     *        act like command properties
     **/
    public ScopedProperties(boolean commandProperties)
    {
        super();
        m_commandProperties = commandProperties;
    }



    /**
     * Initializes a new project-bound supplemental properties dictionary.
     * @param commandProperties <i>true</i> if this helper's properties
     *        act like command properties
     **/
    public ScopedProperties(Project project, boolean commandProperties)
    {
        super();
        m_commandProperties = commandProperties;
        setProject(project);
    }



    /**
     * Initializes this dictionary's enclosing project. Must be
     * called by this dictionary's controlling task <em>before</em>
     * {@linkplain #install install}.
     **/
    public void setProject(Project project)
    {
        m_project = project;
    }


    /**
     * Returns this dictionary's enclosing project. Will return
     * <i>null</i> if never set.
     **/
    public Project getProject()
    {
        return m_project;
    }



    /**
     * Initializes this dictionary's diagnostic label.
     * @param name the label (non-null)
     **/
    public void setName(String name)
    {
        m_label = name;
    }



    /**
     * Returns this dictionary's diagnostic label. Will
     * return the empty string if never defined explicitly.
     **/
    public String getName()
    {
        return m_label;
    }



    /**
     * Tells this dictionary whether <em>standalone</em> keys 
     * should be filtered and any underscores replaced with periods.
     * @param swapEm <i>true</i> to swap underscores.
     * @since JWare/AntX 0.5
     **/
    public final void swapUnderscores(boolean swapEm)
    {
        m_swapUnderscores = swapEm;
    }



    /**
     * Installs this dictionary as a supplemental set of project
     * properties until uninstalled. When this method returns, this
     * dictionary will be at the front of the hook chain.
     **/
    public final void install()
    {
        AntX.verify_(m_conduit==null,IAM_,"install- not installed");
        AntX.verify_(getProject()!=null,IAM_,"install- in project");

        if (!isEmpty()) {
            PropertyHelper ph = PropertyHelper.getPropertyHelper(getProject());
            Conduit c = new Conduit(m_commandProperties);
            synchronized(ph) {
                c.setNext(ph.getNext());
                ph.setNext(c);
            }
            m_conduit = c;
        }
    }



    /**
     * Uninstalls this dictionary as a supplemental set of project
     * properties. This method works even if other hooks have been
     * installed ahead of this dictionary in the hook chain.
     * @param notInstalled [optional] error handler
     **/
    public final void uninstall(ProblemHandler notInstalled)
    {
        if (!isEmpty() && m_conduit!=null) {
            Conduit c = (Conduit)m_conduit;
            m_conduit = null;
            PropertyHelper ph = PropertyHelper.getPropertyHelper(getProject());
            synchronized(ph) {
                PropertyHelper prevPh=null;
                do {
                    if (ph.getNext()==c) { break; }
                    prevPh = ph;
                    ph = ph.getNext();
                } while(ph!=null);
                if (ph!=null) {
                    ph.setNext(c.getNext());
                    if (prevPh!=null) {
                        prevPh.setNext(ph);
                    }
                } else {
                    String warning= AntX.uistrs().get("fixture.supPh.not.instald",getName());
                    if (notInstalled!=null) {
                        notInstalled.problem(warning,Project.MSG_WARN);
                    } else {
                        getProject().log(warning,Project.MSG_WARN);
                    }
                }
                c.setNext(null);
            }
        }
    }



    /**
     * Adds the given attribute to this dictionary ensuring all
     * underscores in <span class="src">key</span> have been
     * replaced by dots.
     **/
    public final Object put(Object key, Object value)
    {
        AntX.require_((key instanceof String),IAM_,"put- nonzro name");
        String s = (String)key;
        if (m_swapUnderscores) {
            s = s.replace('_','.');
        }
        return super.put(s,value);
    }



    /**
     * Convenient variant of {@linkplain #put(Object,Object)
     * put(&#8230;)} that processes a standard Ant property set.
     **/
    public final void put(PropertySet ps)
    {
        AntX.require_(ps!=null,IAM_,"put- nonzro propertyset");
        Iterator itr= ps.getProperties().entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e= (Map.Entry)itr.next();
            super.put(e.getKey(),e.getValue());//NB:skip name swapping
        }
    }



    /**
     * Convenient variant of {@linkplain #put(Object,Object)
     * put(&#8230;)} that processes a standard Ant property declaration.
     * Only works for simple, single-value properties.
     **/
    public final void put(Property p)
    {
        AntX.require_(p!=null,IAM_,"put- nonzro property");
        String name  = p.getName();
        String value = p.getValue();
        if (name!=null && value!=null) {
            super.put(name,value);//NB:skip name swapping
        }
    }



    /**
     * Convenient variant of {@linkplain #put(Object,Object)
     * put(&#8230;)} that processes a standard AntX property list.
     * @param params generic parameter name-value pairs
     * @since JWare/AntX 0.5
     **/
    public final void put(GenericParameters params)
    {
        AntX.require_(params!=null,IAM_,"put- nonzro propertyset");
        Iterator itr= params.copyOfSimpleKeyValues(getProject(),true)
            .entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e= (Map.Entry)itr.next();
            super.put(e.getKey(),e.getValue());
        }
    }



    /**
     * Ensure a "get" handles the AntX data-fetch URL format for values.
     * @param name the property to lookup (non-null)
     **/
    private Object lookup(String name)
    {
        String s = (String)get(name);
        if (s!=null) {
            String actual = FixtureExaminer.findValue(getProject(),s,name);
            if (actual!=null) {
                s = actual;
            }
        }
        return s;
    }



    /**
     * ScopedProperties hook into the standard Ant project property framework.
     * This class must play nice with other installed hooks.
     *
     * @since     JWare/AntX 0.4
     * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version   0.5
     * @.safety   guarded (inherited)
     * @.group    impl,helper
     */
    private class Conduit extends PropertyHelper
    {
        private final boolean m_userOnly;

        Conduit(boolean userOnly) {
            m_userOnly= userOnly;
        }

        /**
         * Returns our overlay properties if necessary. Will delegate
         * to next in hook chain if necessary.
         **/
        public Object getPropertyHook(String ns, String name,
                                      boolean userOnly) {
            Object o;
            if (userOnly && !m_userOnly) {
                o= null;
            } else {
                o= ScopedProperties.this.lookup(name);
            }
            if (o==null && getNext()!=null) {
                o = getNext().getPropertyHook(ns,name,userOnly);
            }
            return o;
        }

        /**
         * Updates our overlay properties if necessary. Handles only our
         * properties. Will delegate to next in hook chain if necessary.
         * @.impl The standard Ant PropertyHelper is _BROKEN_ in how it
         *        calls setPropertyHook from 'set[New]Property' (it does not
         *        call the get hook to test if property already exists
         *        before setting unconditionally). We have to compensate
         *        for this. AntX's own net implementation does not need
         *        this work around.
         **/
        public boolean setPropertyHook(String ns, String name, Object value,
                                       boolean inherited, boolean userOnly,
                                       boolean isNew) {

            if (ScopedProperties.this.containsKey(name)) {
                if ((m_userOnly && (inherited || userOnly)) ||
                    (!m_userOnly && !inherited && !userOnly)) {
                    ScopedProperties.this.put(name,value);
                }
                return true;//@.impl fixes problem (even if not altered)
            }
            if (getNext()!=null) {
                return getNext().setPropertyHook(ns,name,value,inherited,
                                                 userOnly,isNew);
            }
            return false;
        }
    }


    private Project m_project;
    private boolean m_commandProperties;
    private Object  m_conduit;
    private String  m_label="n/d";
    private boolean m_swapUnderscores;//NB: NO (as of AntX 0.5!)
}

/* end-of-ScopedProperties.java */
