/**
 * $Id: ProjectPropertiesNet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Collections;
import  java.util.Hashtable;
import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.ProjectDependent;

/**
 * A <span class="src">ProjectPropertiesNet</span> is usually installed for a
 * well-defined "execution bubble" to block scoped changes from affecting the
 * caller's environment. This class is not written as a complete substitute for
 * the standard Ant PropertyHelper (partially because that class is officially
 * still in flux as is likely to change dramatically in the future).
 * <p>
 * A ProjectPropertiesNet's name and project should be set once by the controlling
 * task as these methods are not guarded against concurrent use/modification.
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   guarded (client APIs only)
 * @.group    impl,helper
 * @.pattern  GoF.Proxy
 * @.expects  Ant 1.6+
 * @.caveat   The PropertyHelper must be manually installed by the controlling
 *            task.
 * @.caveat   As of 1.6.1 custom PropertyHelpers are <em>not</em> propagated to
 *            child projects.
 * @.caveat   Properties provided by a local hook configuration task are not
 *            included in returned property sets (like {@linkplain #getProperties}).
 * @.impl     This net must be workable with our own {@linkplain ScopedProperties}!
 * @.impl     Direct access to project references map is intentional to avoid
 *            the spurious warnings about replaced references.
 **/

public final class ProjectPropertiesNet extends PropertyHelper
    implements ProjectDependent, Nameable
{
    private static final String IAM_ = AntX.utilities+"PropertiesNet";


    /**
     * Reference under which a custom project PropertyHelper
     * will be installed.
     */
    public static final String ANT_HELPER_REFID= "ant.PropertyHelper";



    /**
     * Initializes a new properties net. This net's project
     * must be defined before it is installed.
     **/
    public ProjectPropertiesNet()
    {
        super();
    }



    /**
     * Initializes and installs a new properties net.
     * @param name [optional] this net's script-facing label
     * @param project associated project (non-null)
     * @throws IllegalStateException if this net is not associated
     *         with a project.
     **/
    public ProjectPropertiesNet(String name, Project project)
    {
        super();
        setProject(project);
        if (name!=null) {
            setName(name);
        }
        install(project);
    }



    /**
     * Set this net's initial set of overlayed properties en-masse.
     * This method should be called <em>before</em> this net is installed.
     * Net assumes ownership of incoming properties map. Like any property,
     * empty seeded values are writable once.
     * @param properties initial collection of property values (non-null)
     * @param control <i>true</i> if these are control properties (readonly implied)
     * @param readonly <i>true</i> if these should be immutable properties
     * @since JWare/AntX 0.5
     * @.safety single
     **/
    public void seedProperties(Map properties, boolean control, boolean readonly)
    {
        AntX.require_(properties!=null,IAM_,"seed- nonzro property set");
        if (!properties.isEmpty()) {
            if (control) {
                m_controlProperties.putAll(properties);
                m_cliProperties.putAll(properties);
            } else if (readonly) {
                m_cliProperties.putAll(properties);
            } else {
                Iterator itr= properties.entrySet().iterator();
                while (itr.hasNext()) {
                    Map.Entry e = (Map.Entry)itr.next();
                    String value = (String)e.getValue();
                    if (value==null || value.length()==0) {
                        e.setValue(EMPTY);
                    }
                }
                m_rwProperties = properties;
            }
            m_allProperties.putAll(properties);
        }
    }



    /**
     * Associates this net with a project. This net still has
     * to be {@linkplain #install(Project) installed} explicitly.
     * @.safety single
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_project = project;
    }



    /**
     * Returns the project with which this net associated.
     * Can return <i>null</i> if not linked to any project.
     * @.safety single
     **/
    public Project getProject()
    {
        return m_project;
    }



    /**
     * Initializes this net's diagnostics-friendly name.
     * @param name net's name (non-null)
     * @.safety single
     **/
    public final void setName(String name)
    {
        AntX.require_(name!=null,IAM_,"setName- nonzro name");
        m_nameImpl = name;
    }



    /**
     * Returns this net's diagnostics label. Will return
     * "undefined" if never set explicitly.
     * @.safety single
     */
    public final String getName()
    {
        return m_nameImpl;
    }



    /**
     * Ensure this net is not already installed in the given project's
     * property helper (or hook chain). Prevents infinite (circular) lookups.
     **/
    private void verifyNotInChain_(PropertyHelper ph, Project P)
    {
        AntX.verify_(ph!=null,IAM_,"install- Ant env inited");
        do {
           if (ph==this) {
                String error = AntX.uistrs().get("fixture.ph.install.once",getName());
                P.log(error,Project.MSG_ERR);
                throw new BuildException(error);
           }
           ph = ph.getNext();
        }while (ph!=null);
    }



    /**
     * Installs this net as the given project's property helper.
     * @param project this net's associated project
     * @return <i>true</i> if net installed first time
     * @.safety single
     * @return <i>false</i> if already installed directly or indirectly
     **/
    public boolean install(Project project)
    {
        AntX.require_(project!=null,IAM_,"install- nonzro project");
        setProject(project);
        PropertyHelper current = PropertyHelper.getPropertyHelper(project);

        verifyNotInChain_(current,project);

        log("Installing...",Project.MSG_VERBOSE);
        m_defImpl = current;
        project.getReferences().put(ANT_HELPER_REFID,this);

        AntX.verify_(PropertyHelper.getPropertyHelper(project)==this,
                     IAM_,"install- installed properly");
        return true;
    }



    /**
     * Uninstalls this net as its project's property helper.
     * Will reinstall the helper that was there when this net
     * was installed.
     * @param notInstalled [optional] error handler
     * @throws BuildException if this net is not currently installed
     **/
    public void uninstall(ProblemHandler notInstalled)
    {
        Project P = getProjectNoNull();
        PropertyHelper current = PropertyHelper.getPropertyHelper(P);
        if (current==this) {
            log("Uninstalling...",Project.MSG_VERBOSE);
            P.getReferences().put(ANT_HELPER_REFID,m_defImpl);
            m_defImpl = null;
        } else if (current!=null) {
            String error = AntX.uistrs().get("fixture.ph.not.instald",getName());
            if (notInstalled!=null) {
                notInstalled.problem(error,Project.MSG_ERR);
            }
            throw new BuildException(error);
        }
    }



    /**
     * Installs a property definition if it isn't already 
     * installed from the command or control context.
     **/
    public synchronized boolean setProperty(String ns, String name,
                                            Object value, boolean verbose)
    {
        AntX.require_(name!=null,IAM_,"setProperty- nonzro name");
        boolean written = true;

        if (getUserProperty(ns,name)==null) {
            if (setPropertyHook(ns,name,value,false,false,false)) {
                if (verbose) {
                    log("Set local hook property '"+name+
                        "' -> "+value, Project.MSG_DEBUG);
                }
            } else {
                Object old = m_allProperties.put(name,value);
                if (old!=null && old!=EMPTY) {
                    log("Overwrote script property '"+name+"'");
                }
                else if (verbose) {
                    log("Set script property '"+name+
                        "' -> "+value, Project.MSG_DEBUG);
                }
            }
        } else {
            written = false;
            if (verbose) {
                log("Ignored script definition of '"+name+"'");
            }
        }
        return written;
    }



    /**
     * Installs a new property iff it is not already defined in
     * any context. Never installs new properties to local hooks.
     **/
    public synchronized void setNewProperty(String ns, String name,
                                            Object value)
    {
        if (getProperty(ns,name)!=null) {
            log("Ignored new script definition of '"+name+"'");
        } else {
            m_allProperties.put(name,value);
            log("Set script property '"+name+"' -> "+value,
                Project.MSG_DEBUG);
        }
    }



    /**
     * (Re)defines a command property. This method overrides
     * unconditionally any existing command or general property
     * definition.
     **/
    public synchronized void setUserProperty(String ns, String name,
                                             Object value)
    {
        if (setPropertyHook(ns,name,value,false,true,false)) {
            log("(Re)Set local hook CLI property '"+name+"' -> "+value);
        }
        else {
            m_cliProperties.put(name,value);
            m_allProperties.put(name,value);
            log("(Re)Set CLI property '"+name+"' -> "+value);
        }
    }



    /**
     * (Re)defines a control property. This method overrides
     * unconditionally any existing control, command, or general
     * property definition. (Inherited properties are also
     * considered command properties).
     **/
    public synchronized void setInheritedProperty(String ns, String name,
                                                  Object value)
    {
        if (setPropertyHook(ns,name,value,true,false,false)) {
            log("(Re)Set local hook CTRL property '"+name+"' -> "+value);
        }
        else {
            m_controlProperties.put(name,value);
            m_cliProperties.put(name,value);
            m_allProperties.put(name,value);
            log("(Re)Set CTRL property '"+name+"' -> "+value);
        }
    }



    private Object readLocalURIs(String ns, String name, boolean userOnly)
    {
        Project p = getProjectNoNull();
        if (FixtureExaminer.valueURIInterpreterInstalled(p)) {
            return null;
        }
        if ((ns==null || ns.length()==0) && !userOnly) {
            return FixtureExaminer.findValue(p,name,null);
        }
        return null;
    }



    /**
     * Return a property's value from a custom task-installed
     * hook or a non-property source if indicated. To return a
     * a reference or a variable as a property preface your key
     * with either "reference:" or "variable:".
     * Will return <i>null</i> if no matching item found.
     */
    public Object getPropertyHook(String ns, String name, boolean userOnly)
    {
        Object o;

        if (getOverride()!=null) {
            o = getOverride().getPropertyHook(ns,name,userOnly);
            if (o!=null) {
                return o;
            }
        }
        return readLocalURIs(ns,name,userOnly);
    }



    /**
     * Return a property's definition from any context. Will
     * return <i>null</i> if item undefined <em>or</em>
     * cleared.
     */
    public synchronized Object getProperty(String ns, String name)
    {
        Object o = getPropertyHook(ns,name,false);
        return o==null ? getProperty0(ns,name) : o;
    }



    /**
     * Returns a command or control property's definition. Will
     * return <i>null</i> if item undefined <em>or</em> cleared.
     */
    public synchronized Object getUserProperty(String ns, String name)
    {
        Object o = getPropertyHook(ns,name,true);
        return o==null ? getUserProperty0(ns,name) : o;
    }



    /**
     * Returns a copy of this net's full collection of
     * properties. Never returns <i>null</i>.
     */
    public synchronized Hashtable getProperties()
    {
        Hashtable copy = m_defImpl.getProperties();
        copy.putAll(m_allProperties);
        return copy;
    }



    /**
     * Returns a copy of this net's command and control
     * properties. Never returns <i>null</i>.
     */
    public synchronized Hashtable getUserProperties()
    {
        Hashtable copy = m_defImpl.getUserProperties();
        copy.putAll(m_cliProperties);
        return copy;
    }



    /**
     * Copies this net's control properties to another project.
     */
    public synchronized void copyInheritedProperties(Project other)
    {
        AntX.require_(other!=getProject(),IAM_,"copyInherited- not same");

        if (!m_controlProperties.isEmpty()) {
            Iterator itr = m_controlProperties.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mE = (Map.Entry)itr.next();
                String key = (String)mE.getKey();
                if (other.getUserProperty(key)==null) {
                    other.setInheritedProperty(key,(String)mE.getValue());
                }
            }
        }
        m_defImpl.copyInheritedProperties(other);//NB: won't override ours!
    }



    /**
     * Copies this net's non-inherited command properties to
     * another project.
     */
    public synchronized void copyUserProperties(Project other)
    {
        m_defImpl.copyUserProperties(other);
        Iterator itr= m_cliProperties.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry mE= (Map.Entry)itr.next();
            if (!m_controlProperties.containsKey(mE.getKey())) {
                other.setUserProperty((String)mE.getKey(),(String)mE.getValue());
            }
        }
    }



    /**
     * Returns a a copy of the properties captured by this 
     * net. Should be used <em>after</em> this net is uninstalled.
     * @param filtered <i>true</i> if results should be filtered
     *    using seed values
     * @param blocking <i>true</i> if the seed values should
     *    be removed; otherwise only seed values are returned.
     * @since JWare/AntX 0.5
     * @return final (optionally filtered) properties (non-null)
     **/
    public Map getFinalProperties(boolean filtered, boolean blocking)
    {
        Hashtable ht = new Hashtable(m_allProperties);
        if (filtered) {
            if (blocking) {
                ht.keySet().removeAll(m_rwProperties.keySet());
            } else {
                ht.keySet().retainAll(m_rwProperties.keySet());
                Iterator itr= ht.values().iterator();
                while (itr.hasNext()) {
                    if (itr.next()==EMPTY) {
                        itr.remove();
                    }
                }
            }
        }
        return ht;
    }



    /**
     * Returns this net's project ensuring that value has been
     * defined.
     * @throws IllegalStateException if this net has no project
     */
    private Project getProjectNoNull()
    {
        Project P = getProject();
        AntX.verify_(P!=null,IAM_,"getProj- inited");
        return P;
    }


    /**
     * Internal command property getter. Synchronization must be
     * done by public-facing API.
     */
    private Object getUserProperty0(String ns, String name)
    {
        if (name==null) {
           return null;
        }
        Object o = m_cliProperties.get(name);
        if (o==null) {
            o = m_defImpl.getUserProperty(ns,name);
        }
        return o;
    }


    /**
     * Internal general property getter. Synchronization must be
     * done by public-facing API.
     */
    private Object getProperty0(String ns, String name)
    {
        if (name==null) {
            return null;
        }
        Object o = m_allProperties.get(name);
        if (o==null) {
            o = m_defImpl.getProperty(ns,name);
        } else {
            if (o==EMPTY/*!*/) {//NB:allow for unset local properties!
                o = null;
            }
        }
        return o;
    }


    private void log(String msg, int noiselevel)
    {
        getProjectNoNull().log("PropertiesNet("+getName()+"): "+msg,
                               noiselevel);
    }


    private void log(String msg)
    {
        getProjectNoNull().log("PropertiesNet("+getName()+"): "+msg,
                               Project.MSG_VERBOSE);
    }


    private PropertyHelper getOverride()
    {
        return getNext();//stoopid name!
    }


    private static final String EMPTY= new String("");//NB:unique emptystring
    private Project m_project;
    private PropertyHelper m_defImpl;
    private String m_nameImpl="n/d";
    private Map m_allProperties = AntXFixture.newMap();//NB:public-API mt-safe
    private Map m_cliProperties = AntXFixture.newMap();
    private Map m_controlProperties = AntXFixture.newMap();
    private Map m_rwProperties = Collections.EMPTY_MAP;//@since AntX-0.5
}


/* end-of-ProjectPropertiesNet.java */
