/**
 * $Id: MkdirsTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.BufferedReader;
import  java.io.File;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.FileList;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.FeedbackLevel;
import  com.idaremedia.antx.parameters.RecoveryEnabled;
import  com.idaremedia.antx.starters.ListFriendly;

/**
 * Utility task that lets you create a slew of directories with compact instruction.
 * Bypasses need to create looping construct around a simple &lt;mkdir&gt; task.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;<b>mkdirs</b> base="${workspace}" names="sources,outputs,documents"/&gt;
 *    &lt;<b>mkdirs</b> base="${workspace}" namesfile="${layoutfile}"/&gt;
 *    &lt;<b>mkdirs</b> haltiferror="yes"&gt;
 *       &lt;filelist dir="${root}" files="build,release,workspace"/&gt;
 *       &lt;filelist dir="${root}/workspace" files="sources,outputs,documents"/&gt;
 *    &lt;/mkdirs&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class MkdirsTask extends AssertableTask implements RecoveryEnabled
{
    private static final int BY_NAME=0;
    private static final int BY_NAMELIST=BY_NAME+1;
    private static final int BY_NAMEFILE=BY_NAMELIST+1;


    /**
     * Initializes a new mkdirs task instance.
     **/
    public MkdirsTask()
    {
        super(AntX.mktemp+"MkdirsTask:");
    }


    /**
     * Tells this task whether problems making named directories
     * should signal a build error. Defaults "off" if never set.
     */
    public void setHaltIfError(boolean halt)
    {
        m_haltIfError = halt;
    }
    

    /**
     * Returns whether this task will signal a build error if
     * it encounters any problems making (or verifying) directory.
     * Returns "true" if never set.
     * @see #setHaltIfError setHaltIfError(&#8230;)
     */
    public boolean isHaltIfError()
    {
        return m_haltIfError;
    }


    /**
     * Sets this task's diagnostic feedback level. For no feedback
     * except build errors (haltiferror=true), set level to "none"
     * or "veryquiet".
     * @param fb the feedback level (non-null)
     * @throws BuildException if feedback level unrecognized.
     */
    public void setFeedback(String fb)
    {
        require_(fb!=null,"setFeedback- nonzro value");
        FeedbackLevel fbLevel = FeedbackLevel.from(fb);
        if (fbLevel==null) {
            String error = getAntXMsg("task.illegal.param.value",
                                     getTaskName(), fb, "feedback");
            log(error, Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_fbLevel = fbLevel;
    }


    /**
     * Factory method for a nested filelist element. Ensures new
     * filelist is added to this task's list of to-be-made
     * directories.
     * @return new filelist (never null)
     * @.impl We cannot default the filelist's -dir here because
     *        we must support the -refid parameter.
     */
    public FileList createFileList()
    {
        FileList fl = new FileList();
        fl.setProject(getProject());

        m_dirs.add(fl);
        return fl;
    }


    /**
     * Returns this task's builtin filelist. The builtin filelist
     * is configured using the various "names*" parameters. Never
     * returns <i>null</i>.
     **/
    private FileList builtinFileList()
    {
        if (m_builtin==null) {
            m_builtin = createFileList();
            m_builtin.setDir(getProject().getBaseDir());
        }
        return m_builtin;
    }



    /**
     * Sets this task's default base directory for all filelists.
     * @param baseDir the base directory (non-null)
     * @throws IllegalArgumentException if baseDir is <i>null</i>.
     **/
    public void setBase(File baseDir)
    {
        require_(baseDir!=null,"setBase- nonzro dirname");
        builtinFileList().setDir(baseDir);
    }



    /**
     * Sets this task's list of directories as a comma-delimited
     * list of names. Will update the builtin filelist only.
     * @param subdirNames the comma-delimited list (non-null)
     * @throws IllegalArgumentException if subdirNames is <i>null</i>.
     **/
    public void setNames(String subdirNames)
    {
        require_(subdirNames!=null,"setNames- nonzro namelst");
        setByMethod(BY_NAME,subdirNames);
    }


    /**
     * Pre-execution processing for builtin filelist when specified
     * as a comma-delimited list of directory names.
     **/
    private void handleByNames()
    {
        List nl = Tk.splitList(m_sourceString);
        if (!nl.isEmpty()) {
            FileList builtin = builtinFileList();
            for (int i=0,N=nl.size();i<N;i++) {
                FileList.FileName fn = new FileList.FileName();
                fn.setName((String)nl.get(i));
                builtin.addConfiguredFile(fn);
            }
        }
    }



    /**
     * Sets this task's list of directories as a reference to a
     * string list of names. Will update the builtin filelist only.
     * @param listsRefId the reference id of the string list (non-null)
     * @throws IllegalArgumentException if listsRefId is <i>null</i>.
     **/
    public void setNamesList(String listsRefId)
    {
        require_(listsRefId!=null,"setNamesList- nonzro refid");
        setByMethod(BY_NAMELIST,listsRefId);
    }



    /**
     * Pre-execution processing for the builtin filelist when
     * specified as an AntX StringItemList of some kind.
     * @throws BuildException if referred-to thing is not list-friendly.
     **/
    private void handleByNamesList()
    {
        ListFriendly list = (ListFriendly)getReferencedObject
            (null,m_sourceString,ListFriendly.class);
        if (!list.isEmpty()) {
            FileList builtin = builtinFileList();
            Iterator itr= list.readonlyStringIterator(getProject());
            while (itr.hasNext()) {
                FileList.FileName fn = new FileList.FileName();
                fn.setName(itr.next().toString());
                builtin.addConfiguredFile(fn);
            }
        }
    }



    /**
     * Sets this task's list of directories as an external listing 
     * file. Will update the builtin filelist only.
     * @param listsFile the path to the external file (non-null)
     * @throws IllegalArgumentException if listsFile is <i>null</i>.
     **/
    public void setNamesFile(String listsFile)
    {
        require_(listsFile!=null,"setNamesFile- nonzro filename");
        setByMethod(BY_NAMEFILE,listsFile);
    }



    /**
     * Pre-execution processing for the builtin filelist when
     * specified as an external listing file.
     * @throws BuildException if unable to read information for
     *              any reason and haltiferror=true.
     **/
    private void handleByNamesFile()
    {
        final Project myproject = getProject();
        File file = myproject.resolveFile(m_sourceString);
        if (file.exists()) {
            if (!file.canRead()) {
                String message = getAntXMsg("task.bad.configfile",file.getPath(),
                    "Cannot read file");
                if (isHaltIfError()) {
                    log(message,Project.MSG_ERR);
                    throw new BuildException(message,getLocation());
                }
                log(message,Project.MSG_WARN);
                return;
            }
            try {
                BufferedReader fr = new BufferedReader(new java.io.FileReader(file));
                FileList builtin = builtinFileList();
                FlexString filter = (FlexString)m_modifiers.clone();
                String next;
                while ((next=fr.readLine())!=null) {
                    filter.set(next);
                    next = filter.getValue();
                    if (next!=null) {
                        FileList.FileName fn = new FileList.FileName();
                        fn.setName(Tk.resolveString(myproject,next));
                        builtin.addConfiguredFile(fn);
                    }
                }
            } catch(Exception ioxOrbuildX) {
                String message = getAntXMsg("task.bad.configfile",file.getPath(),
                    ioxOrbuildX.getMessage());
                if (isHaltIfError()) {
                    log(message,Project.MSG_ERR);
                    throw new BuildException(message,getLocation());
                }
                log(message,Project.MSG_WARN);
            }
        }//exists
    }


    /**
     * Sets this task's builtin filelist's directory information.
     * Ensures only one shorthand method is defined.
     * @param method the shorthand method
     * @param source the information descriptor (non-null0
     * @throws BuildException if builtin filelist already defined.
     **/
    private void setByMethod(int method, String source)
    {
        if (m_byMethod>=BY_NAME) {
            String error = getAntXMsg("task.only.oneof.attr",
                                      "names|nameslist|namesfile");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_byMethod = method;
        m_sourceString = source;
    }



    /**
     * Creates all the named directories, including any parent dirs.
     * Does nothing to existing directories.
     * @throws BuildException if unable to create or verify all dirs
     *           and -haltiferror is turned on.
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        if (m_sourceString!=null) {
            switch(m_byMethod) {
                case BY_NAME: handleByNames(); break;
                case BY_NAMELIST: handleByNamesList(); break;
                default: handleByNamesFile();
            }
        } else if (m_builtin!=null) {
            m_dirs.remove(m_builtin);//OK, filelists already inited!
            m_builtin = null;
        }

        final Project myproject = getProject();
        Requester rqlink = new Requester.ForComponent(this);
        FileUtils fu = AntXFixture.fileUtils();

        for (int i=0,N=m_dirs.size();i<N;i++) {
            FileList fl = (FileList)m_dirs.get(i);
            String[] dirnames = fl.getFiles(myproject);
            File basedir = fl.getDir(myproject);
            for (int j=0;j<dirnames.length;j++) {
                File dir = fu.resolveFile(basedir,dirnames[j]);
                Mkdirs.run(dir,m_haltIfError,m_fbLevel,rqlink);
            }
        }
    }



    /**
     * Verifies that at least one directory named for creation. If 
     * empty an alert is issued unless feedback=none|*quiet.
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (!FeedbackLevel.isQuietish(m_fbLevel,false,false)) {
            if (m_sourceString==null && m_builtin!=null && m_dirs.size()==1) {
                String alert = getAntXMsg("mktemp.mkdirs.nothin");
                log(alert,Project.MSG_INFO);
            }
        }

        defaultFileListDirs();
    }



    /**
     * This is an obscene hack that is required because Ant's 
     * ordering of nested item creation and parent parameter 
     * initialization is (delicately speaking) as*-backwards.
     * @since JWare/AntX 0.5
     **/
    private void defaultFileListDirs()
    {
        if (m_builtin!=null) {
            final Project myproject = getProject();
            File defaultDir= m_builtin.getDir(myproject);
            if (defaultDir!=myproject.getBaseDir()) { //NB: identity-compare OK
                for (int i=0,N=m_dirs.size();i<N;i++) {
                    FileList fl= (FileList)m_dirs.get(i);
                    if (fl!=m_builtin && !fl.isReference() && fl.getDir(myproject)==null) {
                        fl.setDir(defaultDir);
                    }
                }
            }
        }
    }


    private boolean m_haltIfError=false;
    private List m_dirs = AntXFixture.newList();
    private FileList m_builtin;//NB:used by shorthand "names*" parameters
    private String m_sourceString;
    private int m_byMethod= -1;
    private FeedbackLevel m_fbLevel= FeedbackLevel.NORMAL;
    private FlexString m_modifiers = new FlexString();
}

/* end-of-MkdirsTask.java */