/**
 * $Id: ObjectHandle.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.io.Serializable;

/**
 * Luggage class to "pass-by-pointer" an object reference. One implementation of
 * Martin Fowler's "pass-in-result-object" pattern. Is as serializable as the underlying
 * object.
 *
 * @since    JWare/core 0.2
 * @author   ssmc, &copy;1997-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @.pattern Fowler.PassInResultObject
 **/

public class ObjectHandle implements Serializable
{
    /* -- seekrits -- */
    private Object m_underlying;


    /**
     * Create new object handle initialized to <i>null</i>.
     **/
    public ObjectHandle()
    {
        super();
        m_underlying=null;
    }


    /**
     * Create new object handle initialized to given object.
     **/
    public ObjectHandle(Object o)
    {
        super();
        m_underlying= o;
    }


    /**
     * Create new object handle for given <span class="src">char</span>.
     **/
    public ObjectHandle(char c)
    {
        this(new Character(c));
    }


    /**
     * Create new object handle for given <span class="src">integer</span>.
     **/
    public ObjectHandle(int i)
    {
        this(new Integer(i));
    }


    /**
     * Create new object handle for given <span class="src">long</span>.
     **/
    public ObjectHandle(long l)
    {
        this(new Long(l));
    }


    /**
     * Create new object handle for given <span class="src">float</span>.
     **/
    public ObjectHandle(float e)
    {
        this(new Float(e));
    }


    /**
     * Create new object handle for given <span class="src">double</span>.
     **/
    public ObjectHandle(double d)
    {
        this(new Double(d));
    }


    /**
     * Create new object handle for given <span class="src">boolean</span>.
     **/
    public ObjectHandle(boolean b)
    {
        this(b ? Boolean.TRUE : Boolean.FALSE);
    }


    /**
     * Returns underlying object (can be <i>null</i>).
     **/
    public Object get()
    {
        return m_underlying;
    }


    /**
     * Updates underlying object (can be set to <i>null</i>).
     * @param o new object
     **/
    public void set(Object o)
    {
        m_underlying= o;
    }


    /**
     * Updates underlying value to new <span class="src">integer</span> value.
     * @param i new value
     **/
    public void set(int i)
    {
        m_underlying= new Integer(i);
    }


    /**
     * Updates underlying value to new <span class="src">long</span> value.
     * @param l new value
     **/
    public void set(long l)
    {
        m_underlying= new Long(l);
    }


    /**
     * Updates underlying value to new <span class="src">float</span> value.
     * @param e new value
     **/
    public void set(float e)
    {
        m_underlying= new Float(e);
    }


    /**
     * Updates underlying value to new <span class="src">double</span> value.
     * @param d new value
     **/
    public void set(double d)
    {
        m_underlying= new Double(d);
    }


    /**
     * Updates underlying value to new <span class="src">boolean</span> value.
     * @param b new value
     **/
    public void set(boolean b)
    {
        m_underlying= b ? Boolean.TRUE : Boolean.FALSE;
    }


    /**
     * Updates underlying value to new <span class="src">char</span> value.
     * @param c new value
     **/
    public void set(char c)
    {
        m_underlying= new Character(c);
    }


    /**
     * Returns underlying object as a string (can be <i>null</i>).
     **/
    public String asString()
    {
        return (m_underlying==null) ? null : m_underlying.toString();
    }


    /**
     * Returns underlying object as a string or
     * "<span class="src">null</span>" if not defined.
     **/
    public String toString()
    {
        return String.valueOf(m_underlying);
    }

    /**
     * Returns underlying object if a non-null <span class="src">String</span>
     * or the empty string if it is <i>null</i>.
     **/
    public String asNonNullString()
    {
        return (m_underlying instanceof String) ? m_underlying.toString() : "";
    }


    /**
     * Returns whether underlying object is <i>null</i>.
     **/
    public final boolean isnull()
    {
        return (m_underlying==null) ? true : false;
    }


    /**
     * Determine if the given object's content is semantically equivalent
     * to this handle's content.
     * @param o object to compare (must be other ObjectHandle
     * @return <i>true</i> if equivalent
     **/
    public boolean equals(Object o)
    {
        if (o==null) {
            return false;
        }
        if (o==this) {
            return true;
        }
        if (getClass().equals(o.getClass())) {
            ObjectHandle oh= (ObjectHandle)o;
            if (m_underlying==null) {
                return oh.m_underlying==null;
            }
            if (oh.m_underlying==null) {
                return false;
            }
            return m_underlying.equals(oh.m_underlying);
        }
        return false;
    }


    /**
     * Returns hash value for this object handle. A handle's underlying
     * object determines its has value (so the hash value changes if
     * {@linkplain #set(java.lang.Object) set()} is used).
     **/
    public int hashCode()
    {
        return (m_underlying==null) ? 13 : m_underlying.hashCode();
    }
}

/* end-of-ObjectHandle.java */
