/**
 * $Id: NameValuePair.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.io.Serializable;
import  java.util.Map;

/**
 * Luggage class implementation of a name-value pair. <em>Security note</em>:
 * NameValuePairs are both <span class="src">Cloneable</span> and
 * <span class="src">Serializable</span> by definition.
 *
 * @since    JWare/core 0.5
 * @author   ssmc, &copy;1997-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      NameValuePair#NONE
 **/

public final class NameValuePair implements Cloneable, Serializable, Map.Entry
{
    /**
     * Initializes new blank name-value item; both name
     * and value are empty strings.
     **/
    public NameValuePair()
    {
        this("","");
    }


    /**
     * Initializes new name-value item with given values.
     **/
    public NameValuePair(String k, Object v)
    {
        m_nam = k;
        m_val = v;
    }


    /**
     * Symbolic singleton NameValuePair that can represent "no value"
     * or "empty".
     * @.pattern JWare.Placeholder
     **/
    public static final NameValuePair NONE = new NameValuePair();



    /**
     * Returns name portion of this name-value item.
     **/
    public String getName()
    {
        return m_nam;
    }


    /**
     * Updates name portion of this name-value item.
     **/
    public void setName(String k)
    {
        m_nam = k;
    }


    /**
     * <span class="src">Map.Entry</span> version of
     * {@linkplain #getName}.
     *  @since JWare/core 0.7
     **/
    public final Object getKey()
    {
        return getName();
    }


    /**
     * Returns value portion of this name-value item.
     **/
    public Object getValue()
    {
        return m_val;
    }


    /**
     * Returns value portion of this name-value item as a
     * <span class="src">boolean</span>.
     **/
    public boolean getBoolean()
    {
        return m_val==Boolean.TRUE ? true : false;
    }


    /**
     * Updates value portion of this name-value item.
     **/
    public Object setValue(Object v)
    {
        Object old = m_val;
        m_val = v;
        return old;
    }


    /**
     * Updates value to a <span class="src">boolean</span>.
     * Normalizes to use shared <span class="src'>Boolean.TRUE</span>
     * or <span class="src">Boolean.FALSE</span> settings.
     **/
    public Object setValue(boolean b)
    {
        Object old = m_val;
        m_val = b ? Boolean.TRUE : Boolean.FALSE;
        return old;
    }


    /**
     * Returns shallow clone of this name-value item; returned
     * clone references this item's name and value.
     **/
    public Object clone()
    {
        try {
            return super.clone();
        }
        catch (CloneNotSupportedException clnx) {
            throw new Error();
        }
    }


    /**
     * Determines if given item is semantically equivalent to
     * this item.
     **/
    public boolean equals(Object o)
    {
        if (o==this) {
            return true;
        }
        if (o!=null && o.getClass()==NameValuePair.class) {
            NameValuePair it= (NameValuePair)o;
            if (nameEquals(it.m_nam)) {
                if (m_val==null) {
                    return it.m_val==null;
                }
                return m_val.equals(it.m_val);
            }
        }
        return false;
    }


    /**
     * Returns a hash value for this item (depends on name
     * and value).
     **/
    public int hashCode()
    {
        int hc = 31;
        if (m_nam!=null) {
            hc = 31*hc + m_nam.hashCode();
        }
        if (m_val!=null) {
            hc = 31*hc + m_val.hashCode();
        }
        return hc;
    }


    /**
     * Check if this item's name is equivalent to given
     * string.
     **/
    public boolean nameEquals(String othernam)
    {
        if (m_nam==null) {
            return othernam==null;
        }
        if (othernam==null) {
            return m_nam==null;
        }
        return m_nam.equals(othernam);
    }


    /**
     * Returns the name of this name-value item.
     **/
    public String toString()
    {
        return m_nam;
    }

    private String m_nam;
    private Object m_val;
}

/* end-of-NameValuePair.java */
