/**
 * $Id: LocalExecutionBubble.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.wrap;

import  java.util.Hashtable;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;
import  org.apache.tools.ant.UnknownElement;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.ownhelpers.ProjectDependentSkeleton;
import  com.idaremedia.antx.ownhelpers.ProjectPropertiesNet;

/**
 * Execution helper that tries to capture and restore a project's immediate environment
 * from a set of local (scoped) changes.
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 * @see       ProjectPropertiesNet
 * @see       IsolatedTaskSet
 **/

public class LocalExecutionBubble extends ProjectDependentSkeleton
    implements ExecutionBubble
{
    private static final String IAM_= AntX.flow+"LocalBubble";


    /**
     * Initializes a new execution bubble.
     * @see #addFixtureReset addFixtureReset(&#8230;)
     **/
    public LocalExecutionBubble()
    {
        super();
    }


    /**
     * Initializes a new named execution bubble.
     * @param label this bubble script-facing label (non-null)
     * @since JWare/AntX 0.5
     **/
    public LocalExecutionBubble(String label)
    {
        super();
        m_label = label;
    }



    /**
     * Tells this bubble to examine object reference differences
     * carefully. If a reference changes (object reference or
     * content hash) a warning will be issued.
     * @param careful <i>true</i> to issue warnings for altered
     *        references
     **/
    public final void setCarefulObjectChecks(boolean careful)
    {
        m_isCareful = careful;
    }



    /**
     * Returns <i>true</i> if this bubble will examine reference
     * differences carefully and issue warnings for changed
     * objects. Is <i>false</i> by default.
     **/
    public final boolean willBeCareful()
    {
        return m_isCareful;
    }



    /**
     * Tells this bubble to update a project property w/ a warning
     * count if any problems encountered. Must be in "careful"
     * mode.
     * @param property name of property to update
     **/
    public final void setWarningsUpdateProperty(String property)
    {
        m_updateProperty = property;
    }



    /**
     * Tells this bubble to reset a particular fixture component
     * on leave. Use this method carefully as it impacts information
     * that spans the entire build-iteration!
     * @param component the fixture component
     * @param aspect the aspect to reset (use "all" for everything)
     **/
    public void addFixtureReset(String component, String aspect)
    {
        if (m_killMethods==null) {
            m_killMethods = AntXFixture.newMap();
        }
        m_killMethods.put(component,aspect);
    }



    /**
     * Adds another inner bubble to this bubble. This
     * hook allows different tasks to add a little setup and
     * a little teardown to the mass of standard processing. The
     * custom bubble is nested "inside" this bubble so its
     * enter method is called <em>after</em> this bubble has setup
     * and its leave method is called <em>before</em> this
     * bubble has reset everything.
     * @param innerBubble custom setup/teardown helper
     **/
    public void setNestedBubble(ExecutionBubble innerBubble)
    {
        m_innerLayer = innerBubble;
    }



    /**
     * Assigns a set of filters that can make this bubble
     * porous. The incoming controls should have been removed from
     * the project's reference table so it is no longer available
     * to other bubbles. This method must be called <em>before</em> 
     * this bubble is entered.
     * @param controls the filter controls
     * @since JWare/AntX 0.5
     **/
    public final void setFilterControls(Locals controls)
    {
        AntX.verify_(m_propertiesNet==null,IAM_,"setCtrls- not active");
        m_controls = controls;
    }


    /**
     * Returns <i>true</i> if this bubble has already been
     * assigned a set of local filters.
     * @see #setFilterControls setFilterControls()
     * @since JWare/AntX 0.5
     **/
    public final boolean isFiltered()
    {
        return m_controls!=null;
    }



    /**
     * Tells this bubble whether default fixture exclusions should
     * be ignored. For backward compatibility, fixture exclusions 
     * are ignored unless explicitly turned on.
     * @param letEmPassThru <i>true</i> to let exclusion thru
     * @since JWare/AntX 0.5
     **/
    public void setFixtureExcludes(boolean letEmPassThru)
    {
        m_defaultExcludes = letEmPassThru;
    }



    /**
     * Captures the calling task's surrounding project state
     * for later restoration. Must be matched with a call to
     * {@linkplain #leave leave}.
     * @param task the controlling task (non-null)
     **/
    public boolean enter(Requester task)
    {
        Project project = task.getProject();
        AntX.verify_(project!=null,IAM_,"enter- inited task");

        setProject(project);
        
        captureEnvironment(project, task);
        applyLocals();

        boolean ok=true;
        if (m_innerLayer!=null) {
            ok = m_innerLayer.enter(task);
            if (!ok) {
                restoreEnvironment(task);
            }
        }
        return ok;
    }



    /**
     * Restores the project's state as best it can to its state before the
     * task was executed. Will affect project properties and references
     * mostly. Should be done <em>once</em> for each call to
     * {@linkplain #enter enter}.
     * @param task the controlling task (non-null)
     * @throws org.apache.tools.ant.BuildException if unable to restore
     *         project's PropertyHelper.
     **/
    public boolean leave(Requester task)
    {
        Project project= task.getProject();
        AntX.verify_(getProject()==project,IAM_,
                     "leave- project same as enter()");

        boolean ok= m_innerLayer==null;
        try {
            if (m_innerLayer!=null) {
               ok = m_innerLayer.leave(task);
            }
        } finally {
            restoreEnvironment(task);
            resetIterationVarStacks(task);
        }
        return ok;
    }



    /**
     * Captures the standard project fixture bits only. The order
     * of capture is important with the properties net being cast
     * last.
     * @param project target project (non-null)
     * @param task call controller (non-null)
     * @since JWare/AntX 0.5
     **/
    private void captureEnvironment(Project project, Requester task)
    {
        captureRefList(project);
        captureVarList(project);
        m_propertiesNet = new ProjectPropertiesNet(m_label,project);
    }



    /**
     * Restores the standard project fixture bits only. Iteration
     * fixture overlays are not touched.
     * @param task the controlling task (non-null)
     * @since JWare/AntX 0.5
     **/
    private void restoreEnvironment(Requester task)
    {
        m_propertiesNet.uninstall(new Responses.LogUsing(task));
        passthruPropList(task);
        restoreVarList(task);
        restoreRefList(task);
    }



    /**
     * Apply local (but isolated) fixture instructions.
     **/
    private void applyLocals()
    {
        if (m_controls!=null) {
            applyLocalVars();
            applyLocalProperties();
        }
    }




    /**
     * Applies local property (un)assignment instructions. These
     * changes are visible to all tasks within bubble from same
     * iteration/VM.
     * @since JWare/AntX 0.5
     **/
    private void applyLocalProperties()
    {
        List from = m_controls.getProperties();
        if (from!=null) {
            synchronized(from) {
                Hashtable ht = new Hashtable(from.size(),0.9f);
                for (int i=0,N=from.size();i<N;i++) {
                    ConditionalLocal local = (ConditionalLocal)from.get(i);
                    if (local.isEnabled()) {
                        //NB: Use the empty string as a proxy for unset!
                        String value = local.getValue();
                        if (value!=null) {
                            ht.put(local.getName(),value);
                        } 
                        else {
                            Boolean f = m_controls.getInheritFlag(local);
                            if (f==null || f==Boolean.FALSE) {
                                ht.put(local.getName(),"");
                            } else {//NB: Properties must be explicitly inherited!
                                value = getProject().getProperty(local.getName());
                                if (value==null) {
                                    value = "";
                                }
                                ht.put(local.getName(),value); 
                            }
                        }
                    }//enabled
                }//foreach
                m_propertiesNet.seedProperties(ht,false,false);
            }//lock
        }//!=null
    }



    /**
     * Apply any passed through property settings to our enclosing
     * context. Must be called <em>after</em> properties net removed.
     * Pre-existing script properties are not altered!
     * @param task controlling task (non-null)
     * @since JWare/AntX 0.5
     **/
    private void passthruPropList(Requester task)
    {
        if (m_controls!=null) {
            boolean allowThru = m_controls.isBlocking();
            Map edits = m_propertiesNet.getFinalProperties(true,allowThru);
            if (!edits.isEmpty()) {
                PropertyHelper ph = PropertyHelper.getPropertyHelper(getProject());
                Iterator itr= edits.entrySet().iterator();
                while (itr.hasNext()) {
                    Map.Entry e = (Map.Entry)itr.next();
                    ph.setNewProperty(null,(String)e.getKey(),
                        (String)e.getValue());//*QUIETLY*
                }
            }
        }
    }



    /**
     * Remembers the project's current set of references. If we're
     * being careful we remember some hash values for comparison
     * on 'leave'.
     **/
    private void captureRefList(Project P)
    {
        Iterator itr;
        Map current = P.getReferences();
        synchronized(current) {
            if (willBeCareful()) {
                Map mp= AntXFixture.newMap();
                itr= current.entrySet().iterator();
                while(itr.hasNext()) {
                    Map.Entry mE= (Map.Entry)itr.next();
                    mp.put(mE.getKey(), new RefInfo(mE));
                }
                if (m_defaultExcludes) {
                    List special = Iteration.defaultFixtureExcludes()
                        .copyOfReferencesIfAny();
                    if (special!=null) {
                       itr= special.iterator();
                       while (itr.hasNext()) {
                           String refid = itr.next().toString();
                           if (!mp.containsKey(refid)) {
                               mp.put(refid, new RefInfo(refid));
                           }
                       }
                    }
                }
                m_refInfo= mp;
            } else {
                List l= AntXFixture.newList();
                itr= current.keySet().iterator();
                while (itr.hasNext()) {
                    l.add(itr.next());
                }
                if (m_defaultExcludes) {
                    Iteration.defaultFixtureExcludes().addReferencesTo(l);
                }
                m_refInfo= l;
            }
        }
    }



    /**
     * Restores the project's set of references as best can.
     * Will <em>remove</em> references that weren't there but will
     * <em>not</em> put back references because we cannot guarantee
     * that the underlying objects are still valid. Also with Ant 1.6x
     * UnknownElement are dynamically flipped to their real objects
     * so we cannot determine if a change is part of Ant's function.
     * If we're being careful we can check for altered object references
     * (and contents if reference implementation supports this).
     * @param task controlling task (non-null)
     **/
    private void restoreRefList(Requester task)
    {
        Project P= task.getProject();
        Map current = P.getReferences();
        synchronized(current) {
            if (willBeCareful()) {
                Map mp = (Map)m_refInfo;
                current.keySet().retainAll(mp.keySet());
                RefInfo ri= new RefInfo();
                int whoopsies=0;
                for (Iterator itr=mp.values().iterator();itr.hasNext();) {
                    RefInfo ri0 = (RefInfo)itr.next();
                    if (!current.containsKey(ri0.key)) {
                        String warning = AntX.uistrs().get
                            ("fixture.buble.oldref.misin",ri0.key);
                        task.log(warning,Project.MSG_WARN);
                        whoopsies++;
                    } else {
                        ri.set(ri0.key,P);
                        String warning = ri0.compare(ri);
                        if (warning!=null) {
                            task.log(warning,Project.MSG_WARN);
                            whoopsies++;
                        }
                    }
                }
                mp.clear();//gc

                if (m_updateProperty!=null && whoopsies>0) {
                    String updateProperty = m_updateProperty+"-refs";
                    FixtureExaminer.checkIfProperty(P,task,updateProperty,true);
                    P.setNewProperty(updateProperty,String.valueOf(whoopsies));
                }
            }
            else {
                List l= (List)m_refInfo;
                current.keySet().retainAll(l);
                l.clear();//gc
            }
        }
        m_refInfo=null;
    }




    /**
     * Remembers the thread's current set of variables iff there
     * are local variable controls to be applied on exit.
     * @since JWare/AntX 0.5
     **/
    private void captureVarList(Project P)
    {
        if (m_controls!=null && m_controls.hasVariables()) {
            if (!m_controls.isBlocking()) {
                m_varInfo = Iteration.exportableProperties().copyOfPropertyNames();
            }
        }
    }
 
 
 
    /**
     * Restores the threads set of variables as defined by the
     * local variables controls. Note that unlike references,
     * AntX variable contents are never tested against any base
     * values.
     * @param task controlling task (non-null)
     * @since JWare/AntX 0.5
     **/
    private void restoreVarList(Requester task)
    {
        if (m_controls!=null && m_controls.hasVariables()) {
            List vars = m_controls.getVariableNames();
            if (m_controls.isBlocking()) {
                ExportedProperties.delete(vars);
            } else {
                List l = (List)m_varInfo;//originally-there
                l.addAll(vars);//+explicitly-saved
                ExportedProperties.retain(l);
                l.clear();//gc
            }
        }
    }



    /**
     * Applies local variable (un)assignment instructions. These
     * changes are visible to all tasks within bubble from same
     * iteration/VM.
     * @since JWare/AntX 0.5
     **/
    private void applyLocalVars()
    {
        List from = m_controls.getVariables();
        if (from!=null) {
            synchronized(from) {
                for (int i=0,N=from.size();i<N;i++) {
                    ConditionalLocal local = (ConditionalLocal)from.get(i);
                    if (local.isEnabled()) {
                        String value = local.getValue();
                        if (value!=null) {
                            ExportedProperties.set(local.getName(),value);
                        } 
                        else if (m_controls.getInheritFlag(local)==Boolean.FALSE) {
                            ExportedProperties.unset(local.getName());
                        }
                    }//enabled
                }//foreach
            }//lock
        }//!=null
    }



    /**
     * Applies this bubble's fixture kill methods.
     * @param task the controlling task (non-null)
     **/
    private void resetIterationVarStacks(Requester task)
    {
        if (m_killMethods!=null) {
            Project P= task.getProject();
            Iterator itr= m_killMethods.entrySet().iterator();
            Responses.LogAndRemember logr = new Responses.LogAndRemember(task);

            while (itr.hasNext()) {
                Map.Entry mE= (Map.Entry)itr.next();
                AntXFixture.reset((String)mE.getKey(),(String)mE.getValue(),logr);
                if (logr.hadProblem) {
                    if (m_updateProperty!=null) {
                        String updateProperty = m_updateProperty+"-"+mE.getKey();
                        FixtureExaminer.checkIfProperty(P,task,updateProperty,true);
                        P.setNewProperty(updateProperty,logr.what);
                    }
                    logr.reset();
                }
            }
        }
    }



    private Locals m_controls;
    private boolean m_isCareful;//NB:=> do not check hashvalues!
    private boolean m_defaultExcludes = Iteration.defaultdefaults().isFixtureExcludesEnabled();
    private ProjectPropertiesNet m_propertiesNet;
    private Object m_refInfo;
    private Map m_killMethods;
    private String m_updateProperty;
    private ExecutionBubble m_innerLayer;
    private Object m_varInfo;//NB: only touched if have local controls
    private String m_label = "isolate";



    /**
     * Keeps track of extra object reference information for closer
     * examination on restore.
     * @since     JWare/AntX 0.4
     * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version   0.5
     * @.safety   single
     * @.group    impl,helper
     **/
    private static class RefInfo
    {
        RefInfo() {
        }
        RefInfo(Map.Entry e) {
            set(e.getKey().toString(),e.getValue());
        }
        RefInfo(String refid) {
            set(refid,null);
        }
        void set(String key, Object value) {
            this.key = key;
            identityHash = System.identityHashCode(value);
            contentHash = identityHash;
            this.unknown = (value instanceof UnknownElement);
            if (value!=null && !this.unknown) {
                contentHash = value.hashCode();
            }
        }
        void set(String key, Project P) {
            set(key, FixtureExaminer.trueReference(P,key)/*!!!*NOT* P.getReference()*/);
        }
        String compare(RefInfo tmp) {
            if (!unknown) {
                if (tmp.identityHash!=identityHash) {
                    return Iteration.uistrs().get("fixture.buble.oldref.switched",key);
                }
                if (tmp.contentHash!=contentHash) {
                    return Iteration.uistrs().get("fixture.buble.oldref.chged",key);
                }
            }
            return null;
        }
        String key;
        int identityHash;
        int contentHash;
        boolean unknown;//true means Ant can resolve behind our backs
    }
}

/* end-of-LocalExecutionBubble.java */
