/**
 * $Id: MatchString.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.match;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.StringEquality;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;

/**
 * Superclass of the AntX equals and like choice tasks. By default works for <i>equals</i>
 * comparision with trim and case-sensitivity variants.
 *
 * @since    JWare/AntX 0.1 (Pushed down from ChoiceTask JWare/AntX 0.5)
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      SwitchTask
 **/

public abstract class MatchString extends ChoiceTask
    implements IgnoreCaseEnabled, TrimEnabled
{
    /**
     * Initializes new string choice task.
     **/
    protected MatchString(String iam)
    {
        super(iam);
    }



    /**
     * Initializes new string choice task with custom delay configuration.
     * @param iam CV-label
     * @param delayConfiguration <i>true</i> if delay nested task configuration
     **/
    protected MatchString(String iam, boolean delayConfiguration)
    {
        super(iam,delayConfiguration);
    }



    /**
     * Initializes the enclosing project of this component. Updates
     * any internal project-components too.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        getTest().setProject(P);
    }



    /**
     * Returns this choice task's underlying StringEquality test.
     * Never returns <i>null</i>.
     **/
    protected StringEquality getTest()
    {
        return m_eqTest;
    }



    /**
     * Returns this choice's underlying match value (as-is). The returned
     * value can represent a literal value, a property's name, a variable's
     * name, or a reference's identifier.
     * @since JWare/AntX 0.2
     **/
    public final String getFlexValue()
    {
        return getTest().getKnownValueGetter().get();
    }



    /**
     * Sets the match value for this choice task. How the incoming value
     * is actually used is up to the concrete choice subclass.
     * @param value the new choice (the known)
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setvalu- nonzro valu");
        getTest().setKnownArg(value,false);
    }



    /**
     * Returns this choice's match value (as set). Will return <i>null</i>
     * if never set or underlying value is not a literal string (used as-is).
     **/
    public final String getValue()
    {
        FlexString eqT= getTest().getKnownValueGetter();
        return eqT.isLiteral() ? eqT.get() : null;
    }



    /**
     * Sets the match property for this choice task. The property will be
     * read at the comparision time; how the returned value is used in the
     * comparision is up to the specific choice subclass.
     * @param property the property containing the choice (the known-later)
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setprop- nonzro name");
        getTest().setKnownArg(property,true);
    }



    /**
     * Returns this choice's property name (as set). Will return <i>null</i>
     * if never set or the underlying value is not a property name.
     **/
    public final String getProperty()
    {
        FlexString eqT= getTest().getKnownValueGetter();
        return eqT.isProperty() ? eqT.get() : null;
    }



    /**
     * Sets the match variable for this choice task. The variable will be
     * read at the comparision time; how the returned value is used in the
     * comparision is up to the specific choice subclass.
     * @param variable the variable containing the choice (the known-later)
     * @since JWare/AntX 0.2
     **/
    public void setVariable(String variable)
    {
        require_(variable!=null,"setVar- nonzro name");
        getTest().setKnownArg(variable,false);
        getTest().getKnownValueGetter().setIsExported(true);
    }



    /**
     * Returns this choice's variable name (as set). Will return <i>null</i>
     * if never set or the underlying value is not a variable's name.
     * @since JWare/AntX 0.2
     **/
    public final String getVariable()
    {
        FlexString eqT= getTest().getKnownValueGetter();
        return eqT.isExported() ? eqT.get() : null;
    }



    /**
     * Sets the match reference for this choice task. The reference will be
     * read at the comparision time; how the returned value is used in the
     * comparision is up to the specific choice subclass.
     * @param refid the reference containing the choice (the known-later)
     * @since JWare/AntX 0.2
     **/
    public void setReference(String refid)
    {
        require_(refid!=null,"setRef- nonzro refid");
        getTest().setKnownArg(refid,false);
        getTest().getKnownValueGetter().setIsReference(true);
    }



    /**
     * Returns this choice's reference identifier (as set). Will return <i>null</i>
     * if never set or the underlying value is not a reference name.
     * @since JWare/AntX 0.2
     **/
    public final String getReference()
    {
        FlexString eqT= getTest().getKnownValueGetter();
        return eqT.isReference() ? eqT.get() : null;
    }



    /**
     * Returns <i>true</i> if this choice has not been defined properly.
     **/
    public final boolean isUndefined()
    {
        return getTest().getKnownValueGetter().isUndefined();
    }



    /**
     * Returns the actual value used in this choice's comparision. Returns
     * <i>null</i> if value never defined or determined value doesn't
     * exist; for example, if the named reference doesn't exist in project.
     **/
    public final String getVUT()
    {
        return getTest().getKnownValue();
    }



    /**
     * Set the case-sensitivity of the comparision function. This setting
     * applies to both the known and unknown values.
     * @param ignore <i>true</i> if comparision should be case insensitive
     **/
    public void setIgnoreCase(boolean ignore)
    {
        getTest().setIgnoreCase(ignore);
    }



    /**
     * Returns <i>true</i> if a case insensitive comparision will be done.
     **/
    public final boolean isIgnoreCase()
    {
        return getTest().isIgnoreCase();
    }



    /**
     * Set the will-trim option of the comparision condition.
     * @param trim <i>true</i> if strings should be trimmed before
     *              comparision
     **/
    public void setTrim(boolean trim)
    {
        getTest().setTrim(trim);
    }



    /**
     * Returns <i>true</i> if both arguments will be trimmed before
     * comparision.
     **/
    public final boolean willTrim()
    {
        return getTest().willTrim();
    }



    /**
     * Evaluates this choice's condition for a hit. Returns
     * <i>true</i> if this choice's criteria are met by the incoming
     * value under consideration.
     * @param vut value under test (vut-vut)
     * @param context [unused] evaluation context
     * @throws BuildException if this task isn't nested properly
     **/
    public boolean eval(String vut, Reference context) throws BuildException
    {
        verifyCanExecute_("eval");

        if (vut==null) {
            return false;//never matches
        }

        StringEquality test= getTest();

        test.setUnknownArg(vut);

        return test.eval();
    }



    private StringEquality m_eqTest = new StringEquality();
}

/* end-of-MatchString.java */
