/**
 * $Id: OnceTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.ExecutionMode;
import  com.idaremedia.antx.parameters.FixturePassthru;


/**
 * A caller implementation that runs a set of named inlined steps or top-level targets
 * once. This task is the internal implementation; the script-facing interface must
 * be provided by a subclass like {@linkplain CallOnceTask}. For a looping construct
 * around the OnceTask, see the {@linkplain ForEachTask}.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 **/

public abstract class OnceTask extends CallerTask
{
    /**
     * Initializes a new OnceTask.
     **/
    protected OnceTask()
    {
        super(AntX.flow+"OnceTask:");
    }


    /**
     * Initializes a new CV-labeled OnceTask.
     * @param iam CV-label (non-null)
     **/
    protected OnceTask(String iam)
    {
        super(iam);
    }

// ---------------------------------------------------------------------------------------
// Optional Script-facing Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Returns the fixture passthru setting to be used when this
     * caller runs its targets. Never returns <i>null</i>;
     * will return <span class="src">PROPERTIES</span> if
     * not set explicitly.
     * @since JWare/AntX 0.4
     **/
    public FixturePassthru getPassthruOption()
    {
        return FixturePassthru.PROPERTIES;
    }


    /**
     * Returns this task's comma-delimited list of step names.
     * Will return <i>null</i> if never set. By default, all
     * steps must exist within the same target as this task.
     **/
    public String getStepNamesList()
    {
        return null;
    }


    /**
     * Returns this task's comma-delimited list of target names.
     * Will return <i>null</i> if never set.
     **/
    public String getTargetNamesList()
    {
        return null;
    }


    /**
     * Returns this task's comma-delimited list of macro names.
     * Will return <i>null</i> if never set. By default, all
     * macro must exist within the same project as this task.
     * @since JWare/AntX 0.5
     **/
    public String getMacroNamesList()
    {
        return null;
    }



    /**
     * Returns this task's execution mode. Never returns <i>null</i>;
     * will return <span class="src">ISOLATED</span> if never set.
     * @since JWare/AntX 0.4
     **/
    public ExecutionMode getMode()
    {
        return ExecutionMode.ISOLATED;
    }


// ---------------------------------------------------------------------------------------
// Execution:
// ---------------------------------------------------------------------------------------

    /**
     * Returns the number of kinds-of runnables specified (basically
     * whether one or more of steps, targets, or macros have been requested).
     **/
    protected int getKindOfRunnablesSpecified()
    {
        int N=0;
        if (getStepNamesList()!=null)   { N++; }
        if (getTargetNamesList()!=null) { N++; }
        if (getMacroNamesList()!=null)  { N++; }
        return N;
    }


    /**
     * Ensures that either a named set of steps, targets, or macros
     * has been specified (not both and not none).
     * @param calr calling method's identifier
     * @throws BuildException if verification fails
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        int Nr= getKindOfRunnablesSpecified();

        if ((Nr==0) || (Nr>1)) { //only one steplist -or- targetlist -or- macrolist
            String ermsg = uistrs().get("flow.targets.or.steps");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        
        if (getMacroNamesList()!=null && getMode()!=ExecutionMode.LOCAL) {
            String ermsg = getAntXMsg("flow.macro.islocal");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Returns list of configured {@linkplain TargetCaller
     * target callers} for either the requested steps or targets.
     * @throws BuildException if unable to create target callers
     **/
    protected List copyOfOrderedTargetCallers()
        throws BuildException
    {
        String candidateNames;
        
        candidateNames = getStepNamesList();
        if (candidateNames!=null) {
            if (getMode()==ExecutionMode.LOCAL) {
                return createLocalTargetCallers(candidateNames,false);
            }
            return createStepLauncherCallers(candidateNames);
        }

        candidateNames = getTargetNamesList();
        if (candidateNames!=null) {
            if (getMode()==ExecutionMode.LOCAL) {
                return createLocalTargetCallers(candidateNames,true);
            }
            return createGenericTargetCallers(candidateNames);
        }

        candidateNames = getMacroNamesList();
        if (candidateNames!=null) {
            return createMacroInstanceCallers(candidateNames);
        }

        return AntXFixture.newList();
    }



    /**
     * Prepares a list of pre-configured StepLauncherCallers to call
     * steps defined within this task's target. Only
     * {@linkplain InlineStep InlineStep} allowed.
     **/
    private List createStepLauncherCallers(String stepNames)
        throws BuildException
    {
        String targetName = StepLauncherCaller.findSpecialTarget(this);

        List wanted = Tk.splitList(stepNames);//names
        List actual = getFilteredStepsLike(InlineStep.class,wanted);//steps

        if (wanted.size()!=actual.size()) {
            String ermsg = uistrs().get("flow.steplaunch.missing.Nsteps",
                                        getOwningTarget().getName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }

        StepLauncherCaller caller;
        List callers = AntXFixture.newList(wanted.size());
        String kindOfStep = InlineStep.class.getName();

        for (int i=0,N=wanted.size();i<N;i++) {
            caller = StepLauncherCaller.create(this,true,targetName,
                                               (String)wanted.get(i),kindOfStep,
                                               2/*allows 1-level nest in 'step'*/);
            transferOverlayParameters(caller);
            callers.add(caller);
        }

        wanted.clear();
        actual.clear();

        return callers;
    }



    /**
     * Prepares a list of pre-configured generic TargetCallers to call
     * existing real Ant targets.
     * @throws BuildException if a requested target does not exist
     **/
    private List createGenericTargetCallers(String targetNames)
    {
        List wanted = Tk.splitList(targetNames);
        List callers= AntXFixture.newList(wanted.size());

        AnyTargetCaller caller;
        String targetName;

        for (int i=0,N=wanted.size();i<N;i++) {
            targetName = (String)wanted.get(i);
            if (!targetExists(targetName)) {
                String ermsg = uistrs().get("flow.steplaunch.missing.target",
                                            targetName, getProject().getName());
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
            caller = new AnyTargetCaller(this, targetName);
            caller.setStepName(targetName);
            transferOverlayParameters(caller);
            callers.add(caller);
        }

        wanted.clear();

        return callers;
    }



    /**
     * Returns a filled in list of target callers for this
     * task's list.
     * @throws BuildException if name list invalid or a called
     *         target fails.
     * @since JWare/AntX 0.4
     */
    private List createLocalTargetCallers(String nameList, boolean topLevel)
    {
        List l= Tk.splitList(nameList);
        List callers= AntXFixture.newList(l.size());
        String tn;
        LocalTargetCaller caller;

        for (int i=0,N=l.size();i<N;i++) {
            tn = l.get(i).toString();
            caller = new LocalTargetCaller(this);
            if (topLevel) {
                caller.setTarget(tn);
            } else {
                caller.setStepName(tn);
            }
            transferOverlayParameters(caller);
            callers.add(caller);
        }

        return callers;
    }




    /**
     * Returns a filled in list of local macro instance callers 
     * for this task's list.
     * @throws BuildException if name list invalid or a called
     *         target fails.
     * @since JWare/AntX 0.5
     */
    private List createMacroInstanceCallers(String nameList)
    {
        List l= Tk.splitList(nameList);
        List callers= AntXFixture.newList(l.size());
        String tn;
        MacroInstanceCaller caller;

        for (int i=0,N=l.size();i<N;i++) {
            tn = l.get(i).toString();
            caller = new MacroInstanceCaller(this);
            caller.setTarget(tn);
            transferOverlayParameters(caller);
            callers.add(caller);
        }

        return callers;
    }
}
/* end-of-OnceTask.java */
