/**
 * $Id: AnyTargetCaller.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.taskdefs.Ant;
import  org.apache.tools.ant.taskdefs.CallTarget;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.parameters.FixturePassthru;

/**
 * TargetCaller implementation that calls any existing top-level project target.
 * Based on the standard ant <span class="src">antcall</span> task so target is
 * executed in a partitioned child project. Modifications to the child project
 * are not transferred back to the (caller) parent.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      OnceTask
 * @.impl    Ant 1.6.3 CallTarget has some new initialization dependencies
 *           with setters and its underlying "callee". Calling setTarget twice
 *           before execution ignores the second target-set if an init has 
 *           been done between.
 **/

public class AnyTargetCaller extends CallTarget implements TargetCaller
{
    /**
     * Initializes a new target caller from a call task.
     * @param from the new target caller's controlling task
     * @param targetName the target to be called
     **/
    public AnyTargetCaller(OnceTask from, String targetName)
    {
        setTaskName("call");
        setProject(from.getProject());
        setOwningTarget(from.getOwningTarget());
        setLocation(from.getLocation());
        init(from,from.getPassthruOption());
        setTarget(targetName);
    }



    /**
     * Initializes a new AntX-based target caller from any
     * source. Caller is responsible for setting this task's
     * project, owning target, inherit flags, etc.
     * @param targetName the target to be called
     * @since JWare/AntX 0.5
     **/
    public AnyTargetCaller(Project project, String targetName)
    {
        setProject(project);
        setTaskName("call");
        setTarget(targetName);
    }



    /**
     * Ensures any presetting of target name is redone with new
     * "ant" callee (inherited).
     * @since JWare/AntX 0.5
     * @.impl Since Ant 1.6.3, force any new callee (inherited) to
     *        be resynced with any presetting of the target's name.
     **/
    public void init()
    {
        super.init();
        if (m_targetName!=null) {
            super.setTarget(m_targetName);
        }
    }



    /**
     * Initializes this caller from the controlling task.
     * @param from controlling task whose context should be
     *        used (non-null)
     * @param fixturecontrol bits of controller's fixture
     *        to pass along (non-null)
     * @since JWare/AntX 0.4
     **/
    public void init(Task from, FixturePassthru fixturecontrol)
    {
        setProject(from.getProject());
        setOwningTarget(from.getOwningTarget());
        setLocation(from.getLocation());

        switch (fixturecontrol.getIndex()) {
            case FixturePassthru.ALL_INDEX: {
                setInheritRefs(true);
                setInheritAll(true);
                break;
            }
            case FixturePassthru.PROPERTIES_INDEX: {
                setInheritAll(true);
                break;
            }
            case FixturePassthru.REFERENCES_INDEX: {
                setInheritRefs(true);
                break;
            }
            case FixturePassthru.NONE_INDEX: {
                setInheritRefs(false);
                setInheritAll(false);
                break;
            }
        }
    }


    /**
     * (Re)Sets the target name of this caller.
     **/
    public void setTarget(String targetName)
    {
        super.setTarget(targetName);
        m_targetName = targetName;//no-getter on inherited
    }


    /**
     * Returns this caller's target's name.
     **/
    public String getTargetName()
    {
        return m_targetName;
    }


    /**
     * Sets the step on whose behave this caller exists.
     **/
    public final void setStepName(String stepName)
    {
        m_stepName = stepName;
    }


    /**
     * Returns the step on whose behave this caller exists.
     **/
    public final String getStepName()
    {
        return m_stepName;
    }


    /**
     * Create a new property to passthru to called target.
     **/
    public Property createProperty()
    {
        return createParam();
    }


    /**
     * Create a new property set to passthru to called
     * target.
     * @since JWare/AntX 0.4
     **/
    public PropertySet createPropertySet()
    {
        PropertySet properties = new PropertySet();
        properties.setProject(getProject());
        addPropertyset(properties);
        return properties;
    }


    /**
     * Create a new reference to passthru to called target.
     * @since JWare/AntX 0.3
     **/
    public Ant.Reference createReference()
    {
        Ant.Reference r = new Ant.Reference();
        addReference(r);
        return r;
    }


    /**
     * Executes the target this caller references.
     **/
    public void run()
        throws BuildException
    {
        maybeConfigure();
        execute();
    }


    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * a required property into the to-be-called target's runtime
     * environmentbefore it's executed.
     * @param property property name (non-null)
     * @param value property value-- used as-is no additional
     *          replacements are done (non-null)
     **/
    public final void run(String property, String value)
        throws BuildException
    {
        Property p= this.createProperty();
        p.setName(property);
        p.setValue(value);
        run();
    }


    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * a set of properties into the to-be-called target's runtime
     * environment before it's executed.
     * @param properties properties file (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void run(Map properties)
        throws BuildException
    {
        if (!properties.isEmpty()) {
            Iterator itr = properties.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry item = (Map.Entry)itr.next();
                Property p= this.createProperty();
                p.setName(item.getKey().toString());
                p.setValue(item.getValue().toString());
            }
        }
        run();
    }


    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * arbitrary kinds of fixture data into the to-be-called target's
     * runtime environment before it's executed.
     * @param prep client preparation snippet (non-null)
     **/
    public final void run(TargetCaller.Prep prep)
        throws BuildException
    {
        prep.prepare(this);
        run();
    }


    private String m_targetName, m_stepName="";
}

/* end-of-AnyTargetCaller.java */
