/**
 * $Id: DataDefTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.fixture.helpers;

import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ComponentHelper;
import  org.apache.tools.ant.DynamicElementNS;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectHelper;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.FeedbackLevel;


/**
 * Fixture configuration instruction that creates a new data item and installs it as
 * a top-level project reference. This task ensures that a data item is not installed
 * into the project's reference table before it should be. Unless told otherwise, a
 * &lt;datadef&gt; will overwrite and existing reference with its nested data object's
 * definition.
 * <p/>
 * <b>Example Usage:</b><pre>
 *      &lt;target name="Might-Never-Be-Called"&gt;
 *          &lt;datadef name="antlibs.in" tellproperty="antlibs.merged"&gt;
 *              &lt;strings suffix="/antlib.xml"&gt;
 *                  &lt;string value="${antxtras.src}"/&gt;
 *                  &lt;string value="${sams.src}"/&gt;
 *                  &lt;string value="${antunit.src}"/&gt;
 *              &lt;/strings&gt;
 *          &lt;/datadef&gt;
 *          &#8230;
 *      &lt;/target&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class DataDefTask extends AssertableTask implements DynamicElementNS
{
    /**
     * Initializes a new DataDefTask instance.
     **/
    public DataDefTask()
    {
        super(AntX.nopackage);
    }


//  ---------------------------------------------------------------------------------------
//  Script-facing Parameters:
//  ---------------------------------------------------------------------------------------

    /**
     * Set the target reference's identifier (non-null)
     * @param dstRefId reference id to install data under (non-null)
     * @see #setHaltIfExists setHaltIfExists(&#8230;)
     **/
    public void setName(String dstRefId)
    {
        require_(dstRefId!=null, "setName- nonzro refid");
        m_dstRefId = dstRefId;
    }



    /**
     * Synonym for {@linkplain #setName setName(&#8230;)}.
     * @param dstRefId reference id to install data under (non-null)
     **/
    public final void setToId(String dstRefId)
    {
        setName(dstRefId);
    }



    /**
     * Returns the (new) target reference's identifer.
     * Returns <i>null</i> if never set and source never
     * set.
     **/
    public String getToRefId()
    {
        return m_dstRefId;
    }



    /**
     * Tells this datadef whether to try to set the new object's
     * 'id' attribute explicitly.
     * @param setIt <i>true</i> to try.
     **/
    public void setSetId(boolean setIt)
    {
        m_setImplId = setIt;
    }



    /**
     * Sets this task's announcement property's name. This project
     * property will be set to "<span class="src">true</span>" before
     * this tasks' execute method returns.
     * @param property name of property (non-null)
     **/
    public void setTellProperty(String property)
    {
        require_(property!=null,"setProperty- nonzro property");
        m_announceProperty = property;
    }



    /**
     * Returns this task's announcement property. Will return
     * <i>null</i> if never set explicitly.
     **/
    public final String getTellProperty()
    {
        return m_announceProperty;
    }



    /**
     * Sets if this task will generate a build error if the
     * destination reference id already exists within its project.
     * Trumps the 'overwrite' option.
     * @param halt <i>true</i> to throw build exception
     * @see #setToId setToId
     * @see #setOverwrite setOverwrite
     **/
    public void setHaltIfExists(boolean halt)
    {
        m_haltIfExists = halt;
    }



    /**
     * Returns <i>true</i> if this task will generate a build error
     * if the destination reference id already exists within its
     * project. Defaults <i>false</i> (supports overwrites).
     **/
    public boolean isHaltIfExists()
    {
        return m_haltIfExists;
    }



    /**
     * Sets if this task will overwrite an existing reference.
     * This option is ignored if the 'haltiferror' option
     * is turned on.
     * @see #setHaltIfExists
     * @param allowOverwrite <i>true</i> if can overwrite old reference
     **/
    public void setOverwrite(boolean allowOverwrite)
    {
        m_allowOverwrite = allowOverwrite;
    }



    /**
     * Returns <i>true</i> if this task will overwrite an existing
     * reference. Defaults <i>true</i>. Ignored if 'haltiferror'
     * option is turned on.
     **/
    public boolean willAllowOverwrite()
    {
        return m_allowOverwrite;
    }



    /**
     * Tells this task how much non-diagnostic feedback to generate.
     * Really only has "loud" vs. "quiet-ish" interpretation. If
     * set quiet, this task will not issue a warning if it overwrites
     * an existing reference.
     * @param level feedback level (non-null)
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fbl = FeedbackLevel.from(level);
        if (fbl==null) {
            String e = getAntXMsg("task.illegal.param.value",
                           getTaskName(), level, "feedback");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fbl;
    }



    /**
     * Returns this task's assigned feedback level. Will return
     * <i>null</i> by default.
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }


//  ---------------------------------------------------------------------------------------
//  Script-facing Nested Elements:
//  ---------------------------------------------------------------------------------------


    /**
     * Returns a new instance of the named object type.
     * @throws BuildException if already have a data item defined.
     **/
    public final Object createDynamicElement(String uri, String localName,
        String qName)
    {
        if (m_dataImpl!=null) {
            String ermsg = uistrs().get("taskset.only.one.specialtask",
                                qName, m_dataQName);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }

        Project P= getProject();
        ComponentHelper ch = ComponentHelper.getComponentHelper(P);
        String fullname = localName;
        if (!localName.equals(qName)) {
            fullname = ProjectHelper.genComponentName(uri,localName);
        }

        Object o= ch.createComponent(fullname);
        if (o!=null) {
            P.setProjectReference(o);
            if (o instanceof Task) {
                Task t = (Task)o;
                t.setOwningTarget(getOwningTarget());
                t.setTaskName(qName);
                t.setTaskType(fullname);
                t.init();
            }
            m_dataQName = qName;
            m_dataImpl = o;
        }

        return m_dataImpl;
    }


//  ---------------------------------------------------------------------------------------
//  Execution:
//  ---------------------------------------------------------------------------------------


    /**
     * Installs the nested data item declaration as a top-level reference
     * in this task's parent project.
     */
    public void execute()
    {
        verifyCanExecute_("exec");
        
        final Project P = getProject();
        boolean quiet = FeedbackLevel.isQuietish(m_fbLevel,true,true);

        Object there = FixtureExaminer.trueReference(P,m_dstRefId);
        if (there==FixtureExaminer.PENDING_REFERENCE) {
            quiet = true;
            there = null;
        }
        if (there!=null) {
            String msg = uistrs().get("task.warn.refid.exists",m_dstRefId);
            if (isHaltIfExists()) {
                log(msg,Project.MSG_ERR);
                throw new BuildException(msg,getLocation());
            }
            if (!willAllowOverwrite()) {
                if (!quiet) {
                    log(msg,Project.MSG_VERBOSE);
                }
                return;
            }
            if (!quiet) {
                log(msg,Project.MSG_WARN);
            }
        }

        if (m_setImplId) {
            tryToSetId(m_dataImpl);
        }

        if (m_dataImpl instanceof Task) {
            ((Task)m_dataImpl).perform();
        }

        Map refs = P.getReferences();
        if (quiet) {
            refs.put(m_dstRefId,m_dataImpl);
        } else {
            P.addReference(m_dstRefId,m_dataImpl);
        }

        if (m_announceProperty!=null) {
            checkIfProperty_(m_announceProperty,true);
            P.setNewProperty(m_announceProperty,"true");
        }
    }



    /**
     * Verifies that this task has at least its source reference
     * identifier set.
     * @throws BuildException if missing any required bits.
     **/
    public void verifyIsDefined()
    {
        if (getToRefId()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",
                getTaskName(),"name");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        if (m_dataImpl==null) {
            String ermsg = uistrs().get("task.needs.oneof.these.nested",
                getTaskName(),"<any datatype>");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Verifies this task is completely defined with a target refid
     * and data definition sub-element.
     * @throws BuildException if not in project or not defined properly.
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
        verifyIsDefined();
    }



    /**
     * Tries to call a 'setId' method on the new instance. If no
     * such method exists or is inaccessible we silently ignore request.
     * @param instance object being constructed.
     **/
    private void tryToSetId(Object instance)
    {
        try {
            Tk.invokeit(instance, "setId", new Class[]{String.class},
                new String[]{getToRefId()});
        } catch(Exception reflectX) {
            /*burp*/
        }
    }


    private String m_dstRefId;
    private String m_announceProperty;
    private boolean m_haltIfExists;//NB:false allow overwrites!
    private boolean m_allowOverwrite=true;//NB:true allow overwrites!
    private boolean m_setImplId;//NB:no unless asked-to explicitly
    private FeedbackLevel m_fbLevel=FeedbackLevel.NORMAL;//NB:=>overwrite noise
    private Object m_dataImpl;
    private String m_dataQName;

}

/* end-of-InstallRefTask.java */