/**
 * $Id: EmitTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  com.idaremedia.apis.DiagnosticsEmitter;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ErrorSnapshot;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.helpers.Setting;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.MsgTask;

/**
 * Ant-isque link to a standard log4j Logger/Category's api. Some examples:<pre>
 *   &lt;emit msgid="precompile-checks-done" echo="yes"/&gt;
 *   &lt;emit level="debug" properties="all" message="Hi" echo="no"/&gt;
 *   &lt;emit from="${feedback.compile}" message="Past precompile checks" echo="yes"/&gt;
 *   &lt;emit level="fatal" thrown="${err.last}" msgid="build.failed" properties="all"/&gt;
 *   &lt;emit message="javac.options" properties="javac.debug,javac.depend" level="verbose"/&gt;
 *   &lt;emit timestamp="on" echo="off" properties="none"/&gt;
 *   &lt;emit from="${feedback.distrib}.web" msgid="checkpoint.upload.finished"/&gt;
 *   &lt;emit from="${grp.compiling}.apis" task="javac[templates]" msgid="banner.compiling"/&gt;
 *   &lt;emit thrown="build.error" level="fatal"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @see      EmitConfiguration
 * @see      EmitConfigureTask
 * @see      EmitContext
 **/

public class EmitTask extends MsgTask implements EmitConfigurable
{
    /**
     * Initializes a new EmitTask instance.
     **/
    public EmitTask()
    {
        super(AntX.feedback);
    }


    /**
     * Initializes a new CV-labeled EmitTask instance.
     **/
    public EmitTask(String iam)
    {
        super(iam);
    }


    /**
     * Adds an inclusion declaration to this task. This inclusion's
     * value (property, reference, variable, etc.) will be added to
     * any environment or error snapshot broadcast by this task.
     * @param include the inclusion (non-null)
     * @see #setProperties
     **/
    public void addConfiguredInclude(EmitInclude include)
    {
        require_(include!=null,"addIncl- nonzro incl");
        getNestedInclusions().add(include);
    }


    /**
     * Returns this task's list of variable inclusions. Never
     * returns <i>null</i> but can be empty.
     **/
    protected final List getNestedInclusions()
    {
        return m_nestedIncludes;
    }


    /**
     * Sets whether this task should echo its message to the standard
     * Ant logging system <em>in addition</em> to the direct-to-log4j
     * broadcast.
     * @param wantIt preference setting (yes,no,inherit) (non-null)
     **/
    public void setEcho(String wantIt)
    {
        require_(wantIt!=null,"setEcho- nonzro arg");
        m_echoPref= Setting.from(wantIt, Setting.INHERITED);//NB:allow aliases
    }


    /**
     * Returns the echo-to-Ant-log preference for this task. Will
     * return <i>Setting.INHERITED</i> if never set.
     **/
    public final Setting getPreferEcho()
    {
        return m_echoPref;
    }


    /**
     * Returns <i>true</i> if this task should echo its emitted message
     * to the standard Ant logging system (via Project).
     **/
    protected final boolean shouldEcho(EmitConfiguration defaults)
    {
        switch (getPreferEcho().getIndex()) {
            case Setting.ON_INDEX: {
                return true;
            }
            case Setting.OFF_INDEX: {
                return false;
            }
            default: {
                return defaults.shouldEcho();
            }
        }
    }


    /**
     * Returns the AntX noiselevel this task will associate with
     * its broadcast message.
     **/
    protected final NoiseLevel getNoiseLevel(EmitConfiguration defaults)
    {
        NoiseLevel nl = getPreferredMsgLevel();
        if (nl==null) {
            nl= defaults.getNoiseLevel();
        }
        if (nl==null) {
            nl= getDefaultMsgLevel();
            ensure_(nl!=null,"getEffPri- nonzro lvl");
        }
        return nl;
    }


    /**
     * Sets the list of property names automatically attached
     * to any emitted message's snapshot. Really just an inlined
     * 'include' set shortcut.
     * @param nameList comma-delimited list of property names
     * @see #addConfiguredInclude
     **/
    public void setProperties(String nameList)
    {
        require_(nameList!=null, "setProps- nonzro lst");
        m_propertyNames= nameList;
    }


    /**
     * Returns the list of properties names automatically included
     * in any messages emitted from this task. Returns <i>null</i> if
     * never set.
     **/
    public final String getPropertyNamesList()
    {
        return m_propertyNames;
    }


    /**
     * Captures a list of named properties to the given error snapshot.
     * Returns <i>true</i> if all properties captured as result of
     * this call.
     * @param es the environment snapshot to update (non-null)
     * @param nameList the comma-delimited list of property names
     **/
    private boolean captureNamedProperties(ErrorSnapshot es, String nameList)
    {
        boolean all=false;

        if (nameList!=null) {
            String normalized = Tk.lowercaseFrom(nameList.trim());
            if (Strings.USER.equals(normalized)) {
                es.captureUserProperties();
            } else if (Strings.ALL.equals(normalized)) {
                es.captureAllProperties();
                all=true;
            } else {
                es.captureProperties(nameList);
            }
        }
        return all;
    }


    /**
     * Captures the appropriate properties to the given snapshot.
     * Captured properties include properties inherited from this task's
     * enclosing configuration, inlined property names, and nested
     * include declarations.
     * @param es the environment snapshot to update (non-null)
     **/
    private void captureProperties(EmitConfiguration defaults,
                                   StringBuffer defaultProperties,
                                   ErrorSnapshot es)
    {
        boolean ignore = false;

        if (defaultProperties.length()>0) {
            ignore = captureNamedProperties(es,defaultProperties.substring(0));
        }

        if (!ignore) {
            String nameList = getPropertyNamesList();
            if (nameList!=null) {
                ignore= captureNamedProperties(es,nameList);
            }
        }

        if (!getNestedInclusions().isEmpty()) {
            Iterator itr= getNestedInclusions().iterator();
            while (itr.hasNext()) {
                ((EmitInclude)itr.next()).apply(es);
            }
        }
    }


    /**
     * Sets this task's thrown-build-exception's refid. The refid
     * should point to a BuildException that should be included in
     * the snapshot passed to the log4j object render. If the reference
     * points to an ErrorSnapshot, its thrown BuildException is used.
     * The reference isn't extracted until this task is executed.
     **/
    public void setThrownRefId(String refId)
    {
        require_(refId!=null,"setThrownRefId- nonzro refid");
        m_thrownRefId= refId;
    }


    /**
     * The script-facing synonymn for {@linkplain #setThrownRefId
     * setThrownRefId}.
     **/
    public final void setThrown(String refId)
    {
        setThrownRefId(refId);
    }


    /**
     * Returns this task's thrown-build-exception's refid. Returns
     * <i>null</i> if never set.
     **/
    public final String getThrownRefId()
    {
        return m_thrownRefId;
    }


    /**
     * Returns the thrown build exception this task should attach to
     * its error snapshot. Returns <i>null</i> if no exception defined,
     * or this method is unable to retrieve a build exception from the
     * {@linkplain #setThrownRefId reference}.
     **/
    private Exception getThrown()
    {
        String refid = getThrownRefId();
        if (refid!=null) {
            Object ox= getProject().getReference(refid);
            if (ox instanceof BuildException) {
                return (BuildException)ox;
            } else if (ox instanceof ErrorSnapshot) {
                return ((ErrorSnapshot)ox).getThrown();
            }
            String ermsg = uistrs().get("task.echo.bad.refid", refid);
            log(ermsg, Project.MSG_WARN);
        }
        return null;
    }


    /**
     * Sets whether this emit task should include a local timestamp
     * with emitted message.
     * @param setin setting option (on,off,inherited), non-null
     **/
    public void setTimestamp(String setin)
    {
        require_(setin!=null,"setTm- nonzro setin");
        m_timestampPref = Setting.from(setin, Setting.INHERITED);//NB: allow aliases
    }


    /**
     * Returns <i>Setting.ON</i> if this task will include a local
     * timestamp with emitted messages. Defaults to <i>Setting.INHERITED</i>.
     **/
    public final Setting getPreferTimestamp()
    {
        return m_timestampPref;
    }


    /**
     * Returns the stringized timestamp this emission's time-of-execution.
     * Returns the empty string if a timestamp is not specified in this
     * task's (or its configuration's) definition.
     **/
    protected final String getTimestampLiteral(EmitConfiguration defaults, final long NOW)
    {
        String tsl = "";
        switch (getPreferTimestamp().getIndex()) {
            case Setting.OFF_INDEX: {
                break;
            }
            case Setting.ON_INDEX: {
                tsl = defaults.stampify(NOW);
                break;
            }
            default: {
                if (defaults.wantTimestamp()) {
                    tsl = defaults.stampify(NOW);
                }
            }
        }
        return tsl;
    }


    /**
     * Determine what if any local message should be broadcast or
     * attached to the emitted snapshot. Current EmitTasks support two forms
     * of local message: msgid (inherited) and a literal 'message'
     * attribute. Can have one or the other; not both.
     **/
    private String getLocalComment(EmitConfiguration defaults, final long NOW)
    {
        String tsl= getTimestampLiteral(defaults,NOW);
        String msg= null;

        if (getMsgId()!=null) {

            /**
             * Handle the optional msg arguments feature we inherited from
             * our superclass 'MsgTask'. The timestamp is *always* argument
             * number 3 after the fixed taskname(0) and location(1) args.
             **/
            String arg3 = getMsgArg1();
            String arg4 = getMsgArg2();

            if (arg3!=null || arg4!=null) {
                if (arg3!=null) {
                    if (arg4!=null) {
                        msg = getMsg(newMsgGetter(tsl,arg3,arg4));
                    } else {
                        msg = getMsg(newMsgGetter(tsl,arg3));
                    }
                } else {
                    msg = getMsg(newMsgGetter(tsl,"",arg4));
                }
            }
            else {
                msg = getMsg(newMsgGetter(tsl));
            }
        }

        /**
         * Fallback to a default message (if defined) from MsgTask. This
         * might also have been an inlined 'message' attribute.
         **/
        if (Tk.isWhitespace(msg) && getDefaultMsg()!=null) {
            msg = getDefaultMsg();
        }

        /**
         * Sheesh...ok just use the timestamp string as the message.
         **/
        if (Tk.isWhitespace(msg)) {
            msg = tsl;
        }

        return msg;
    }


    /**
     * Customize the broadcasting log4j logger to something other
     * than the one defined by this task's enclosing context.
     * @param categoryId category of logger to use (non-null)
     **/
    public void setFrom(String categoryId)
    {
        require_(categoryId!=null,"setFrom- nonzro category");
        m_categoryId= categoryId;
    }


    /**
     * Returns the custom log4j grouping with which all messages sent
     * from this  task are associated. Returns <i>null</i> if never
     * set.
     **/
    public final String getFrom()
    {
        return m_categoryId;
    }


    /**
     * Associates a custom task name with any emitted snapshots.
     **/
    public void setTask(String pseudoTaskName)
    {
        require_(pseudoTaskName!=null,"setTsk- nonzro pseudoNam");
        m_customTaskName= pseudoTaskName;
    }


    /**
     * Returns the task name associated with emitted snapshot. Will
     * return this task's names if a custom task name never set.
     **/
    public final String getEmittedTaskName()
    {
        String name = getTaskName();
        if (m_customTaskName!=null) {
            name = m_customTaskName;
        }
        return name;
    }


    /**
     * Returns this task's diagnostics emitter. Never returns <i>null</i>.
     **/
    protected final synchronized DiagnosticsEmitter getEmitter(EmitConfiguration defaults)
    {
        if (m_emitr!=null) {
            return m_emitr;
        }
        String grpId = getFrom();
        if (grpId!=null) {
            m_emitr = defaults.getCustomEmitter(grpId);
        } else {
            m_emitr = defaults.getEmitter();
        }
        return m_emitr;
    }


    /**
     * Factory method for this task's environment snapshot. This snapshot
     * is passed to the emitter service provider. For log4j-based emitters,
     * a custom object renderer is required to make sense of the snapshot.
     **/
    private ErrorSnapshot getSnapshot(EmitConfiguration defaults,
                                      StringBuffer properties,
                                      final long NOW)
    {
        ErrorSnapshot es =null;

        //capture thrown exception or not
        Exception thrown= getThrown();
        if (thrown!=null) {
            es = new ErrorSnapshot(this, thrown);
        } else {
            es = new ErrorSnapshot(this);
        }
        if (getThrownRefId()!=null) {
            es.setName(getThrownRefId());
        }

        //make it look like it's coming from caller
        es.setEffectiveTaskName(getEmittedTaskName());

        //capture properties, etc.
        captureProperties(defaults,properties,es);

        //attach a special commentary message if defined
        es.setComment(getLocalComment(defaults,NOW));

        return es;
    }


    /**
     * Returns <i>true</i> if enuf kruft has been attached to this
     * emit task that an environment snapshot is the best object to broadcast
     * to the emitter service (instead of a simple String). Basically if
     * any properties must be captured, it needs a snapshot.
     **/
    private boolean needsSnapshot(EmitConfiguration defaults,
                                  StringBuffer defaultProperties)
    {
        return (defaults.getPropertiesNameList(defaultProperties) ||
                defaultProperties.length()>0 ||
                getPropertyNamesList()!=null ||
                !getNestedInclusions().isEmpty()
                );
    }


    /**
     * Returns this tasks closest configuration controller. Usually
     * an enclosing 'emitconfigure' outer task.
     **/
    public EmitConfiguration getDefaults()
    {
        EmitConfiguration ec = EmitContext.getConfiguration();
        if (ec==null) {
            ec= DefaultEmitConfiguration.INSTANCE;
        }
        return ec;
    }


    /**
     * Sends message to the appropriate DiagnosticsEmitter. Optionally
     * echoes message to standard Ant logging infrastructure.
     * @throws BuildException if unable to execute
     **/
    public void execute() throws BuildException
    {
        final long NOW = System.currentTimeMillis();

        verifyCanExecute_("execute");

        EmitConfiguration defaults = getDefaults();//NB:cache on my stack

        DiagnosticsEmitter emitter = getEmitter(defaults);
        NoiseLevel nl = getNoiseLevel(defaults);

        boolean isEchoed  = shouldEcho(defaults);
        boolean isEmitted = emitter.emits(nl.getIndex());

        if (isEmitted || isEchoed) {
            StringBuffer defaultProperties = new StringBuffer(71);

            if (needsSnapshot(defaults, defaultProperties)) {
                ErrorSnapshot es = getSnapshot(defaults,defaultProperties,NOW);

                if (isEmitted) {
                    Emit.broadcast(emitter, es, es.getThrown(), nl);
                }
                if (isEchoed) {
                    log(es.toString(), nl.getNativeIndex());
                }
            }
            else {
                String msg = getLocalComment(defaults,NOW);
                Exception thrown = getThrown();
                if (thrown!=null) {
                    msg += uistrs().get("emit.cause.msg.addon",thrown.getMessage());
                }

                if (isEmitted) {
                    Emit.broadcast(emitter, msg, thrown, nl);
                }
                if (isEchoed) {
                    log(msg, nl.getNativeIndex());
                }
            }
        }
    }

    private List m_nestedIncludes = AntXFixture.newList(4);//NB:embeded <include>
    private String m_customTaskName;//NB:=> this.getTaskName();
    private String m_propertyNames;//NB:use inherited
    private String m_thrownRefId;//NB:nothing
    private Setting m_timestampPref=Setting.INHERITED;//NB:does what enclosing configure wants
    private Setting m_echoPref= Setting.INHERITED;//NB:ditto
    private String m_categoryId=null;//NB:use inherited
    private DiagnosticsEmitter m_emitr;//NB:cached on first use
}

/* end-of-EmitTask.java */
