/**
 * $Id: TallySet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.util.Iterator;
import  java.util.Stack;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.Available;
import  org.apache.tools.ant.taskdefs.Checksum;
import  org.apache.tools.ant.taskdefs.UpToDate;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.condition.AllSet;
import  com.idaremedia.antx.condition.AllSetTrue;
import  com.idaremedia.antx.condition.AnySet;
import  com.idaremedia.antx.condition.AnySetTrue;
import  com.idaremedia.antx.condition.FileNotEmpty;
import  com.idaremedia.antx.condition.IsAntVersion;
import  com.idaremedia.antx.condition.IsClass;
import  com.idaremedia.antx.condition.IsResource;
import  com.idaremedia.antx.condition.NoneSet;

/**
 * A shareable set of availability checks. A TallySet evaluates all of its nested checks
 * regardless of their individual results.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;<b>tallyset</b> id="tally.j2se.libs"&gt;
 *      &lt;isclass name="java.lang.StrictMath" trueproperty="atleast-j2se12-present"/&gt;
 *      &lt;isclass name="java.sql.Clob" trueproperty="atleast-j2se13-present"/&gt;
 *      &lt;isclass name="java.nio.ReadOnlyBufferException" trueproperty="atleast-j2se14-present"/&gt;
 *      &lt;isclass name="java.lang.Enum" trueproperty="atleast-j2se15-present"/&gt;
 *   &lt;/tallyset&gt;
 *
 *   &lt;<b>tallyset</b> id="tally.testing.thirdparty.libs"&gt;
 *      &lt;isclass name="junit.framework.Assert" trueproperty="junit-present"/&gt;
 *      &lt;isclass name="org.apache.log4j.Logger" trueproperty="log4j-present"/&gt;
 *   &lt;/tallyset&gt;
 *
 *   &lt;<b>tallyset</b> id="tally.libs"&gt;
 *      &lt;<b>tallyset</b> refid="tally.j2se.libs"/&gt;
 *      &lt;<b>tallyset</b> refid="tally.testing.thirdparty.libs"/&gt;
 *   &lt;/tallyset&gt;
 *
 *   &lt;<b>tallyset</b> id="tally.test.env"&gt;
 *      &lt;<b>tallyset</b> refid="tally.j2se.libs"/&gt;
 *      &lt;<b>tallyset</b> refid="tally.testing.thirdparty.libs"/&gt;
 *      &lt;allset trueproperty="base-test-env-present"&gt;
 *        &lt;property name="atleast-j2se13-present"/&gt;
 *        &lt;property name="junit-present"/&gt;
 *      &lt;/allset&gt;
 *   &lt;/tallyset&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @.expects Ant 1.6 or later for automatic freeform
 * @see      TallyTask
 **/

public class TallySet extends RuleType implements FreeformRule
{
    /**
     * Initializes a new tallyset.
     **/
    public TallySet()
    {
        super(AntX.rules+"tally");
        setStopQuickEvaluation(false);
    }


    /**
     * Initializes a new CV-labeled tallyset.
     * @param iam CV-label (non-null)
     **/
    public TallySet(String iam)
    {
        super(iam);
        setStopQuickEvaluation(false);
    }


    /**
     * Returns a clone of this tallyset. If this tallyset is a reference
     * to another set, that object's clone method is invoked.
     **/
    public Object clone()
    {
        if (isReference()) {
            return getReference().clone();
        }
        return super.clone();
    }


    /**
     * Set this tallyset to evaluate as its own definition another tallyset.
     * Saves the reference for conversion to a livin' tallyset when needed.
     * @param r reference to other tallyset (non-null)
     * @throws BuildException if this tallyset already has own conditions
     **/
    public void setRefId(Reference r)
    {
        require_(r!=null,"setRefId- nonzro ref");
        checkModify("setRefId");
        if (!isEmpty()) {
            throw new BuildException(uistrs().get("task.too.many.attrs",getId()));
        }
        m_ref = r;
    }


    /**
     * Returns this tallyset's reference. Returns <i>null</i> if no
     * reference has been set.
     **/
    public final Reference getRefId()
    {
        return m_ref;
    }


    /**
     * Returns <i>true</i> if this tallyset is setup as a reference to
     * another tallyset.
     */
    public boolean isReference()
    {
        return getRefId()!=null;
    }


    /**
     * Returns the effect if this a tallyset; by default returns
     * WARNING (no build stop).
     **/
    public NoiseLevel getFailureEffect()
    {
        return NoiseLevel.WARNING;
    }


// ---------------------------------------------------------------------------------------
// Nestable Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Adds an arbitrary application-defined condition to
     * this tallyset.
     * @param c custom condition definition (non-null)
     * @since JWare/AntX 0.3
     **/
    public void addConfigured(Condition c)
    {
        require_(c!=null,"add- nonzro condition");
        xaddCondition(c);
    }


    /**
     * Adds a sub-tally &lt;tallyset&gt; to this tallyset.
     **/
    public void addTallyset(TallySet tt)
    {
        xaddCondition(tt);
    }


    /**
     * Adds an &lt;isclass&gt; condition to this tallyset.
     **/
    public void addIsClass(IsClass isc)
    {
        xaddCondition(isc);
    }


    /**
     * Adds an &lt;isresource&gt; condition to this tallyset.
     **/
    public void addIsResource(IsResource isr)
    {
        xaddCondition(isr);
    }


    /**
     * Adds a &lt;filenotempty&gt; condition to this tallyset.
     **/
    public void addFileNotEmpty(FileNotEmpty fne)
    {
        xaddCondition(fne);
    }


    /**
     * Adds an &lt;allset&gt; condition to this tallyset.
     **/
    public void addAllSet(AllSet allset)
    {
        xaddCondition(allset);
    }


    /**
     * Adds an &lt;alltrue&gt; condition to this tallyset.
     * @since JWare/AntX 0.5
     **/
    public void addAllSetTrue(AllSetTrue allset)
    {
        xaddCondition(allset);
    }


    /**
     * Adds an &lt;anyset&gt; condition to this tallyset.
     **/
    public void addAnySet(AnySet anyset)
    {
        xaddCondition(anyset);
    }


    /**
     * Adds an &lt;anytrue&gt; condition to this tallyset.
     * @since JWare/AntX 0.5
     **/
    public void addAnySetTrue(AnySetTrue anyset)
    {
        xaddCondition(anyset);
    }


    /**
     * Adds a &lt;noneset&gt; condition to this tallyset.
     **/
    public void addNoneSet(NoneSet noneset)
    {
        xaddCondition(noneset);
    }


    /**
     * Adds an &lt;ismatch&gt; (task) condition to this rule.
     * @since JWare/AntX 0.3
     **/
    public void addIsMatch(MatchesTask matches)
    {
        xaddCondition(matches);
    }


    /**
     * Adds an &lt;available&gt; condition to this tallyset.
     **/
    public void addAvailable(Available av)
    {
        xaddCondition(av);
    }


    /**
     * Adds a &lt;checksum&gt; condition to this tallyset.
     **/
    public void addChecksum(Checksum cs)
    {
        xaddCondition(cs);
    }


    /**
     * Adds an &lt;uptodate&gt; condition to this tallyset.
     **/
    public void addUpToDate(UpToDate up)
    {
        xaddCondition(up);
    }


    /**
     * Adds an &lt;antversion&gt; condition to this tallyset.
     * @since JWare/AntX 0.4
     **/
    public void addAntVersion(IsAntVersion vc)
    {
        xaddCondition(vc);
    }

// ---------------------------------------------------------------------------------------
// Managing Referrals:
// ---------------------------------------------------------------------------------------

    /**
     * Appends new condition to this tallyset's conditions sequence if
     * its not a reference.
     * @throws BuildException if tallyset is a reference or is frozen
     **/
    public void xaddCondition(Condition c)
    {
        if (isReference()) {
            throw new BuildException(uistrs().get("task.too.many.attrs",getId()));
        }
        super.xaddCondition(c);
    }


    /**
     * Appends the only condition allowed for this tallyset.
     * @param c new root conditon (non-null)
     * @return <i>true</i> if could add as only condition
     * @throws BuildException if tallyset is a reference or is frozen
     **/
    public boolean xaddRootCondition(Condition c)
    {
        if (isReference()) {
            throw new BuildException(uistrs().get("task.too.many.attrs",getId()));
        }
        return super.xaddRootCondition(c);
    }


    /**
     * Returns the strongly-typed tallyset to which we refer. Must be
     * a reference. Does not verify reference other than to check its type.
     **/
    protected final TallySet getReference()
    {
        verify_(m_ref!=null,"getReference- has refid");
        Object robj = m_ref.getReferencedObject(getProject());

        if (!(robj instanceof TallySet)) {
            String ermsg = uistrs().get("brul.bad.ruleid",m_ref.getRefId());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
        return (TallySet)robj;
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------


    /**
     * Ensure this tallyset does not contain any circular references
     * in its definition. Checks references as well as tallyset's
     * nested elements.
     * @param stk stack of referred-to (or used) rules (non-null)
     * @param clnt call controls (non-null)
     * @throws BuildException if circular dependency discovered
     * @.impl sets latch after 1st successful call
     **/
    public void verifyNoCircularDependency(Stack stk, Requester clnt)
    {
        if (!m_circularityChecked) {
            if (isReference()) {
                Object robj = m_ref.getReferencedObject(getProject());
                if (stk.contains(robj)) {
                    String ermsg = uistrs().get("brul.circular.referals",m_ref.getRefId());
                    clnt.problem(ermsg,Project.MSG_ERR);
                    throw new BuildException(ermsg,clnt.getLocation());
                }
                if (robj instanceof BooleanRule) {
                    stk.push(robj);
                    ((BooleanRule)robj).verifyNoCircularDependency(stk,clnt);
                    stk.pop();
                }
                m_circularityChecked = true;
            }
            else if (!isEmpty()) {
                Stack itemstk= new Stack();
                Iterator itr= getConditions().iterator();
                int I=0;
                while (itr.hasNext()) {
                    Object oc = itr.next();
                    I++;
                    if (oc instanceof BooleanRule) {
                        if (stk.contains(oc)) {
                            String posI = getId()+"["+String.valueOf(I)+"]";
                            String ermsg = uistrs().get("brul.circular.referals",posI);
                            clnt.problem(ermsg,Project.MSG_ERR);
                            throw new BuildException(ermsg,clnt.getLocation());
                        }
                        itemstk.addAll(stk);
                        itemstk.push(oc);
                        ((BooleanRule)oc).verifyNoCircularDependency(itemstk,clnt);
                        itemstk.clear();
                    }
                }
                m_circularityChecked = true;
            }
        }//!checked
    }


    /**
     * Evaluates this tallyset as a root instance. If is reference
     * calls referred-to-tallyset's evaluation method.
     * @throws BuildException if an error occurs during evaluation
     **/
    public boolean eval(ShareableConditionUser calr)
        throws BuildException
    {
        if (isReference()) {
            return getReference().eval(calr);
        }
        return super.eval(calr);
    }


    /**
     * Evaluates this tallyset as a nested component of another
     * tallyset. If is reference calls referred-to-tallyset's eval
     * method.
     * @throws BuildException if an error occurs during evaluation
     **/
    public boolean eval()
        throws BuildException
    {
        if (isReference()) {
            return getReference().eval();
        }
        return super.eval();
    }


    /**
     * Default evaluation result handler method does nothing except
     * log diagnostics information.
     **/
    protected void setEvalResult(boolean istrue, String listing)
    {
        if (istrue) {
            log("TallySet ("+getId()+") evaluates TRUE",Project.MSG_DEBUG);
        } else {
            log("TallySet ("+getId()+") evaluates FALSE; condition in order "+listing,
                Project.MSG_DEBUG);
        }
    }


    private Reference m_ref;
}

/* end-of-TallySet.java */
