/**
 * $Id: ShortHandHelper.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ProjectComponent;
import  org.apache.tools.ant.RuntimeConfigurable;
import  org.apache.tools.ant.taskdefs.Available;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.taskdefs.condition.Http;
import  org.apache.tools.ant.taskdefs.condition.IsFalse;
import  org.apache.tools.ant.taskdefs.condition.IsTrue;
import  org.apache.tools.ant.taskdefs.condition.Os;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.StringEquality;

import  com.idaremedia.antx.condition.AllSet;
import  com.idaremedia.antx.condition.AnySet;
import  com.idaremedia.antx.condition.FlexCondition;
import  com.idaremedia.antx.condition.IsAntVersion;
import  com.idaremedia.antx.condition.IsBoolean;
import  com.idaremedia.antx.condition.IsNotSet;
import  com.idaremedia.antx.condition.IsNotWhitespace;
import  com.idaremedia.antx.condition.IsNumeric;
import  com.idaremedia.antx.condition.IsReference;
import  com.idaremedia.antx.condition.IsSet;
import  com.idaremedia.antx.condition.IsSetTrue;
import  com.idaremedia.antx.condition.NoneSet;
import  com.idaremedia.antx.condition.ShortHandConditions;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.IgnoreWhitespaceEnabled;
import  com.idaremedia.antx.parameters.IsA;
import  com.idaremedia.antx.parameters.MalformedCheckEnabled;
import  com.idaremedia.antx.parameters.SynonymsEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.ValueMatchEnabled;

/**
 * Shareable implementation of the {@linkplain ShortHandConditions} interface. Public-facing
 * rule tasks can implement the entire ShortHandConditions interface by delegating to
 * an instance of this class. Implementation Note: the owning rule <em>must</em> synchronize
 * this helper's project with its own; otherwise, none of the conditions created by this
 * helper will function.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      AssertTask
 * @see      PreferTask
 **/

public final class ShortHandHelper extends AssertableProjectComponent
    implements ShortHandConditions, ValueMatchEnabled
{
    /**
     * Initializes a new condition short hand helper for specific
     * rule.
     * @param rule owning rule (non-null)
     **/
    public ShortHandHelper(BooleanRule rule)
    {
        super(rule.cvlabel_());
        m_rule = rule;
    }


    /**
     * Initializes a new CV-labeled condition short hand helper
     * for specific rule
     * @param iam CV-label (non-null)
     * @param rule owning rule (non-null)
     **/
    public ShortHandHelper(String iam, BooleanRule rule)
    {
        super(iam);
        require_(rule!=null,"ctor- nonzro rul");
        m_rule = rule;
    }


    /**
     * Returns the rule with which this helper is associated.
     * Never returns <i>null</i>; defined at construction.
     **/
    public BooleanRule getOwningRule()
    {
        return m_rule;
    }


    /**
     * Returns <i>true</i> if this short-hand helper has been activated
     * with a short-hand condition. If <i>true</i> owning rule will allow
     * only the short-hand root condition at evaluation time.
     **/
    public final boolean isActivated()
    {
        return m_isActivated;
    }


    /**
     * Marks this helper as activated into short-hand (single
     * condition) mode.
     * @param c inlined conditon (root condition)
     * @param who identifier used in error message if already activated
     **/
    public final void activateCondition(Condition c, String who)
    {
        boolean ok= m_rule.xaddRootCondition(c);
        ensure_(ok,who+BooleanRule.ONLY_ONE);
        applyPendingModifiers(c);
        m_isActivated = true; //Latch flipped
    }


    /**
     * Returns "true" of boolean is <i>true</i>; otherwise returns
     * "false".
     **/
    private static final String tf(boolean b)
    {
        return b ? Strings.TRUE : Strings.FALSE;
    }


// ---------------------------------------------------------------------------------------
// Latched literal-based conditions inherited from Ant:
// ---------------------------------------------------------------------------------------

    public void setIsTrue(String value)
    {
        IsTrue t= new IsTrue();
        t.setProject(getProject());
        Boolean B = Tk.string2PosBool(value);
        t.setValue(B==null ? false : B.booleanValue());
        activateCondition(t,"setIsTrue");
    }


    public void setIsFalse(String value)
    {
        IsFalse f= new IsFalse();
        f.setProject(getProject());
        Boolean B = Tk.string2NegBool(value);
        f.setValue(B==null ? false : B.booleanValue());
        activateCondition(f,"setIsFalse");
    }


    public void setOSFamily(String family)
    {
        Os t= new Os(family);
        activateCondition(t,"setOSFamily");
    }


    public void setOS(String selector)
    {
        IsOS t= new IsOS(getProject());
        t.setSelector(selector);
        activateCondition(t,"setOS");
    }


    public void setHttpAlive(String url)
    {
        Http t= new Http();
        t.setProject(getProject());
        t.setUrl(url);
        activateCondition(t,"setHttpAlive");
    }


    public void setIsFile(File file)
    {
        Available t= getIsAvailableCondition();
        t.setFile(file);
        Available.FileDir filetype= new Available.FileDir();
        Tk.initEnum(filetype,"file");
        t.setType(filetype);
        activateCondition(t,"setIsFile");
    }


    public void setIsDirectory(File dir)
    {
        Available t= getIsAvailableCondition();
        t.setFile(dir);
        Available.FileDir dirtype= new Available.FileDir();
        Tk.initEnum(dirtype,"dir");
        t.setType(dirtype);
        activateCondition(t,"setIsDirectory");
    }


    public void setFilepath(Path filp)
    {
        m_rule.checkModify("setFilpath");
        Available t= getIsAvailableCondition();
        t.setFilepath(filp);
    }


    public void setIsResource(String resource)
    {
        Available t= getIsAvailableCondition();
        t.setResource(resource);
        activateCondition(t,"setIsResource");
    }


    public void setIsClass(String classname)
    {
        Available t= getIsAvailableCondition();
        t.setClassname(classname);
        activateCondition(t,"setIsClass");
    }


    public void setClasspath(Path clsp)
    {
        m_rule.checkModify("setClazpath");
        Available t= getIsAvailableCondition();
        t.setClasspath(clsp);
    }


    public void setClasspathRef(Reference cpr)
    {
        m_rule.checkModify("setClzpath");
        Available t= getIsAvailableCondition();
        t.setClasspathRef(cpr);
    }


    public void setSystemClasses(boolean included)
    {
        m_rule.checkModify("setSystemClazes");
        Available t= getIsAvailableCondition();
        t.setIgnoresystemclasses(!included);
    }

// ---------------------------------------------------------------------------------------
// Latched literal-based <antversion> variants:
// ---------------------------------------------------------------------------------------

    /**
     * Defines short hand &lt;antversion is="&#8230;"/&gt;
     * condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntIs(String version)
    {
        IsAntVersion t= new IsAntVersion();
        t.setProject(getProject());
        t.setIs(version);
        activateCondition(t,"setAntIs");
    }

    /**
     * Defines short hand &lt;antversion like="&#8230;"/&gt;
     * condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntLike(String pattern)
    {
        IsAntVersion t= new IsAntVersion();
        t.setProject(getProject());
        t.setLike(pattern);
        activateCondition(t,"setAntLike");
    }


// ---------------------------------------------------------------------------------------
// Literal-based unless overridden with an 'isa' modifier:
// ---------------------------------------------------------------------------------------

    public void setIsBoolean(String value)
    {
        IsBoolean t= (IsBoolean)getHalfBakedObject(IsBoolean.class);
        t.setValue(value);
        activateCondition(t,"setIsBoolean");
    }


    public void setIsNotWhitespace(String value)
    {
        IsNotWhitespace t= (IsNotWhitespace)getHalfBakedObject(IsNotWhitespace.class);
        t.setValue(value);
        activateCondition(t,"setIsNotWSpc");
    }


    public void setIsNumeric(String value)
    {
        IsNumeric t= (IsNumeric)getHalfBakedObject(IsNumeric.class);
        t.setValue(value);
        activateCondition(t,"setIsNumeric");
    }


    /**
     * 'greather-than-or-equal' modifier for a short-hand
     * 'isnumeric' condition.
     * @since JWare/AntX 0.3
     **/
    public void setGT(String gt)
    {
        m_rule.checkModify("setGT");
        IsNumeric t = (IsNumeric)getHalfBakedObject("gt",gt);
        if (t!=null) {
            t.setGT(Tk.longFrom(gt,Tk.NO_NUM));
        }
    }


    /**
     * 'greather-than' modifier for a short-hand 'isnumeric'
     * condition.
     * @since JWare/AntX 0.3
     **/
    public void setGTE(String gte)
    {
        m_rule.checkModify("setGTE");
        IsNumeric t = (IsNumeric)getHalfBakedObject("gte",gte);
        if (t!=null) {
            t.setGTE(Tk.longFrom(gte,Tk.NO_NUM));
        }
    }


    /**
     * 'less-than' modifier for a short-hand 'isnumeric' condition.
     * @since JWare/AntX 0.3
     **/
    public void setLT(String lt)
    {
        m_rule.checkModify("setLT");
        IsNumeric t = (IsNumeric)getHalfBakedObject("lt",lt);
        if (t!=null) {
            t.setLT(Tk.longFrom(lt,Tk.NO_NUM));
        }
    }


    /**
     * 'less-than-or-equal' modifier for a short-hand 'isnumeric'
     * condition.
     * @since JWare/AntX 0.3
     **/
    public void setLTE(String lte)
    {
        m_rule.checkModify("setLTE");
        IsNumeric t = (IsNumeric)getHalfBakedObject("lte",lte);
        if (t!=null) {
            t.setLTE(Tk.longFrom(lte,Tk.NO_NUM));
        }
    }

// ---------------------------------------------------------------------------------------
// Property-based unless overridden with an 'isa' modifier:
// ---------------------------------------------------------------------------------------

    public void setIsSet(String property)
    {
        IsSet t= (IsSet)getHalfBakedObject(IsSet.class);
        t.setProperty(property);
        activateCondition(t,"setIsSet");
    }


    public void setIsSetTrue(String property)
    {
        IsSetTrue t= (IsSetTrue)getHalfBakedObject(IsSetTrue.class);
        t.setProperty(property);
        activateCondition(t,"setIsSetTrue");
    }


    public void setIsNotSet(String property)
    {
        IsNotSet t= (IsNotSet)getHalfBakedObject(IsNotSet.class);
        t.setProperty(property);
        activateCondition(t,"setIsNotSet");
    }


// ---------------------------------------------------------------------------------------
// Latched variable-based <isset>,<isnotset>,<issettrue>:
// ---------------------------------------------------------------------------------------

    public void setVarSet(String variable)
    {
        IsSet t= (IsSet)getHalfBakedObject(IsSet.class);
        t.setVariable(variable);
        disableIsA();
        activateCondition(t,"setVarSet");
    }


    public void setVarSetTrue(String variable)
    {
        IsSetTrue t= (IsSetTrue)getHalfBakedObject(IsSetTrue.class);
        t.setVariable(variable);
        disableIsA();
        activateCondition(t,"setVarSetTrue");
    }


    public void setVarNotSet(String variable)
    {
        IsNotSet t= (IsNotSet)getHalfBakedObject(IsNotSet.class);
        t.setVariable(variable);
        disableIsA();
        activateCondition(t,"setVarNotSet");
    }


// ---------------------------------------------------------------------------------------
// Latched reference-based <isreference>:
// ---------------------------------------------------------------------------------------


    /**
     * Parameter short-hand for a 'isreference' condition.
     **/
    public void setIsRef(String refId)
    {
        IsReference t= (IsReference)getHalfBakedObject(IsReference.class);
        t.setName(refId);
        activateCondition(t,"setIsRef");
    }


    /**
     * Parameter short-hand for a &lt;not&gt;'isreference' condition.
     * @since JWare/AntX 0.5
     **/
    public void setIsNotRef(String refId)
    {
        IsReference t= (IsReference)getHalfBakedObject(IsReference.class);
        t.setName(refId);
        t.xsetNoExist(true);
        activateCondition(t,"setIsNotRef");
    }


    /**
     * Modifier for a short-hand for a 'isreference' condition.
     **/
    public void setClassName(String clzId)
    {
        m_rule.checkModify("setRefClaz");
        IsReference ref= getIsReferenceCondition();
        ref.setClassName(clzId);
    }


    /**
     * Modifier for a short-hand for a 'isreference' condition.
     **/
    public void setIsKindOf(boolean isKindOf)
    {
        m_rule.checkModify("setRefKndOf");
        IsReference ref= getIsReferenceCondition();
        ref.setKindOf(isKindOf);
    }


// ---------------------------------------------------------------------------------------
// Latched properties-based <allset>,<allsetLike>,<noneset>,<anyset>:
// ---------------------------------------------------------------------------------------

    public void setAnySet(String properties)
    {
        require_(!Tk.isWhitespace(properties),"setAnySet- propertylist");
        AnySet t= new AnySet(properties,getProject());
        setHalfBakedObject(t);
        activateCondition(t,"setAnySet");
    }


    public void setAllSet(String properties)
    {
        require_(!Tk.isWhitespace(properties),"setAllSet- propertylist");
        AllSet t= new AllSet(properties,getProject());
        setHalfBakedObject(t);
        activateCondition(t,"setAllSet");
    }


    /**
     * Like {@linkplain #setAllSet setAllSet()} but against the set
     * of existing script properties.
     * @param pattern the filtering pattern
     * @since JWare/AntX 0.4
     **/
    public void setAllSetLike(String pattern)
    {
        require_(!Tk.isWhitespace(pattern),"setAllSetLike- pattern");
        AllSet t= new AllSet();
        t.setProject(getProject());
        t.setPropertiesLike(pattern);
        setHalfBakedObject(t);
        activateCondition(t,"setAllSetLike");
    }


    public void setNoneSet(String properties)
    {
        require_(!Tk.isWhitespace(properties),"setNoneSet- propertylist");
        NoneSet t= new NoneSet(properties,getProject());
        setHalfBakedObject(t);
        activateCondition(t,"setNoneSet");
    }


// ---------------------------------------------------------------------------------------
// Shorthand (flex) string comparision condition and parameters:
// ---------------------------------------------------------------------------------------

    public void setMatches(String pattern)
    {
        StringEquality t= getEqualityCondition();
        t.setMatch(pattern);
        activateCondition(t,"setMatches");
    }


    public final void setMatch(String pattern)
    {
        setMatches(pattern);
    }


    public void setVariable(String variable)
    {
        m_rule.checkModify("setStrEqVar");
        StringEquality eqt = getEqualityCondition();
        eqt.setUnknownArg(variable);
        eqt.getUnknownValueGetter().setIsExported(true);
    }


    public void setProperty(String property)
    {
        m_rule.checkModify("setStrEqProp");
        StringEquality eqt = getEqualityCondition();
        eqt.setUnknownArg(property);
        eqt.getUnknownValueGetter().setIsProperty(true);
    }


    public void setReference(String reference)
    {
        m_rule.checkModify("setStrEqRef");
        StringEquality eqt = getEqualityCondition();
        eqt.setUnknownArg(reference);
        eqt.getUnknownValueGetter().setIsReference(true);
    }


    public void setValue(String value)
    {
        m_rule.checkModify("setStrEqValu");
        Object t = getHalfBakedObject("value",value);
        if (t!=null) {
            ((ValueMatchEnabled)t).setValue(value);
        }
    }

// ---------------------------------------------------------------------------------------
// Multiple-condition options (stuf that applies to more than one condition):
// ---------------------------------------------------------------------------------------

    public void setSynonyms(boolean allowAll)
    {
        m_rule.checkModify("setSynonyms");
        Object t = getHalfBakedObject("synonyms",tf(allowAll));
        if (t!=null) {
            ((SynonymsEnabled)t).setSynonyms(allowAll);
        }
    }


    public boolean allowSynonyms()
    {
        if (getHalfBakedObject() instanceof SynonymsEnabled) {
            return ((SynonymsEnabled)getHalfBakedObject()).allowSynonyms();
        }
        return true;//Ant-default
    }


    public void setIgnoreCase(boolean ignore)
    {
        m_rule.checkModify("setIgnoreCase");
        Object t = getHalfBakedObject("ignorecase",tf(ignore));
        if (t!=null) {
            ((IgnoreCaseEnabled)t).setIgnoreCase(ignore);
        }
    }


    public boolean isIgnoreCase()
    {
        if (getHalfBakedObject() instanceof IgnoreCaseEnabled) {
            return ((IgnoreCaseEnabled)
                    getHalfBakedObject()).isIgnoreCase();
        }
        return false;
    }


    public void setTrim(boolean trim)
    {
        m_rule.checkModify("setTrim");
        Object t = getHalfBakedObject("trim",tf(trim));
        if (t!=null) {
            ((TrimEnabled)t).setTrim(trim);
        }
    }


    public boolean willTrim()
    {
        if (getHalfBakedObject() instanceof TrimEnabled) {
            return ((TrimEnabled)getHalfBakedObject()).willTrim();
        }
        return false;
    }


    public void setWhitespace(Handling response)
    {
        m_rule.checkModify("setWhtSpc");
        Object t = getHalfBakedObject("whitespace",response.getValue());
        if (t!=null) {
            ((IgnoreWhitespaceEnabled)t).setWhitespace(response);
        }
    }


    public boolean ignoreWhitespace()
    {
        if (getHalfBakedObject() instanceof IgnoreWhitespaceEnabled) {
            return ((IgnoreWhitespaceEnabled)
                    getHalfBakedObject()).ignoreWhitespace();
        }
        return false;
    }


    public void setMalformed(Handling response)
    {
        m_rule.checkModify("setMalformd");
        Object t = getHalfBakedObject("malformed",response.getValue());
        if (t!=null) {
            ((MalformedCheckEnabled)t).setMalformed(response);
        }
    }


    public Handling getMalformedHandling()
    {
        if (getHalfBakedObject() instanceof MalformedCheckEnabled) {
            return ((MalformedCheckEnabled)
                    getHalfBakedObject()).getMalformedHandling();
        }
        return null;
    }


// ---------------------------------------------------------------------------------------
// Type cast for flexible conditions that support value,property,variable,references:
// ---------------------------------------------------------------------------------------

    /**
     * Tells this helper that the 'isa' option is no longer a valid
     * option. Used if the type is fixed by the shorthand condition's name
     * like "<i>varnotset</i>". Latch operation; once set cannot be unset.
     **/
    public void disableIsA()
    {
        m_disableIsA=true;
    }


    /**
     * Returns <i>true</i> if the 'isa' option is no longer a valid
     * choice.
     **/
    public boolean noIsA()
    {
        return m_disableIsA;
    }


    /**
     * Setter method to type-cast an inlined flex condition. Flex
     * conditions are usually defaulted to literal values if never
     * type-cast. This 'isa' parameter is unique to our rule tasks--
     * it is a condensed version of the explicit FlexValueSupport
     * interface. Soooo, we cannot put it with the regular "delayed
     * modifiers" configurable; we need to interpret it again later.
     * @param isa the explicit type-cast (non-null)
     **/
    public void setIsA(IsA isa)
    {
        m_rule.checkModify("setIsA");

        require_(!noIsA(),"setIsA- Flex condition");

        Object t = getHalfBakedObject();
        if (t==null && m_delayedIsA==null) {//Wait-for-target-c!
            m_delayedIsA = new IsAHandle(isa);
            return;
        }

        require_((t instanceof FlexCondition),"setIsA- Flex condition");

        FlexCondition c = (FlexCondition)t;

        switch (isa.getIndex()) {
            case IsA.PROPERTY_INDEX: {
                c.setProperty(c.getFlexValue());
                break;
            }
            case IsA.VARIABLE_INDEX: {
                c.setVariable(c.getFlexValue());
                break;
            }
            case IsA.REFERENCE_INDEX:{
                c.setReference(c.getFlexValue());
                break;
            }
            default: {
                c.setLiteral(c.getFlexValue());
            }
        }
    }

// ---------------------------------------------------------------------------------------
// Multiple-option inlined attributes:
// ---------------------------------------------------------------------------------------

    /**
     * Getter/factory-method for a multi-option inlined condition.
     * This method create a new condition instance if one does not
     * already exist. Note that the generated condition must be a
     * self-validating entity because this rule won't be able to tell
     * if the condition is "fully baked" before trying to use it
     *(hence the colorful name). Pending modifiers will be applied
     * when a new condition is activated.
     * @see #activateCondition activateCondition
     **/
    public final Object getHalfBakedObject(Class ofClass)
    {
        if (m_halfbakedObject==null) {
            try {
                m_halfbakedObject = ofClass.newInstance();
            } catch(Exception anyX) {
                throw new BuildException(anyX,m_rule.getLocation());
            }
            if (m_halfbakedObject instanceof ProjectComponent) {
                ((ProjectComponent)m_halfbakedObject).setProject(getProject());
            }
        }
        else {
            require_(ofClass.isInstance(m_halfbakedObject),
                     "construction of a single condition-type inline");
        }
        return m_halfbakedObject;
    }


    /**
     * Returns raw underlying half-baked condition as-is. Will return
     * <i>null</i> if short hand condition not yet created.
     **/
    public final Object getHalfBakedObject()
    {
        return m_halfbakedObject;
    }


    /** Internal method that automatically inserts the modifier to the
     *  pending list if there is no active condition.
     *  @since JWare/AntX 0.4
     **/
    private Object getHalfBakedObject(String attr, String saved)
    {
        Object t = getHalfBakedObject();
        if (t==null) {
            getPendingModifiers().setAttribute(attr,saved);
        }
        return t;
    }


    /**
     * Initializes the raw underlying half-baked object directly. Use
     * when target condition is created outside our getter/factory
     * API.
     **/
    private void setHalfBakedObject(Object object)
    {
        m_halfbakedObject = object;
    }


    /**
     * Getter/factory method for an inlined string comparision
     * condition like <span class="src">equals</span> or
     * <span class="src">matches</span>. Never returns <i>null</i>.
     **/
    StringEquality getEqualityCondition()
    {
        return (StringEquality)getHalfBakedObject(StringEquality.class);
    }


    /**
     * Getter/factory method for an inlined is-available
     * condition like <span class="src">isfile</span> or
     * <span class="src">isresource</span>. Never returns <i>null</i>.
     **/
    private Available getIsAvailableCondition()
    {
        return (Available)getHalfBakedObject(Available.class);
    }


    /**
     * Getter/factory method for an inlined
     * <span class="src">isreference</span> condition. Never returns
     * <i>null</i>.
     **/
    private IsReference getIsReferenceCondition()
    {
        return (IsReference)getHalfBakedObject(IsReference.class);
    }


// ---------------------------------------------------------------------------------------
// MacroDef+PreDef Support: Handle reordered parameter setting (Ant1.6)
// ---------------------------------------------------------------------------------------

    private static final Object NOTHIN= new int[0];


    /**
     * Returns <i>true</i> if this helper has parameters
     * configuration that is pending. Parameter application can
     * be delayed when the Ant runtime reorders the parsed
     * ordering of item attributes (like MacroDef does).
     * @since JWare/AntX 0.4
     **/
    public final boolean havePendingModifiers()
    {
        return m_delayedParams!=null || m_delayedIsA!=null;
    }


    /**
     * Applies any pending parameter configuration to given test.
     * Also clears the set of pending items. No-op if no pending
     * operations.
     * @since JWare/AntX 0.4
     **/
    public final void applyPendingModifiers(Object test)
    {
        require_(test!=null,"applyMods- nonzro condition");
        if (m_delayedParams!=null) {
            m_delayedParams.setProxy(test);
            RuntimeConfigurable helper = m_delayedParams;
            m_delayedParams = null;
            helper.reconfigure(getProject());
            helper = null;
        }
        if (m_delayedIsA!=null) {
            setIsA(m_delayedIsA.get());
            m_delayedIsA = null;
        }
    }


    /**
     * Returns a holder for all modifiers that must wait for their
     * target condition to be created. Never returns <i>null</i>.
     * @see #applyPendingModifiers applyPendingModifiers
     * @since JWare/AntX 0.4
     **/
    private RuntimeConfigurable getPendingModifiers()
    {
        if (m_delayedParams==null) {
            m_delayedParams = new RuntimeConfigurable(NOTHIN,"shh");
        }
        return m_delayedParams;
    }


// ---------------------------------------------------------------------------------------
// Seekrit Members:
// ---------------------------------------------------------------------------------------

    private Object m_halfbakedObject;//NB:to support multi-option inlined conditons
    private final BooleanRule m_rule;
    private boolean m_disableIsA;//NB:latch off; allow by default
    private boolean m_isActivated;//NB:false=>no short-hand condition has been set
    private RuntimeConfigurable m_delayedParams;//NB:if non-null need to apply
    private IsAHandle m_delayedIsA;//NB:if non-null need to apply
}

/* end-of-ShortHandHelper.java */
