/**
 * $Id: BatchChecksTaskSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.helpers.TaskHandle;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Aggregating taskset that always executes each of its nested assertions and/or fixture
 * checks tasks even if one or more of them fail. Once all checks have been performed,
 * this taskset will propagate a wrapper assertion exception if at least one check failed.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;<b>batchchecks</b> failproperty="..."&gt;
 *       &lt;fixturecheck isset="module_basedir" msgid="..."/&gt;
 *       &lt;assert allset="module_id,module_name" whitespace="reject" msgid="..."&gt;
 *    &lt;/batchchecks&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,infra
 * @see       AssertTask
 * @see       VerifyFixture
 **/

public final class BatchChecksTaskSet extends TaskSet implements Condition, AntLibFriendly
{
    /**
     * Initializes a new batch checks taskset.
     **/
    public BatchChecksTaskSet()
    {
        super(AntX.conditions+"BatchChecks:");
    }



    /**
     * Sets name of the property set if this batchcheck fails.
     * @param property the property to update (non-null, non-whitespace)
     **/
    public final void setFailProperty(String property)
    {
        require_(!Tk.isWhitespace(property),"setfailp- nonwspc");
        m_failProperty = property;
    }


    /**
     * Returns the property updated if this assertion fails. Returns
     * default property-name if this attribute never set.
     **/
    public final String getFailProperty()
    {
        return m_failProperty;
    }



    /**
     * Returns <i>true</i> if candidate task is a boolean rule.
     * @param taskH the task to be nested (as handle)
     **/
    protected boolean includeTask(TaskHandle taskH)
    {
        Class c = TaskExaminer.trueClass(taskH.getTask());
        return c!=null && BooleanRule.class.isAssignableFrom(c);
    }



    /**
     * Runs each nested assertion or fixture check task in order
     * capturing any thrown signals until all checks have been done
     * @throws BuildException if any nested check does.
     **/
    protected void performNestedTasks() throws BuildException
    {
        Task[] tasks = getTasks();
        int signals = 0;
        for (int i=0;i<tasks.length;i++) {
            try {
                tasks[i].perform();
            } catch(RuntimeException rtX) {
                signals++;
                if (!(rtX instanceof BuildAssertionException)) {//make sure it's seen!
                    log(rtX.getMessage(), Project.MSG_ERR);
                }
            }
        }
        if (signals>0) {
            String what = String.valueOf(signals);
            String msg = getAntXMsg("brul.batch.asserts.failed",what);
            if (m_failProperty!=null) {
                checkIfProperty_(m_failProperty,true);
                getProject().setNewProperty(m_failProperty, what);
                log("BatchCheck false; setting failure property "+m_failProperty+
                    " to "+what, Project.MSG_DEBUG);
            } else {
                log(msg, Project.MSG_WARN);
            }
            throw new BuildAssertionException(msg,getLocation());
        }
    }
    
    
    /**
     * Executes this collection of checks. Will always return
     * <i>true</i> if all assertions pass; otherwise an assertion
     * exception is always signalled (nothing returned).
     * @throws BuildException if any nested check does.
     **/
    public boolean eval()
    {
        execute();
        return true;
    }


    private String m_failProperty;//NB:null => none!
}

/* end-of-BatchChecks.java */