/**
 * $Id: Matches.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.StringEquality;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.ValueMatchEnabled;

/**
 * Condition that evaluates <i>true</i> if a string value matches a known regular
 * expression. Usually defined as &lt;matches&gt;;for example:<pre>
 *  &lt;assert ...&gt;
 *    &lt;matches pattern="(internal)|(distribution)" value="${build.type}"/&gt;
 *    &lt;matches pattern="[0-9]+" property="build.number"/&gt;
 *    &lt;matches pattern="(true)|(yes)|(okidoki)|(on)" var="_loop.haltiferror"/&gt;
 *  &lt;/assert&gt;
 *
 *  &lt;tally trueproperty="loop.DIE"&gt;
 *    &lt;matches pattern="(true)|(yes)|(okidoki)|(on)" var="_loop.haltiferror"/&gt;
 *  &lt;/tally&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      StringEquality
 * @see      com.idaremedia.antx.condition.solo.MatchesTask MatchesTask
 **/

public final class Matches extends FlexCondition
    implements IgnoreCaseEnabled, TrimEnabled, ValueMatchEnabled
{
    /**
     * Initializes new Matches condition.
     **/
    public Matches()
    {
        super();
        m_impl.setOperator(StringEquality.OP_MATCHES);
        getValueHelper().setLenient(false);
    }


    /**
     * Initializes a new Matches condition with specific pattern.
     * @param pattern regular expression to be evaluated against
     * @param P condition's project
     **/
    public Matches(String pattern, Project P)
    {
        this();
        setProject(P);
        setPattern(pattern);
    }


    /**
     * Initializes the enclosing project of this condition.
     * Also updates this condition's helper bits.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_impl.setProject(P);
    }


    /**
     * Returns the underlying value's flex string for this condition.
     * Never returns <i>null</i>.
     **/
    protected final FlexString getValueHelper()
    {
        return m_impl.getUnknownValueGetter();
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this condition's regular expression pattern.
     * @param pattern pattern against which value matched (non-null)
     **/
    public void setPattern(String pattern)
    {
        require_(pattern!=null,"setPattern- nonzro");
        m_impl.setKnownArg(pattern);
    }


    /**
     * Returns pattern against which values matched. Returns
     * <i>null</i> if never set.
     **/
    public final String getPattern()
    {
        return m_impl.getKnownArg();
    }


    /**
     * Synonym for {@linkplain #setPattern setPattern()} that is
     * more natural for some uses of a Matches condition.
     * @since JWare/AntX 0.4
     **/
    public final void setMatch(String pattern)
    {
        setPattern(pattern);
    }


    /**
     * Sets this condition's to-be-matched literal value.
     * @param value value to be matched (non-null)
     **/
    public final void setValue(String value)
    {
        require_(value!=null,"setValue- nonzro");
        setLiteral(value);
    }



    /**
     * Set whether the value should be trimmed of whitespace
     * before it's compared.
     **/
    public void setTrim(boolean trim)
    {
        getValueHelper().setTrim(trim);
    }


    /**
     * Returns <i>true</i> if the value will be trimmed before
     * it's compared.
     **/
    public final boolean willTrim()
    {
        return getValueHelper().isTrimmed();
    }



    /**
     * Set whether the value should be lower-cased before
     * it's compared.
     **/
    public void setIgnoreCase(boolean ignore)
    {
        getValueHelper().setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if the value will be lower-cased
     * before it's compared.
     **/
    public final boolean isIgnoreCase()
    {
        return getValueHelper().isIgnoreCase();
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if the value matches the pattern.
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        return m_impl.eval();
    }


    /**
     * Call to verify that this match condition is in valid project
     * and has both its pattern and value defined.
     * @param calr caller's identifier
     * @throws BuildException if not in project or all bits not defined
     **/
    protected void verifyCanEvaluate_(String calr)
    {
        super.verifyCanEvaluate_(calr);

        if (getPattern()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",
                                        getTypicalName(),"pattern");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }


    private final StringEquality m_impl = new StringEquality();
}

/* end-of-Matches.java */
