/**
 * $Id: IsReference.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Simple Is-A-Reference condition check. Example usage:
 * <pre>
 *   &lt;isref name="the.err"/&gt;
 *   &lt;isref name="module.classes" class="org.apache.tools.ant.types.DirSet"/&gt;
 *   &lt;isref name="last.err" class="java.lang.Throwable" kindof="true"/&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public class IsReference extends AssertableProjectComponent
    implements Condition, URIable
{
    /**
     * Creates new IsReference condition; defaults
     * to returning <i>true</i>.
     **/
    public IsReference()
    {
    }


    /**
     * Creates pre-defined IsReference condition.
     * @param refid the refid against which condition checks
     **/
    public IsReference(String refid)
    {
        setName(refid);
    }


    /**
     * Sets reference name used by evaluation method.
     * @param refId the reference's name (non-null)
     **/
    public void setName(String refId)
    {
        m_refId = refId;
    }


    /**
     * Returns reference name used by evaluation method.
     **/
    public final String getName()
    {
        return m_refId;
    }



    /**
     * Sets this condition's reference name as part of 
     * a value URI.
     * @param fragment the value uri bits (non-null)
     * @since JWare/AntX 0.5
     */
    public void xsetFromURI(String fragment)
    {
        setName(fragment);
    }



    /**
     * Tells this condition that the reference shoudld <em>not</em>
     * exist in order for condition to pass.
     * @param noexist <i>true</i> if reference should not exist.
     * @since JWare/AntX 0.5
     **/
    public void xsetNoExist(boolean noexist)
    {
        m_noexist = noexist;
    }



    /**
     * Returns current referenced this condition would use
     * (will return <i>null</i> if never set or reference is
     * undefined or reference is out-of-scope (defined in a
     * target that has yet to execute).
     **/
    public Object getReferencedObject()
    {
        require_(getProject()!=null,"getRefObj- in project");
        String refid= Tk.resolveString(getProject(),getName());
        Object ref = null;
        if (refid!=null) {
            ref = FixtureExaminer.trueReference(getProject(),refid);
            if (ref==FixtureExaminer.IGNORED_REFERENCE) {
                ref = null;
            }
        }
        return ref;
    }


    /**
     * Sets whether reference must be of an exact class
     * match or just a "kind-of" the class.
     **/
    public void setKindOf(boolean kindOf)
    {
        m_kindOf = kindOf;
    }


    /**
     * Returns <i>true</i> if reference only has to be a
     * "kind-of" class match and not an exact class match.
     **/
    public final boolean isKindOf()
    {
        return m_kindOf;
    }


    /**
     * Sets the class the reference be an instance of.
     * @since JWare/AntX 0.5
     **/
    public void setClassName(String clazzname)
    {
        m_requiredClass = clazzname;
    }


    /**
     * (Older)Synonym for {@linkplain #setClassName setClassName()}.
     **/
    public final void setClass(String clazzname)
    {
        setClassName(clazzname);
    }


    /**
     * Returns required instance-of class class name for
     * this condition.
     * Returns <i>null</i> if never set.
     * @since JWare/AntX 0.5
     **/
    public final String getClassName()
    {
        return m_requiredClass;
    }



    /**
     * Returns <i>true</i> if the condtion's property has not
     * been explicitly set. This condition should behave reasonably 
     * even if it was loaded by a classloader different from
     * referenced object.
     **/
    public boolean eval()
    {
        verifyInProject_("eval");

        boolean result = false;
        Object object = getReferencedObject();

        if (object!=null) {
            if (!m_noexist) {
                final String classname = getClassName();
                if (classname!=null) {
                    if (object==FixtureExaminer.PROBLEM_REFERENCE) {
                        result= false;
                    } else {
                        result= object.getClass().getName().equals(classname);
                        if (result==false && isKindOf()) {
                            result = isCompatible(object,classname);
                        }
                    }
                } else {
                    result= true;//=>just need to exist!
                }
            }
        } else if (m_noexist) {
            result = true;
        }

        return result;
    }



    /**
     * Tries to determine if object is assignment compatible with
     * the named class. Has to work w/ names to avoid issues when condition
     * and object loaded by different classloaders.
     * @param object the referenced object
     * @param classname class we're interested in.
     * @return <i>true</i> if (sortof) compatible.
     * @since JWare/AntX 0.5
     **/
    private boolean isCompatible(Object object, String classname)
    {
        boolean compatible = false;
        ClassLoader cl = object.getClass().getClassLoader();
        if (cl!=null) {
            try {
                Class c = cl.loadClass(classname);
                compatible = c.isInstance(object);
            } catch(ClassNotFoundException cnfx) {
                /*burp*/
            }
        } else {
            try {
                Class c = Class.forName(classname);
                compatible = c.isInstance(object);
            }catch(ClassNotFoundException cnfx) {
                /*burp*/
            }
        }
        return compatible;
    }


    private String  m_refId;//required
    private boolean m_kindOf;//NB:=>*exact* match
    private String  m_requiredClass;//NB:most flex to support diff classpaths
    private boolean m_noexist;
}

/* end-of-IsReference.java */
