/**
 * $Id: FileNotEmpty.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  java.io.File;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.helpers.Strings;

/**
 * Shortcut condition that checks if a file exists and is not empty (its length greater
 * than zero). If the path defined is not an absolute path, this condition resolves it
 * with respect to its project.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @.ideas   If ever important (using Available as a guide) support a
 *           'filepath' attribute
 **/

public class FileNotEmpty extends AssertableProjectComponent
    implements Condition, URIable
{
    /**
     * Initializes new undefined FileNotEmpty condition.
     **/
    public FileNotEmpty()
    {
        super(AntX.rules+"filenotempty");
    }


    /**
     * Initializes new fully defined FileNotEmpty condition.
     * @param path path to be evaluated
     * @param P condition's project
     **/
    public FileNotEmpty(String path, Project P)
    {
        super(AntX.rules+"filenotempty");
        setProject(P);
        setPath(path);
    }


    /**
     * Initializes new fully defined FileNotEmpty condition.
     * @param file file to be evaluated
     * @param P condition's project
     **/
    public FileNotEmpty(File file, Project P)
    {
        super(AntX.rules+"filenotempty");
        setProject(P);
        setFile(file);
    }


    /**
     * Sets path of file to verify. The incoming path is not
     * resolved until this condition is evaluated.
     * @param path path to verify (non-null)
     **/
    public void setPath(String path)
    {
        require_(path!=null,"setpath- nonzro");
        m_filepath = path;
    }


    /**
     * Returns path of file to verify. Returns <i>null</i>
     * if never set.
     **/
    public final String getPath()
    {
        return m_filepath;
    }


    /**
     * Sets file to verify. If the incoming file  is not an
     * absolute path, it is immediately resolved relative to this
     * condition's enclosing project.
     * @param file file to verify (non-null)
     **/
    public void setFile(File file)
    {
        require_(file!=null,"setfile- nonzro");
        m_file = file;
    }


    /**
     * Returns file to verify. Returns <i>null</i>
     * if never set.
     **/
    public final File getFile()
    {
        return m_file;
    }


    /**
     * Sets property name updated by <i>true</i> evaluation.
     * @param property the property's name (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setTrueProperty(String property)
    {
        require_(property!=null,"setTrueProp- nonzro name");
        m_updateProperty = property;
    }


    /**
     * Returns property name updated by evaluation method. Returns
     * <i>null</i> if never set.
     * @since JWare/AntX 0.3
     **/
    public final String getTrueProperty()
    {
        return m_updateProperty;
    }


    /**
     * Returns <i>true</i> if the file exists, is not a directory,
     * and has a length greater than zero bytes.
     **/
    public boolean eval()
    {
        if (getTrueProperty()!=null) {
            verifyInProject_("eval");
        }

        File thefile = getFile();

        if (thefile==null && getPath()==null) {
            return false;
        }

        boolean istrue=false;
        try {
            if (thefile==null) {
                if (getProject()!=null) {
                    thefile = getProject().resolveFile(getPath());
                } else {
                    thefile = new File(getPath());
                }
            }
            if (!thefile.isDirectory()) {
                istrue = thefile.length()>0L;//NB:returns 0L if doesnt-exist
            }
        } catch (SecurityException secX) {
            log(secX.getMessage(),Project.MSG_WARN);
        }

        if (istrue && getTrueProperty()!=null) {
            String prop = getTrueProperty();
            log("FileNotEmpty was true; setting true-property '"+prop+
                "' property",  Project.MSG_DEBUG);
            getProject().setNewProperty(prop,Strings.TRUE);
        }

        return istrue;
    }



    /**
     * Sets this condition's path as part of a value URI.
     * @param fragment the value uri bits (non-null)
     * @since JWare/AntX 0.5
     */
    public void xsetFromURI(String fragment)
    {
        setPath(fragment);
    }


    private String m_filepath;
    private File m_file;
    private String m_updateProperty;
}

/* end-of-FileNotEmpty.java */
