/**
 * $Id: CapturedLogs.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureAdministrator;
import  com.idaremedia.antx.FixtureOverlays;
import  com.idaremedia.antx.FixtureIds;
import  com.idaremedia.antx.KillMethod;
import  com.idaremedia.antx.KillMethodSkeleton;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.Strings;

/**
 * Manages iteration-based log recorders for the active thread. The {@linkplain
 * AssertLoggedTask &lt;assertlogged&gt;} task uses this class to determine source of
 * logs to be verified.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      AssertLoggedTask
 * @see      CaptureLogsTask
 **/

public final class CapturedLogs implements FixtureAdministrator
{
    private static final String WHOAMI_ = "CapturedLogs";


    /**
     * The iteration category for all captured logs's information.
     **/
    public static final String ANTLOG=
        FixtureIds.ANT_LOGS_RECORDER;


    /**
     * The iteration category for all captured stdio information.
     **/
    public static final String SYSTEM=
        FixtureIds.SYSTEM_STREAMS_RECORDER;



    /**
     * Returns the current thread's frontmost LogsRecorder instance
     * if one exists. Returns <i>null</i> if no recorder installed
     * for executing thread.
     **/
    public static LogsRecorder getRecorder()
    {
        FixtureOverlays context = FixtureOverlays.getContextInstance();
        return (LogsRecorder)context.nearest(ANTLOG);
    }



    /**
     * Helper to {@linkplain #getRecorder getRecorder} that returns
     * the LogsRecorder {@linkplain EmptyLogs placeholder} if the current
     * thread has no installed recorders. Never returns <i>null</i>.
     **/
    public static LogsRecorder getRecorderNoNull()
    {
        LogsRecorder r = getRecorder();
        return (r!=null) ? r : EmptyLogs.INSTANCE;
    }



    /**
     * Installs a new LogsRecorder instance for the current thread. This
     * recorder becomes the active (and only visible) recorder until it
     * is unwound or another recorder is installed.
     * @param lr the new logs recorder (non-null)
     * @param noInstallHandler [optional] used to if unable to install recorder
     * @return previous recorder if any (can be <i>null</i>)
     * @throws BuildException if incoming recorder already on iteration stack
     **/
    public static LogsRecorder installRecorder(LogsRecorder lr,
                                               ProblemHandler noInstallHandler)
    {
        if (lr==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("brul.logs.nul.recorder"));
        }
        return (LogsRecorder)FixtureOverlays.installIfNot
            (ANTLOG, lr, noInstallHandler, WHOAMI_);
    }



    /**
     * Removes the most recently installed LogsRecorder instance for the
     * current thread. The previous recorder is reactivated, or if this was
     * the only recorder, the current thread's logs recorder becomes
     * undefined (should use EmptyLogs placeholder if necessary).
     * @param noUninstallHandler [optional] used to if unable to uninstall
     **/
    public static void unwindRecorder(ProblemHandler noUninstallHandler)
    {
        FixtureOverlays.uninstallIfIs(ANTLOG,noUninstallHandler,WHOAMI_);
    }



    /**
     * Returns the current thread's frontmost stdio LogsRecorder
     * instance if one exists. Returns <i>null</i> if no recorder installed
     * for executing thread.
     * @since JWare/AntX 0.3
     **/
    public static LogsRecorder getStdIORecorder()
    {
        FixtureOverlays context = FixtureOverlays.getContextInstance();
        return (LogsRecorder)context.nearest(SYSTEM);
    }



    /**
     * Helper to {@linkplain #getStdIORecorder getStdIORecorder} that
     * returns the LogsRecorder {@linkplain EmptyLogs placeholder} if
     * the current thread has no installed stdio recorders. Never
     * returns <i>null</i>.
     * @since JWare/AntX 0.3
     **/
    public static LogsRecorder getStdIORecorderNoNull()
    {
        LogsRecorder r = getStdIORecorder();
        return (r!=null) ? r : EmptyLogs.INSTANCE;
    }



    /**
     * Installs a new stdio streams LogsRecorder instance for the current
     * thread. This recorder becomes the active (and only visible) stdio
     * recorder until it is unwound or another recorder is installed.
     * @param lr the new stdio output recorder (non-null)
     * @param noInstallHandler [optional] used to if unable to install recorder
     * @return previous recorder if any (can be <i>null</i>)
     * @throws BuildException if incoming recorder already on iteration stack
     * @since JWare/AntX 0.3
     **/
    public static LogsRecorder installStdIORecorder(LogsRecorder lr,
                                              ProblemHandler noInstallHandler)
    {
        if (lr==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("brul.logs.nul.recorder"));
        }
        return (LogsRecorder)FixtureOverlays.installIfNot
            (SYSTEM, lr, noInstallHandler, WHOAMI_);
    }



    /**
     * Removes the most recently installed stdio stream LogsRecorder instance
     * for the current thread. The previous recorder is reactivated, or if this
     * was the only recorder, the current thread's logs recorder becomes
     * undefined (should use EmptyLogs placeholder if necessary).
     * @param noUninstallHandler [optional] used to if unable to uninstall
     * @since JWare/AntX 0.3
     **/
    public static void unwindStdIORecorder(ProblemHandler noUninstallHandler)
    {
        FixtureOverlays.uninstallIfIs(SYSTEM,noUninstallHandler,WHOAMI_);
    }



    /** Disallow; only public static utility methods. **/
    private CapturedLogs()
    { }



    /**
     * Installs test-harness cleanup method to clear up the various output
     * recorder components. Your application should never use this helper
     * directly; it is provided so that test harnesses can reset the environment
     * to a known state.
     * @since JWare/AntX 0.4
     **/
    static {
        AntXFixture.setKillMethod
            (SYSTEM,
             new String[] {"stdio","stdout","stderr"},
             new KillMethodSkeleton(SYSTEM, WHOAMI_));

        AntXFixture.setKillMethod
            (ANTLOG,
             new String[] {"antlogs"},
             new KillMethodSkeleton(ANTLOG, WHOAMI_));


        //Both
        AntXFixture.setKillMethod
            (FixtureIds.CAPTURED_LOG_OUTPUTS,
             new String[] {"iocapture", "outputrecorders"},
             new KillMethod() {
                     public boolean kill(ProblemHandler from) {
                         FixtureOverlays iteration = FixtureOverlays.getContextInstance();
                         synchronized(iteration) {
                             iteration.clear(CapturedLogs.ANTLOG);
                             iteration.clear(CapturedLogs.SYSTEM);
                         }
                         return true;
                     }
                     public boolean kill(String target, ProblemHandler from) {
                         if (Strings.ALL.equals(target)) {
                             return kill(from);
                         }
                         return true;
                     }
                 }
             );
    }
}

/* end-of-CapturedLogs.java */
