/**
 * $Id: LogEnabled.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.apis;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectComponent;

/**
 * Any application object that supports the two standard Ant log APIs.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   n/a
 * @.group    api,infra
 * @.pattern  GoF.Strategy
 * @see       Requester
 * @see       ProblemHandler
 **/

public interface LogEnabled
{
    /**
     * Should do whatever service user does to record a
     * diagnostic trace message.
     * @param message the message (non-null)
     **/
    void log(String message);



    /**
     * Should do whatever service user does to record either a
     * problem or a diagnostic trace message. (The type of
     * message is indicated by the accompanying noise level.)
     * @param message the message (non-null)
     * @param msgLevel the <em>Ant</em> noise level
     **/
    void log(String message, int msgLevel);



    /**
     * Default {@linkplain LogEnabled} adapter for the System consoles.
     * @since    JWare/AntX 0.5
     * @version  0.5
     **/
    public static final LogEnabled SYSTEM= new LogEnabled() {
        public void log(String message) {
            System.out.println(message);
        }
        public void log(String message, int msgLevel) {
            if (msgLevel>=Project.MSG_INFO) {
                System.err.println(message);
            } else {
                System.out.println(message);
            }
        }
    };



    /**
     * Default {@linkplain LogEnabled} adapter for a generic Ant
     * project component. Use this adapter to pass non AntX-based items
     * into AntX APIs.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static class ForComponent implements LogEnabled
    {
        final ProjectComponent m_PC;

        /**
         * Initializes a new adapter for given project component.
         * @param pc the component being wrapped (non-null)
         */
        public ForComponent(ProjectComponent pc) {
            if (pc==null) {
                throw new IllegalArgumentException();
            }
            this.m_PC = pc;
        }

        public void log(String message) {
            m_PC.log(message);
        }

        public void log(String message, int msgLevel) {
            m_PC.log(message,msgLevel);
        }
        /** Returns the underlying component for this adapter. **/
        public final ProjectComponent getImpl() {
            return m_PC;
        }
    }




    /**
     * Default {@linkplain LogEnabled} adapter for a utility class. If
     * at call time the utility's project is <i>null</i> this adapter will
     * delegate to one of the two {@linkplain #SYSTEM system} consoles.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static class ForUtility implements LogEnabled
    {
        final ProjectDependent m_utility;

        /**
         * Initializes a new adapter for given project component.
         * @param utility the utility object being wrapped (non-null)
         */
        public ForUtility(ProjectDependent utility) {
            if (utility==null) {
                throw new IllegalArgumentException();
            }
            this.m_utility = utility;
        }

        public void log(String message) {
            if (m_utility.getProject()!=null) {
                m_utility.getProject().log(message);
            } else {
                SYSTEM.log(message);
            }
        }

        public void log(String message, int msgLevel) {
            if (m_utility.getProject()!=null) {
                m_utility.getProject().log(message,msgLevel);
            } else {
                SYSTEM.log(message,msgLevel);
            }
        }
        /** Returns the underlying utility for this adapter. **/
        public final ProjectDependent getImpl() {
            return m_utility;
        }
    }



    /**
     * Default {@linkplain LogEnabled} adapter for a generic Ant
     * project. Use this adapter to pass request-based project references
     * into AntX APIs. To facilitate utilities that allow <i>null</i>
     * projects, this adapter will use one of the two
     * {@linkplain #SYSTEM system} consoles if incoming project is <i>null</i>.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
    public static class ForProject implements LogEnabled
    {
        final Project m_project;

        /**
         * Initializes a new adapter for given project.
         * @param project the project being wrapped
         */
        public ForProject(Project project) {
            this.m_project = project;
        }

        public void log(String message) {
            if (m_project!=null) {
                m_project.log(message);
            } else {
                SYSTEM.log(message);
            }
        }

        public void log(String message, int msgLevel) {
            if (m_project!=null) {
                m_project.log(message,msgLevel);
            } else {
                SYSTEM.log(message,msgLevel);
            }
        }
        /** Returns the underlying project for this adapter. **/
        public final Project getImpl() {
            return m_project;
        }
    }



    /**
     * Default {@linkplain LogEnabled} adapter for a compound requester.
     * Use this adapter when creating {@linkplain Requester requesters}
     * comprised of multiple sources.
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.group   impl,helper
     **/
//    public static class ForComposite implements LogEnabled
//    {
//        private final LogEnabled m_singularity;
//
//        /**
//         * Initializes a new adapter for given source.
//         * @param original the original source of request (non-null)
//         */
//        public ForComposite(LogEnabled original) {
//            if (original==null) {
//                throw new IllegalArgumentException();
//            }
//            this.m_singularity = original;
//        }
//
//        public void log(String message) {
//            m_singularity.log(message);
//        }
//
//        public void log(String message, int msgLevel) {
//            m_singularity.log(message,msgLevel);
//        }
//        /** Returns the underlying source for this adapter. **/
//        public final LogEnabled getLogImpl() {
//            return m_singularity;
//        }
//    }
}

/* end-of-LogEnabled.java */
