/**
 * $Id: NoiseLevel.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.BuildEvent;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * Enumeration of (log) noise levels. Exposed as standalone class since reused by
 * multiple AntX tasks and conditions.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class NoiseLevel extends EnumSkeleton
{
    /** Index of {@linkplain #ERROR ERROR}. Implementation: must be indice 0.**/
    public static final int ERROR_INDEX= 0;
    /** Index of {@linkplain #WARNING WARNING}. **/
    public static final int WARNING_INDEX= ERROR_INDEX+1;
    /** Index of {@linkplain #INFO INFO}. **/
    public static final int INFO_INDEX= WARNING_INDEX+1;
    /** Index of {@linkplain #VERBOSE VERBOSE}. **/
    public static final int VERBOSE_INDEX= INFO_INDEX+1;
    /** Index of {@linkplain #DEBUG DEBUG}. **/
    public static final int DEBUG_INDEX= VERBOSE_INDEX+1;
    /** Index of {@linkplain #FATAL FATAL}. **/
    public static final int FATAL_INDEX= DEBUG_INDEX+1;

    /** Singleton "fatal" noise level. **/
    public static final NoiseLevel FATAL = new NoiseLevel("fatal",FATAL_INDEX);
    /** Singleton "error" noise level. **/
    public static final NoiseLevel ERROR = new NoiseLevel("error",ERROR_INDEX);
    /** Singleton "warning" noise level. **/
    public static final NoiseLevel WARNING = new NoiseLevel("warning",WARNING_INDEX);
    /** Singleton "info" noise level. **/
    public static final NoiseLevel INFO = new NoiseLevel("info",INFO_INDEX);
    /** Singleton "verbose" noise level. **/
    public static final NoiseLevel VERBOSE = new NoiseLevel("verbose",VERBOSE_INDEX);
    /** Singleton "debug" noise level. **/
    public static final NoiseLevel DEBUG = new NoiseLevel("debug",DEBUG_INDEX);



    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public NoiseLevel()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private NoiseLevel(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible noise level values as an ordered
     * string array. Note: should be same ordering as EchoLevel.
     **/
    public String[] getValues()
    {
        return new String[] {"error","warning","info","verbose","debug","fatal"};
    };



    /**
     * Helper that converts a scalar to a known noise level.
     * Returns <i>null</i> if value does not match any of expected
     * noise levels index.
     * @param i the index to be matched
     **/
    public static NoiseLevel from(int i)
    {
        switch(i) {
            case ERROR_INDEX:
                return ERROR;
            case WARNING_INDEX:
                return WARNING;
            case INFO_INDEX:
                return INFO;
            case VERBOSE_INDEX:
                return VERBOSE;
            case DEBUG_INDEX:
                return DEBUG;
            case FATAL_INDEX:
                return FATAL;
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known noise level
     * index.
     * @param i the index to be matched
     * @param dfltNL the default NoiseLevel if necessary
     **/
    public static NoiseLevel from(int i, NoiseLevel dfltNL)
    {
        NoiseLevel nl= from(i);
        return (nl==null) ? dfltNL : nl;
    }


    /**
     * Helper that converts a string to a known noise level
     * singleton. Returns <i>null</i> if string unrecognized.
     * String can be either noise level's symbolic name or
     * its index.
     **/
    public static NoiseLevel from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if ("default".equals(s))     { return getAntXDefault(); }
                if (ERROR.value.equals(s))   { return ERROR; }
                if (WARNING.value.equals(s)) { return WARNING; }
                if (INFO.value.equals(s))    { return INFO; }
                if (VERBOSE.value.equals(s)) { return VERBOSE; }
                if (DEBUG.value.equals(s))   { return DEBUG; }
                if (FATAL.value.equals(s))   { return FATAL; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * noise level name.
     * @param s the symbolic name to be matched
     * @param dfltNL the default NoiseLevel if necessary
     **/
    public static NoiseLevel from(String s, NoiseLevel dfltNL)
    {
        NoiseLevel nl= from(s);
        return (nl==null) ? dfltNL : nl;
    }



    /**
     * Helper that converts a standard Ant message priority to a 
     * known noise level. Returns <i>null</i> if priority does not 
     * match any of expected project message levels.
     * @param msgLevel the standard Ant message level
     * @since JWare/AntX 0.5
     **/
    public static NoiseLevel fromNativeIndex(int msgLevel)
    {
        switch(msgLevel) {
            case Project.MSG_ERR:
                return ERROR;
            case Project.MSG_WARN:
                return WARNING;
            case Project.MSG_INFO:
                return INFO;
            case Project.MSG_VERBOSE:
                return VERBOSE;
            case Project.MSG_DEBUG:
                return DEBUG;
        }
        return null;
    }


    /**
     * Helper that converts a BuildEvent's priority to a known noise
     * level. Returns <i>null</i> if priority does not match any of
     * expected noise levels.
     * @param e the event (non-null)
     **/
    public static NoiseLevel from(BuildEvent e)
    {
        return fromNativeIndex(e.getPriority());
    }


    /**
     * Returns a "pinned" level that's within the standard Ant
     * &lt;echo&gt; and Project MSG_* level set.
     **/
    public final int getNativeIndex()
    {
        int i= getIndex();
        if (i==FATAL_INDEX) {
            return ERROR_INDEX;
        }
        return i;
    }


    /**
     * Returns the <i>AntX</i> default noise level (INFO).
     * @see #getDefault
     **/
    public static final NoiseLevel getAntXDefault()
    {
        return NoiseLevel.INFO;
    }



    /**
     * Helper that returns some context's default noise level.
     * Looks in the project to see if a script-specified
     * {@linkplain AntX#DEFAULT_NOISELEVEL_PROP default} noise
     * level has been defined. If none specified in project (which
     * also includes System properties) the returns the AntX
     * {@linkplain #getAntXDefault default}. Never returns
     * <i>null</i>.
     * @param P [optional] caller's project
     * @see #getAntXDefault
     **/
    public static final NoiseLevel getDefault(final Project P)
    {
        String slvl=null;
        if (P!=null) {
            slvl = P.getProperty(AntX.DEFAULT_NOISELEVEL_PROP);
        }
        if (slvl==null) {
            slvl = System.getProperty(AntX.DEFAULT_NOISELEVEL_PROP);
        }
        NoiseLevel nl = getAntXDefault();
        if (slvl!=null) {
            nl= NoiseLevel.from(slvl,nl);
        }
        return nl;
    }



    /**
     * Determine if the given noise level is as bad (in effect) as the 
     * named threshold.
     * @param unk noise level to be tested (non-null)
     * @param threshold threshold level (non-null)
     * @return <i>true</i> if is as bad (in effect) as threshold
     * @since JWare/AntX 0.5
     **/
    public static final boolean isAsBadAs(NoiseLevel unk, NoiseLevel threshold)
    {
        return unk.getNativeIndex() <= threshold.getNativeIndex();
    }



    /**
     * Determine if the given Ant log level is as bad (in effect) as the 
     * named noise threshold.
     * @param loglevel Ant log level to be tested (non-null)
     * @param threshold threshold level (non-null)
     * @return <i>true</i> if is as bad (in effect) as threshold
     * @since JWare/AntX 0.5
     **/
    public static final boolean isAsBadAs(int loglevel, NoiseLevel threshold)
    {
        return loglevel <= threshold.getNativeIndex();
    }



    /**
     * Convenient (and visually clearer) instance variant of 
     * {@linkplain #isAsBadAs(NoiseLevel, NoiseLevel)
     * isAsBadAs(NoiseLevel,NoiseLevel)}.
     * @param threshold threshold level (non-null)
     * @return <i>true</i> if is as bad (in effect) as threshold
     * @since JWare/AntX 0.5
     **/
    public final boolean isAsBadAs(NoiseLevel threshold)
    {
        return isAsBadAs(this,threshold);
    }
}

/* end-of-NoiseLevel.java */
