/*
 * Decompiled with CFR 0.152.
 */
package org.pageseeder.psml.md;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.List;
import java.util.Objects;
import org.eclipse.jdt.annotation.Nullable;
import org.pageseeder.psml.md.MarkdownOutputOptions;
import org.pageseeder.psml.md.MarkdownTable;
import org.pageseeder.psml.model.PSMLElement;
import org.pageseeder.psml.model.PSMLNode;
import org.pageseeder.psml.model.PSMLText;
import org.pageseeder.psml.util.DiagnosticCollector;
import org.pageseeder.psml.util.NilDiagnosticCollector;
import org.pageseeder.psml.util.Subscripts;
import org.pageseeder.psml.util.Superscripts;

public class MarkdownSerializer {
    MarkdownOutputOptions options = MarkdownOutputOptions.defaultOptions();

    public void setOptions(MarkdownOutputOptions options) {
        this.options = options;
    }

    public MarkdownOutputOptions getOptions() {
        return this.options;
    }

    public void serialize(PSMLElement element, Appendable out) throws IOException {
        Instance instance = new Instance(this.options, new NilDiagnosticCollector());
        instance.serialize(element, out);
    }

    public void serialize(PSMLElement element, Appendable out, DiagnosticCollector collector) throws IOException {
        Instance instance = new Instance(this.options, collector);
        instance.serialize(element, out);
    }

    static String normalizeText(String text) {
        return text.replaceAll("\\s+", " ");
    }

    private static class Instance {
        private final MarkdownOutputOptions options;
        private final State state = new State();
        private final DiagnosticCollector collector;

        Instance(MarkdownOutputOptions options, DiagnosticCollector collector) {
            this.options = options;
            this.collector = Objects.requireNonNull(collector);
        }

        void serialize(PSMLElement element, Appendable out) throws IOException {
            PSMLElement.Name name = element.getElement();
            this.state.push(name);
            switch (name) {
                case BLOCK: {
                    this.serializeBlock(element, out);
                    break;
                }
                case BLOCKXREF: {
                    this.serializeBlockXref(element, out);
                    break;
                }
                case BOLD: {
                    this.serializeBold(element, out);
                    break;
                }
                case BR: {
                    this.serializeBreak(out);
                    break;
                }
                case DOCUMENT: {
                    this.serializeDocument(element, out);
                    break;
                }
                case DOCUMENTINFO: {
                    this.serializeDocumentInfo(element, out);
                    break;
                }
                case FRAGMENT: {
                    this.serializeFragment(element, out);
                    break;
                }
                case HEADING: {
                    this.serializeHeading(element, out);
                    break;
                }
                case IMAGE: {
                    this.serializeImage(element, out);
                    break;
                }
                case ITALIC: {
                    this.serializeItalic(element, out);
                    break;
                }
                case ITEM: {
                    break;
                }
                case LINK: {
                    this.serializeLink(element, out);
                    break;
                }
                case LIST: {
                    this.serializeList(element, out);
                    break;
                }
                case MEDIA_FRAGMENT: {
                    this.serializeMediaFragment(element, out);
                    break;
                }
                case MONOSPACE: {
                    this.serializeMonospace(element, out);
                    break;
                }
                case NLIST: {
                    this.serializeNlist(element, out);
                    break;
                }
                case PARA: {
                    this.serializePara(element, out);
                    break;
                }
                case PREFORMAT: {
                    this.serializePreformat(element, out);
                    break;
                }
                case PLACEHOLDER: {
                    this.serializePlaceholder(element, out);
                    break;
                }
                case PROPERTY: {
                    this.serializeProperty(element, out);
                    break;
                }
                case PROPERTIES_FRAGMENT: {
                    this.serializePropertiesFragment(element, out);
                    break;
                }
                case CAPTION: 
                case CELL: 
                case COL: 
                case ROW: {
                    break;
                }
                case TABLE: {
                    this.serializeTable(element, out);
                    break;
                }
                case METADATA: 
                case PROPERTIES: 
                case SECTION: 
                case URI: 
                case XREF_FRAGMENT: {
                    this.processChildElements(element, out);
                    break;
                }
                case SUB: {
                    this.serializeSub(element, out);
                    break;
                }
                case SUP: {
                    this.serializeSup(element, out);
                    break;
                }
                case TITLE: {
                    this.serializeTitle(element, out);
                    break;
                }
                case UNDERLINE: {
                    this.serializeUnderline(element, out);
                    break;
                }
                case TOC: 
                case AUTHOR: 
                case FRAGMENTINFO: 
                case COMPARE: 
                case COMPARETO: 
                case VERSIONS: 
                case REVERSEXREFS: {
                    break;
                }
                case XREF: {
                    this.serializeXref(element, out);
                    break;
                }
                default: {
                    this.processChildren(element, out);
                }
            }
            this.state.pop();
        }

        private void serializeBlock(PSMLElement block, Appendable out) throws IOException {
            switch (this.options.block()) {
                case QUOTED: {
                    this.serializeBlockAsQuoted(block, out);
                    break;
                }
                case FENCED: {
                    this.serializeBlockAsFenced(block, out);
                    break;
                }
                default: {
                    this.serializeBlockAsLabelText(block, out);
                }
            }
        }

        private void serializeBlockAsQuoted(PSMLElement block, Appendable out) throws IOException {
            String label = block.getAttribute("label");
            out.append("> ");
            if (label != null) {
                out.append("**").append(block.getAttribute("label")).append("**: ");
            }
            StringBuilder buffer = new StringBuilder();
            this.processChildren(block, buffer);
            String text = String.join((CharSequence)"\n> ", buffer.toString().split("\n"));
            out.append(text).append('\n');
        }

        private void serializeBlockAsFenced(PSMLElement block, Appendable out) throws IOException {
            String label = block.getAttributeOrElse("label", "");
            out.append("~~~").append(label);
            this.processChildren(block, out);
            out.append("~~~\n");
        }

        private void serializeBlockAsLabelText(PSMLElement block, Appendable out) throws IOException {
            String label = block.getAttribute("label");
            if (label != null) {
                out.append("**").append(block.getAttribute("label")).append("**: ");
            }
            this.processChildren(block, out);
            out.append('\n');
        }

        private void serializeBlockXref(PSMLElement blockXref, Appendable out) throws IOException {
            List<PSMLElement> children = blockXref.getChildElements(PSMLElement.Name.DOCUMENT, PSMLElement.Name.FRAGMENT, PSMLElement.Name.XREF_FRAGMENT, PSMLElement.Name.MEDIA_FRAGMENT, PSMLElement.Name.XREF_FRAGMENT);
            if (!children.isEmpty()) {
                PSMLElement child = children.get(0);
                if (child.getElement() == PSMLElement.Name.DOCUMENT) {
                    List<PSMLElement> sections = child.getChildElements(PSMLElement.Name.SECTION);
                    for (PSMLElement section : sections) {
                        List<PSMLElement> fragments = section.getChildElements();
                        for (PSMLElement fragment : fragments) {
                            this.serialize(fragment, out);
                        }
                    }
                } else {
                    this.serialize(child, out);
                }
            } else {
                this.serializeXref(blockXref, out);
            }
        }

        private void serializeBold(PSMLElement bold, Appendable out) throws IOException {
            out.append("__");
            this.processChildren(bold, out);
            out.append("__");
        }

        private void serializeBreak(Appendable out) throws IOException {
            if (this.state.isDescendantOf(PSMLElement.Name.TABLE)) {
                out.append("<br>");
            } else {
                out.append("\n");
            }
        }

        private void serializeDocument(PSMLElement document, Appendable out) throws IOException {
            if (this.options.metadata()) {
                String status = document.getAttribute("status");
                PSMLElement documentInfo = document.getFirstChildElement(PSMLElement.Name.DOCUMENTINFO);
                PSMLElement metadata = document.getFirstChildElement(PSMLElement.Name.METADATA);
                if (documentInfo != null || metadata != null) {
                    out.append("---\n");
                    if (documentInfo != null) {
                        this.serialize(documentInfo, out);
                    }
                    if (status != null) {
                        out.append("Status: ").append(status).append('\n');
                    }
                    if (metadata != null) {
                        this.serialize(metadata, out);
                    }
                    out.append("---\n");
                }
            }
            boolean firstSection = true;
            for (PSMLElement section : document.getChildElements(PSMLElement.Name.SECTION)) {
                if (firstSection) {
                    firstSection = false;
                } else {
                    out.append("\n---\n");
                }
                this.processChildren(section, out);
            }
        }

        private void serializeDocumentInfo(PSMLElement documentInfo, Appendable out) throws IOException {
            PSMLElement uri = documentInfo.getFirstChildElement(PSMLElement.Name.URI);
            if (uri != null) {
                String title = Instance.textOf(uri.getFirstChildElement(PSMLElement.Name.DISPLAYTITLE));
                String description = Instance.textOf(uri.getFirstChildElement(PSMLElement.Name.DESCRIPTION));
                String labels = Instance.textOf(uri.getFirstChildElement(PSMLElement.Name.LABELS));
                this.state.host = uri.getAttributeOrElse("host", "");
                this.state.path = uri.getAttributeOrElse("path", "");
                if (!title.isEmpty()) {
                    out.append("Title: ").append(title).append('\n');
                }
                if (!description.isEmpty()) {
                    out.append("Description: ").append(description).append('\n');
                }
                if (!labels.isEmpty()) {
                    boolean firstLabel = true;
                    out.append("Labels: [");
                    for (String label : labels.split(",")) {
                        if (!firstLabel) {
                            out.append(", ");
                        }
                        out.append(label);
                        firstLabel = false;
                    }
                    out.append("]\n");
                }
            }
        }

        private void serializeFragment(PSMLElement fragment, Appendable out) throws IOException {
            this.processChildElements(fragment, out);
        }

        private void serializeHeading(PSMLElement heading, Appendable out) throws IOException {
            int level = heading.getAttributeOrElse("level", 1);
            String prefix = heading.getAttribute("prefix");
            out.append("\n");
            for (int i = 0; i < Math.min(level, 6); ++i) {
                out.append('#');
            }
            out.append(' ');
            if (prefix != null) {
                out.append(prefix).append(' ');
            } else if ("true".equals(heading.getAttribute("numbered"))) {
                out.append(this.state.nextHeadingPrefix(level - 1));
            }
            this.processChildren(heading, out);
            out.append("\n");
        }

        private void serializeImage(PSMLElement image, Appendable out) throws IOException {
            String src = image.getAttributeOrElse("src", "");
            String alt = image.getAttribute("alt");
            if (this.options.captions()) {
                out.append("**").append(this.state.nextImage()).append("**");
                if (alt != null) {
                    out.append(": ").append(alt);
                }
                out.append("\n");
            }
            if (alt == null) {
                alt = !src.isEmpty() ? src.substring(src.lastIndexOf(47) + 1) : "";
            }
            switch (this.options.image()) {
                case LOCAL: {
                    String href = this.state.toRelativeLocalPath(src, this.collector);
                    out.append("![").append(alt).append("](").append(href).append(")");
                    break;
                }
                case EXTERNAL: {
                    if (this.state.host.isEmpty() || this.state.path.isEmpty()) {
                        this.collector.warn("Cannot generate external image URL without host and path information");
                    }
                    String url = this.state.toExternalUrl(src);
                    out.append("![").append(alt).append("](").append(url).append(")");
                    break;
                }
                case DATA_URI: {
                    this.collector.warn("Data URI images are not currently supported");
                    break;
                }
                case IMG_TAG: {
                    String width = image.getAttribute("width");
                    String height = image.getAttribute("height");
                    out.append("<img src=\"").append(src).append('\"');
                    out.append(" alt=\"").append(alt).append('\"');
                    if (width != null && !width.isEmpty()) {
                        out.append(" width=\"").append(width).append('\"');
                    }
                    if (height != null && !height.isEmpty()) {
                        out.append(" height=\"").append(height).append('\"');
                    }
                    out.append(" />");
                    break;
                }
            }
        }

        private void serializeItalic(PSMLElement italic, Appendable out) throws IOException {
            out.append("*");
            this.processChildren(italic, out);
            out.append("*");
        }

        private void serializeLink(PSMLElement link, Appendable out) throws IOException {
            String text = MarkdownSerializer.normalizeText(link.getText());
            String url = link.getAttribute("href");
            out.append("[").append(text).append("](").append(url).append(")");
        }

        private void serializeList(PSMLElement list, Appendable out) throws IOException {
            List<PSMLElement> items = list.getChildElements(PSMLElement.Name.ITEM);
            int level = this.state.listLevel();
            out.append('\n');
            for (PSMLElement item : items) {
                this.state.push(PSMLElement.Name.ITEM);
                for (int i = 0; i < level; ++i) {
                    out.append("    ");
                }
                out.append("* ");
                this.processChildren(item, out);
                out.append('\n');
                this.state.pop();
            }
        }

        private void serializeMediaFragment(PSMLElement fragment, Appendable out) throws IOException {
            String text;
            out.append("\n```");
            String mediatype = fragment.getAttribute("mediatype");
            if (mediatype != null) {
                out.append(Instance.toLang(mediatype));
            }
            if (!(text = fragment.getText()).startsWith("\n")) {
                out.append("\n");
            }
            out.append(text.replaceAll("\\s+$", "")).append("\n```\n");
        }

        private void serializeMonospace(PSMLElement monospace, Appendable out) throws IOException {
            out.append("`");
            this.processChildren(monospace, out);
            out.append("`");
        }

        private void serializeNlist(PSMLElement nlist, Appendable out) throws IOException {
            List<PSMLElement> items = nlist.getChildElements(PSMLElement.Name.ITEM);
            int level = this.state.listLevel();
            out.append('\n');
            int start = nlist.getAttributeOrElse("start", 1);
            for (PSMLElement item : items) {
                this.state.push(PSMLElement.Name.ITEM);
                for (int i = 0; i < level; ++i) {
                    out.append("    ");
                }
                out.append(Integer.toString(start++)).append(". ");
                this.processChildren(item, out);
                out.append('\n');
                this.state.pop();
            }
        }

        private void serializePara(PSMLElement para, Appendable out) throws IOException {
            String prefix = para.getAttribute("prefix");
            boolean inTableOrItem = this.state.isDescendantOf(PSMLElement.Name.TABLE) || this.state.isDescendantOf(PSMLElement.Name.ITEM);
            int indent = para.getAttributeOrElse("indent", 0);
            if (!inTableOrItem) {
                out.append("\n");
            }
            if (indent > 0) {
                for (int i = 0; i < Math.min(indent, 6); ++i) {
                    out.append("\u00a0\u00a0\u00a0\u00a0");
                }
            }
            if (prefix != null) {
                out.append(prefix).append(" ");
            } else if ("true".equals(para.getAttribute("numbered"))) {
                out.append(this.state.nextParaPrefix(indent));
            }
            this.processChildren(para, out);
            if (!inTableOrItem) {
                out.append("\n");
            }
        }

        private void serializePreformat(PSMLElement element, Appendable out) throws IOException {
            String text;
            out.append("\n```");
            String role = element.getAttribute("role");
            if (role != null && (role.startsWith("language-") || role.startsWith("lang-"))) {
                String language = role.substring(role.indexOf(45) + 1);
                out.append(language);
            }
            if (!(text = element.getText()).startsWith("\n")) {
                out.append("\n");
            }
            out.append(text.replaceAll("\\s+$", "")).append("\n```\n");
        }

        private void serializePlaceholder(PSMLElement placeholder, Appendable out) throws IOException {
            String name = placeholder.getAttributeOrElse("name", "");
            String text = MarkdownSerializer.normalizeText(placeholder.getText());
            out.append("[[").append(text.isEmpty() ? name : text).append("]]");
        }

        private void serializeProperty(PSMLElement property, Appendable out) throws IOException {
            String name = property.getAttributeOrElse("name", "");
            String title = property.getAttributeOrElse("title", name);
            String value = this.toPropertyValue(property);
            if (this.state.isDescendantOf(PSMLElement.Name.METADATA) || this.options.properties() == MarkdownOutputOptions.PropertiesFormat.VALUE_PAIRS) {
                out.append(MarkdownSerializer.normalizeText(title)).append(": ").append(value).append("\n");
            } else {
                out.append("| ").append(MarkdownSerializer.normalizeText(title)).append(" | ").append(value).append(" |\n");
            }
        }

        private String toPropertyValue(PSMLElement property) throws IOException {
            String value = property.getAttribute("value");
            if (value != null) {
                return MarkdownSerializer.normalizeText(value.replace("\n", "<br>"));
            }
            PSMLElement markdown = property.getFirstChildElement(PSMLElement.Name.MARKDOWN);
            if (markdown != null) {
                return markdown.getText().replace("\n", "<br>");
            }
            StringBuilder out = new StringBuilder();
            List<PSMLElement> values = property.getChildElements(PSMLElement.Name.VALUE);
            if (!values.isEmpty()) {
                out.append("[");
                boolean first = true;
                for (PSMLElement v : values) {
                    if (first) {
                        first = false;
                    } else {
                        out.append(", ");
                    }
                    out.append(MarkdownSerializer.normalizeText(v.getText()));
                }
                out.append("]");
                return out.toString();
            }
            List<PSMLElement> xrefs = property.getChildElements(PSMLElement.Name.XREF);
            if (!xrefs.isEmpty()) {
                boolean first = true;
                for (PSMLElement xref : xrefs) {
                    if (first) {
                        first = false;
                    } else {
                        out.append(", ");
                    }
                    this.serialize(xref, out);
                }
                return out.toString();
            }
            this.processChildren(property, out);
            return MarkdownSerializer.normalizeText(out.toString().replace("\n", "<br>"));
        }

        private void serializePropertiesFragment(PSMLElement fragment, Appendable out) throws IOException {
            out.append('\n');
            if (this.options.properties() == MarkdownOutputOptions.PropertiesFormat.TABLE) {
                if (this.options.captions()) {
                    out.append("**").append(this.state.nextProperties()).append("**: ").append(fragment.getAttribute("id")).append('\n');
                }
                out.append("| Name | Value |\n");
                out.append("|---|---|\n");
            }
            for (PSMLElement property : fragment.getChildElements(PSMLElement.Name.PROPERTY)) {
                this.serializeProperty(property, out);
            }
        }

        private void serializeSup(PSMLElement element, Appendable out) throws IOException {
            if (element.isEmpty()) {
                return;
            }
            switch (this.options.superSub()) {
                case CARET_TILDE: {
                    out.append("^");
                    this.processChildren(element, out);
                    out.append("^");
                    break;
                }
                case HTML: {
                    out.append("<sup>");
                    this.processChildren(element, out);
                    out.append("</sup>");
                    break;
                }
                case UNICODE: {
                    if (!element.hasChildElements() && Superscripts.isReplaceable(element.getText())) {
                        out.append(Superscripts.toSuperscript(element.getText()));
                        break;
                    }
                    this.collector.warn("Superscript text contains character for which there is no Unicode equivalent");
                    this.processChildren(element, out);
                    break;
                }
                default: {
                    this.collector.warn("Superscripts are ignored in Markdown output format");
                    this.processChildren(element, out);
                }
            }
        }

        private void serializeSub(PSMLElement element, Appendable out) throws IOException {
            if (element.isEmpty()) {
                return;
            }
            switch (this.options.superSub()) {
                case CARET_TILDE: {
                    out.append("~");
                    this.processChildren(element, out);
                    out.append("~");
                    break;
                }
                case HTML: {
                    out.append("<sub>");
                    this.processChildren(element, out);
                    out.append("</sub>");
                    break;
                }
                case UNICODE: {
                    if (!element.hasChildElements() && Subscripts.isReplaceable(element.getText())) {
                        out.append(Subscripts.toSubscript(element.getText()));
                        break;
                    }
                    this.collector.warn("Subscript text contains character for which there is no Unicode equivalent");
                    this.processChildren(element, out);
                    break;
                }
                default: {
                    this.collector.warn("Subscripts are ignored in Markdown output format");
                    this.processChildren(element, out);
                }
            }
        }

        private void serializeTable(PSMLElement table, Appendable out) throws IOException {
            MarkdownTable tableOut = new MarkdownTable(table, this.state.nextTable(), this.collector);
            tableOut.format(out, this.options);
        }

        private void serializeTitle(PSMLElement title, Appendable out) throws IOException {
            out.append("\n");
            out.append("## ");
            this.processChildren(title, out);
            out.append("\n");
        }

        private void serializeUnderline(PSMLElement element, Appendable out) throws IOException {
            if (element.isEmpty()) {
                return;
            }
            if (this.options.underline() == MarkdownOutputOptions.UnderlineFormat.HTML) {
                out.append("<u>");
                this.processChildren(element, out);
                out.append("</u>");
            } else {
                this.collector.warn("Underlines are ignored in Markdown output format");
                this.processChildren(element, out);
            }
        }

        private void serializeXref(PSMLElement link, Appendable out) throws IOException {
            String text = MarkdownSerializer.normalizeText(link.getText());
            String url = link.getAttributeOrElse("href", "");
            switch (this.options.xref()) {
                case EXTERNAL_LINK: {
                    String externalHref = this.state.toExternalUrl(url);
                    out.append("[").append(text).append("](").append(externalHref).append(")");
                    break;
                }
                case LOCAL_LINK: {
                    String localHref = this.state.toRelativeLocalPath(url, this.collector);
                    out.append("[").append(text).append("](").append(localHref).append(")");
                    break;
                }
                case BOLD_TEXT: {
                    out.append("**").append(text).append("**");
                    break;
                }
                default: {
                    out.append(text);
                }
            }
        }

        private void processChildren(PSMLElement element, Appendable out) throws IOException {
            for (PSMLNode node : element.getNodes()) {
                if (node instanceof PSMLText) {
                    String text = MarkdownSerializer.normalizeText(node.getText());
                    if (this.state.isDescendantOf(PSMLElement.Name.ITALIC)) {
                        text = text.replace("*", "\\*");
                    }
                    if (this.state.isDescendantOf(PSMLElement.Name.BOLD)) {
                        text = text.replace("_", "\\_");
                    }
                    if (this.state.isDescendantOf(PSMLElement.Name.MONOSPACE)) {
                        text = text.replace("`", "\\`");
                    }
                    out.append(text);
                    continue;
                }
                this.serialize((PSMLElement)node, out);
            }
        }

        private void processChildElements(PSMLElement element, Appendable out) throws IOException {
            for (PSMLElement child : element.getChildElements()) {
                this.serialize(child, out);
            }
        }

        private static String toLang(String mediatype) {
            switch (mediatype) {
                case "text/html": {
                    return "html";
                }
                case "text/css": {
                    return "css";
                }
                case "text/javascript": {
                    return "js";
                }
                case "text/x-csrc": {
                    return "c";
                }
                case "text/x-c++src": {
                    return "cpp";
                }
                case "text/x-java": {
                    return "java";
                }
                case "text/x-csharp": {
                    return "cs";
                }
                case "text/x-groovy": {
                    return "groovy";
                }
                case "text/x-ruby": {
                    return "rb";
                }
                case "text/x-python": {
                    return "py";
                }
                case "text/x-php": {
                    return "php";
                }
                case "text/x-yaml": {
                    return "yaml";
                }
                case "text/x-xml": {
                    return "xml";
                }
                case "text/x-markdown": {
                    return "md";
                }
                case "text/x-latex": {
                    return "latex";
                }
                case "text/x-tex": 
                case "application/x-tex": {
                    return "tex";
                }
            }
            return "";
        }

        private static String textOf(@Nullable PSMLElement element) {
            return element == null ? "" : MarkdownSerializer.normalizeText(element.getText());
        }
    }

    private static class State {
        private final Deque<PSMLElement.Name> context = new ArrayDeque<PSMLElement.Name>();
        private int imageCounter = 0;
        private int tableCounter = 0;
        private int propertiesCounter = 0;
        private final int[] headingPrefix = new int[]{0, 0, 0, 0, 0, 0};
        private final int[] paraPrefix = new int[]{0, 0, 0, 0, 0};
        private String host = "";
        private String path = "";

        private State() {
        }

        public void push(PSMLElement.Name name) {
            this.context.push(name);
        }

        public PSMLElement.Name pop() {
            return this.context.pop();
        }

        String nextImage() {
            return "Image " + ++this.imageCounter;
        }

        String nextTable() {
            return "Table " + ++this.tableCounter;
        }

        String nextProperties() {
            return "Properties " + ++this.propertiesCounter;
        }

        String toRelativeLocalPath(String href, DiagnosticCollector collector) {
            if (href.startsWith("http://") || href.startsWith("https://")) {
                return href;
            }
            if (this.path.isEmpty()) {
                return href;
            }
            try {
                Path base = Paths.get(this.path, new String[0]).getParent();
                Path target = Paths.get(href, new String[0]);
                if (base == null) {
                    return href;
                }
                Path relative = base.relativize(target);
                return relative.toString().replace('\\', '/');
            }
            catch (IllegalArgumentException ex) {
                collector.warn("Unable to relativize path " + href + " to " + this.path);
                return href;
            }
        }

        String toExternalUrl(String href) {
            if (href.startsWith("http://") || href.startsWith("https://")) {
                return href;
            }
            if (this.host.isEmpty()) {
                return href;
            }
            String baseUrl = "https://" + this.host;
            if (href.startsWith("/")) {
                return baseUrl + href;
            }
            Path basePath = Paths.get(this.path, new String[0]).getParent();
            Path resolvedPath = basePath != null ? basePath.resolve(href).normalize() : Paths.get(href, new String[0]).normalize();
            Object externalPath = resolvedPath.toString().replace('\\', '/');
            if (!((String)externalPath).startsWith("/")) {
                externalPath = "/" + (String)externalPath;
            }
            return baseUrl + (String)externalPath;
        }

        String nextHeadingPrefix(int level) {
            if (level <= 0 || level >= this.headingPrefix.length) {
                return "";
            }
            this.headingPrefix[level - 1] = this.headingPrefix[level - 1] + 1;
            for (int i = level; i < this.headingPrefix.length; ++i) {
                this.headingPrefix[i] = 0;
            }
            StringBuilder prefix = new StringBuilder();
            prefix.append(this.headingPrefix[0]);
            if (level > 1) {
                prefix.append('.').append(this.headingPrefix[1]);
            }
            if (level > 2) {
                prefix.append('.').append(this.headingPrefix[2]);
            }
            if (level > 3) {
                prefix.append('.').append(this.headingPrefix[3]);
            }
            if (level > 4) {
                prefix.append('.').append(this.headingPrefix[4]);
            }
            if (level > 5) {
                prefix.append('.').append(this.headingPrefix[5]);
            }
            return prefix.append(" ").toString();
        }

        String nextParaPrefix(int indent) {
            if (indent < 0 || indent >= this.paraPrefix.length) {
                return "";
            }
            this.paraPrefix[indent] = this.paraPrefix[indent] + 1;
            for (int i = indent + 1; i < this.paraPrefix.length; ++i) {
                this.paraPrefix[i] = 0;
            }
            switch (indent) {
                case 0: {
                    return "(" + (char)(97 + (this.paraPrefix[0] - 1)) + ") ";
                }
                case 1: {
                    return "(" + (char)(97 + (this.paraPrefix[1] - 1)) + ") ";
                }
                case 2: {
                    return "(" + (char)(105 + (this.paraPrefix[2] - 1)) + ") ";
                }
                case 3: {
                    return "(" + (char)(65 + (this.paraPrefix[3] - 1)) + ") ";
                }
                case 4: {
                    return "(" + (char)(73 + (this.paraPrefix[4] - 1)) + ") ";
                }
            }
            return "";
        }

        boolean isDescendantOf(PSMLElement.Name name) {
            for (PSMLElement.Name n : this.context) {
                if (n != name) continue;
                return true;
            }
            return false;
        }

        int listLevel() {
            int level = -1;
            for (PSMLElement.Name n : this.context) {
                if (n != PSMLElement.Name.LIST && n != PSMLElement.Name.NLIST) continue;
                ++level;
            }
            return level;
        }
    }
}

