/**
 * $Id: PasswordValueURIHandlerTest.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  java.util.Properties;

import  org.apache.tools.ant.Project;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.valueuri.info.DefaultsValueURIHandler;
import  com.idaremedia.antx.valueuri.info.PasswordValueURIHandler;

/**
 * Testsuite for {@linkplain PasswordValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class PasswordValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public PasswordValueURIHandlerTest(String methodname)
    {
        super("PasswordValueURIHandler::",methodname);
    }


    /**
     * Returns full test suite for PasswordValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(PasswordValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    private static final String PASSWORDSMAP = "build.passwords";
    private static final String DEFAULTPASSWORD = "defaults.password";


    protected String xmlTestFile()
    {
        return "password.xml";
    }

    protected ValueURIHandler newOUT()
    {
        return new PasswordValueURIHandler();
    }

    protected void setUp() throws Exception
    {
        super.setUp();

        Project myproject = getProject();
        assertNil(myproject.getProperty(PASSWORDSMAP),"'build.passwords' property");
        assertNil(myproject.getProperty(DEFAULTPASSWORD),"'defaults.password' property");
        assertNil(myproject.getReference(PASSWORDSMAP),"'build.passwords' reference");

        Iteration.initValueURIHandler("default",DefaultsValueURIHandler.class);
        Iteration.initValueURIHandler("df",DefaultsValueURIHandler.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify a blank uri fragment is evaluated as null.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();

        String result = newOUT().valueFrom("","$password:",m_rqlink);
        assertNil(result,"$password:");
    }


    /**
     * Verify the capitalization of the key is unimportant.
     * @since JWare/AntX 0.5
     **/
    public void testFallbackOnDefaultsPassword_AntX05()
    {
        Project myproject = getProject();

        ValueURIHandler out = newOUT();
        String userid = SIDs.next("user");
        String result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertNil(result,"$password:"+userid);

        myproject.setProperty(DEFAULTPASSWORD,"superseekrit");
        result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertEqual(result,"superseekrit","$password:"+userid+"(defaults)");
        
        Properties passwords = new Properties();
        passwords.setProperty("user0","password0");
        myproject.addReference(PASSWORDSMAP,passwords);

        result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertEqual(result,"superseekrit","$password:"+userid+"(passwords+defaults)");
    }



    /**
     * Verify that if both a default password and a build.passwords
     * property set are defined, the build.passwords win (if a match
     * found).
     * @since JWare/AntX 0.5
     **/
    public void testBuildPasswordsBlocksDefaultsPassword_AntX05()
    {
        String userid = SIDs.next("user");
        String password= SIDs.next("pw");
        
        Project myproject = getProject();
        myproject.setProperty(DEFAULTPASSWORD,"superseekrit");

        Properties passwords = new Properties();
        passwords.setProperty("user0","password0");
        passwords.setProperty("user1","password1");
        passwords.setProperty(userid,password);
        myproject.addReference(PASSWORDSMAP,passwords);

        ValueURIHandler out = newOUT();
        String result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertEqual(result,password,"$password:"+userid);

        passwords.setProperty(userid,password+"-delta");
        result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertEqual(result,password+"-delta","$password:"+userid);
    }


    /**
     * Verify that an unrecognized data type stored under the
     * 'build.passwords' refid, is ignored.
     * @since JWare/AntX 0.5
     **/
    public void testIllegalBuildPasswordsTypesIgnored_AntX05()
    {
        Project myproject = getProject();
        myproject.addReference(PASSWORDSMAP, new Thingy());

        String userid = SIDs.next("user");
        ValueURIHandler out = newOUT();
        String result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertNil(result,"$password:"+userid+"(badtype passwords)");

        myproject.setProperty(DEFAULTPASSWORD,"lightweight");
        result = out.valueFrom(userid,"$password:"+userid,m_rqlink);
        assertEqual(result,"lightweight","$password:"+userid+"(badtype passwords+defaults)");
    }


    /**
     * Verify handler resolve property references in fragment
     * before looking up name.
     * @since JWare/AntX 0.5
     **/
    public void testPropertyRefsInterpreted_AntX05()
    {
        runTarget("testPropertyRefsInterpreted_AntX05");
    }


    /**
     * Verifies that a cleartext password file in:
     * $user&#46;home/&#46;ant/build&#46;passwords used as a 
     * fallback measure.
     **/
    public void testUserHomePasswordFileUsed_AntX05()
    {
        runTarget("testUserHomePasswordFileUsed_AntX05");
    }


    /**
     * Verifies that a password file pointed to by the standard 
     * AntX passwordfile defaults setting is used.
     * @since JWare/AntX 0.5
     */
    public void testDefaultsPasswordFileOptionUsed_AntX05()
    {
        runTarget("testDefaultsPasswordFileOptionUsed_AntX05");
    }


    /**
     * Verifies the order in which handler searches for 
     * password files.
     * @since JWare/AntX 0.5
     */
    public void testSearchOrdering_AntX05()
    {
        runTarget("testSearchOrdering_AntX05");
    }


    /**
     * Ensure we install a reference to a "thing" no one else 
     * could know about.
     */
    private static class Thingy {
        private Thingy() {}
    }
}

/* end-of-DefaultsValueURIHandlerTest.java */