/**
 * $Id: BasenameValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  java.io.File;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that is an inline variant of the standard Ant &lt;basename&gt;
 * task. The general format of the URI is: 
 * <span class="src">$basename:path[?suffix]</span>. The path will be resolved relative
 * to the uri's owning project if necessary and like the &lt;basename&gt; task and empty
 * path string is interpreted as the project's base directory.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;parameter name="label" value="${$property:other-mainlibs.dir|$basename:}"/&gt;
 * 
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="basename"
 *             value="com.idaremedia.antx.valueuri.BasenameValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

public final class BasenameValueURIHandler extends ValueURIHandlerSkeleton
{
    /**
     * Initializes a new truefalse value uri handler.
     **/
    public BasenameValueURIHandler()
    {
        super();
    }


    /**
     * Converts a Ant-isque boolean string into a normalized 
     * "<span class="src">true</span>" or "<span class="src">false</span>".
     * If a custom pair of true/false strings specified as query parameters,
     * one of these strings will be returned instead.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project P = clnt.getProject();
        String path = uriFragment;
        String suffix = null;

        int i = uriFragment.indexOf("?");
        if (i>=0) {
            path = uriFragment.substring(0,i++);
            if (i<uriFragment.length()) {
                suffix = Tk.resolveString(P,uriFragment.substring(i),true);
            }
        }
        path = Tk.resolveString(P,path,true);
        return baseFrom(path,suffix,clnt);
    }



    /**
     * Extracts the extensionless base name of given file.
     * @param inpath path as passed in uri fragment
     * @param suffix [optional] suffix to be stripped (if exists)
     * @param clnt call controls (non-null)
     * @return basename (never <i>null</i>)
     **/
    private String baseFrom(String inpath, String suffix, Requester clnt)
    {
        File file;
        if (inpath.length()==0) {
            file = clnt.getProject().getBaseDir();
            if (file==null) {
                return null;
            }
        } else {
            file = clnt.getProject().resolveFile(inpath);
        }
        /* --
         * This is a duplicate of the Ant basename task's core 
         * functionality to ensure we do not stray from that impl.
         */
        String filename = file.getName();
        if (!Tk.isWhitespace(suffix) && filename.endsWith(suffix)) {
            int pos = filename.length() - suffix.length();
            if (pos>0 && suffix.charAt(0)!= '.'
                && filename.charAt(pos-1)=='.') {
                pos--;
            }
            filename = filename.substring(0,pos);
        }
        return filename;
    }
}

/* end-of-BasenameValueURIHandler.java */