/**
 * $Id: HAbortingResult.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ut;

import  junit.framework.AssertionFailedError;
import  junit.framework.Test;
import  junit.framework.TestResult;

/**
 * Test Result collector that will automatically signal a suite to stop if the number
 * of errors and/or failures hit some predefined limit. For example, can be setup
 * to stop if any errors are detected.
 *
 * @since    JWare/core 0.4
 * @author   ssmc, &copy;1997-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded once initialized
 * @.group   impl,test
 **/

public class HAbortingResult extends TestResult
{
    /**
     * Create new aborting result that doesn't (no limits).
     **/
    public HAbortingResult()
    {
        super();
    }


    /**
     * Create new aborting result that will allow at most <code>maxFailures</code>
     * failures and <code>maxErrors</code> errors before signalling all tests should
     * be stopped (to caller).
     **/
    public HAbortingResult(int maxFailures, int maxErrors)
    {
        super();
        m_maxFailures= maxFailures;
        m_maxErrors= maxErrors;
    }



    /**
     * Tells this result the positions from which its should calculate
     * stop criteria. If this result is being used across multiple distinct
     * test suites, each suite should reset this result's start points
     * when it starts.
     * @param failure0th failure start point
     * @param error0th error start point
     * @.safety guarded
     * @since JWare/AntX 0.4
     **/
    public void setStartPoints(int failure0th, int error0th)
    {
        m_0thFailure = failure0th; //NB: need a stack if ever nestable!
        m_0thError = error0th;
    }



    /**
     * Convenient alternative to {@linkplain #setStartPoints(int,int)
     * setStartPoints(int,int)} that uses the current error and failure
     * counts as the start points.
     * @.safety guarded
     * @since JWare/AntX 0.4
     **/
    public final void newStartPoints()
    {
        setStartPoints(fFailures.size(),fErrors.size());
    }



    /**
     * Convenience to uninstall the previous call to 
     * {@linkplain #newStartPoints}.
     * @since JWare/AntX 0.4
     **/
    public final void popStartPoints()
    {
        setStartPoints(0,0); //NB: need a stack if ever nestable!
    }



    /**
     * Add error to this result; will signal stop to caller (suite|test) if
     * setup to stop on any error. See junit.framework.TestResult#shouldStop.
     **/
    public synchronized void addError(Test test, Throwable t)
    {
        super.addError(test,t);
        if (fErrors.size()-get0thError()>=m_maxErrors) {
            this.stop();
        }
    }


    /**
     * Add a failure to this result; will signal stop to caller (suite|test)
     * if a maximum failure-count has been specified.
     **/
    public synchronized void addFailure(Test test, AssertionFailedError t)
    {
        super.addFailure(test,t);
        if (fFailures.size()-get0thFailure()>=m_maxFailures) {
            this.stop();
        }
    }


    /**
     * Return the maximum number of failures recorded by this result before
     * it signals a stop. Will return <code>Integer.MAX_VALUE</code> if
     * no such limit.
     **/
    public int getMaximumFailures()
    {
        return m_maxFailures;
    }


    /**
     * Set the maximum number of failures recorded by this result before
     * it signals a stop.
     **/
    public void setMaximumFailures(int maxFailures)
    {
        m_maxFailures = maxFailures;
    }


    /**
     * Return the maximum number of errors recorded by this result before
     * it signals a stop. Will return <code>Integer.MAX_VALUE</code> if
     * no such limit.
     **/
    public int getMaximumErrors()
    {
        return m_maxErrors;
    }


    /**
     * Set the maximum number of errors recorded by this result before
     * it signals a stop.
     **/
    public void setMaximumErrors(int maxErrors)
    {
        m_maxErrors = maxErrors;
    }


    /**
     * Returns whether this result is configured to signal stop as soon
     * as any failure or error occurs.
     **/
    public final boolean isFailQuick()
    {
        return getMaximumErrors()==1 && getMaximumFailures()==1;
    }


    /**
     * Shortcut to ensure this result signal stop as-soon-as <em>any</em>
     * failure or error is recorded.
     **/
    public final void setFailQuick()
    {
        setMaximumErrors(1);
        setMaximumFailures(1);
    }



    private int get0thError()
    {
        return m_0thError;  //NB: need stack if ever suites are nestable!
    }
    
    private int get0thFailure()
    {
        return m_0thFailure;//NB: ditto!
    }

    /** max failures and max errors. **/
    protected int m_maxFailures= Integer.MAX_VALUE;
    protected int m_maxErrors= Integer.MAX_VALUE;
    protected int m_0thError= 0;
    protected int m_0thFailure= 0;
}

/* end-of-HAbortingResult.java */

