/**
 * $Id: SimpleManagerTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.util.Collections;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.InnerNameValuePair;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.FeedbackLevel;

/**
 * Common form of an AntX manager task. Basically a single "<span class="src">action</span>"
 * parameter is specified and action configuration is done primarily using simple nested
 * <span class="src">&lt;parameter&gt;</span> elements. Subclasses must supply the
 * various public "do" methods like <span class="src">doInstall</span>. In addition to the
 * inherited <span class="src">haltiferror</span> option, this common manager task lets
 * you control how much diagnostic feedback is generated through its
 * <span class="src">feedback</span> option.
 * <p/>
 * To facilitate script readability, a simple manager task can scrub an action's name to
 * remove human-friendly spacer characters that could never belong to an action method's
 * name. For example, the actions "<span class="src">install-default</span>" and
 * "<span class="src">throw error</span>" would be interpreted as
 * "<span class="src">installdefault</span>" and "<span class="src">throwerror</span>"
 * respectively. The matching methods would be named <span class="src">doInstallDefault</span>
 * and <span class="src">doThrowError</span>.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;managertask action="install"&gt; <i>//managertask is name of some subclass</i>
 *       &lt;parameter name="arg0" value="Hello"/&gt;
 *       &lt;parameter name="arg1" value="World"/&gt;
 *    &lt;managertask&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 **/

public abstract class SimpleManagerTask extends ManagerTask
{
    /**
     * Initializes a new simple manager task.
     * @param iam CV-label (non-null)
     **/
    protected SimpleManagerTask(String iam)
    {
        super(iam);
    }



    /**
     * Initializes a new manager task with a default action.
     * @param iam CV-label (non-null)
     * @param defaultAction default action to execute (non-null)
     * @.impl This sets the action field <em>directly</em>; the
     *    <span class="src">setAction</span> method is not called.
     **/
    protected SimpleManagerTask(String iam, String defaultAction)
    {
        super(iam);
        m_actionName = defaultAction;
    }



    /**
     * Sets the control action for this manager task. An inlined
     * URI parameter can be specified after a single "&#63;" marker
     * at end of action's name.
     * @param action requested action's name (non-null)
     **/
    public void setAction(String action)
    {
        require_(action!=null,"setAction- nonzro action");
        int i = action.indexOf("?");
        if (i>0) {
            String choice = scrub(action.substring(0,i++));
            m_actionName = Tk.lowercaseFrom(choice);
            if (i<action.length()) {
                m_uriParams = action.substring(i);
            }
        } else {
            m_actionName = Tk.lowercaseFrom(scrub(action));
        }
    }



    /**
     * Returns the current setting for this task's action name.
     * Will return <i>null</i> if not yet set.
     **/
    protected final String getActionName()
    {
        return m_actionName;
    }



    /**
     * Returns the query string after the "&#63;" in the
     * action's name. Will return <i>null</i> if no such string
     * defined. The string returned is as it was entered in the
     * action name URI; no lowercasing or other normalization is
     * applied.
     **/
    protected final String getUriParams0()
    {
        return m_uriParams;
    }



    /**
     * Adds a custom parameter to this control task. This
     * parameter is passed to the selected action's method
     * as a simple <span class="src">Map.Entry</span>.
     * @param param the parameter information (non-null)
     * @throws BuildException if parameter is not named.
     **/
    public void addConfiguredParameter(InnerNameValuePair param)
    {
        require_(param!=null,"addParam- nonzro param");
        param.verifyNamed();

        if (m_nestedParams==null) {
            m_nestedParams = AntXFixture.newMap();
        }
        m_nestedParams.put(param.getName(),param.getValue());
    }



    /**
     * Adds a placeholder parameter to this control task. Usually
     * this method is used for dynamic nested elements which cannot
     * be verified until the execute method is triggered.
     * @param param to placeholder  (non-null)
     **/
    protected final void addCheckLaterParameter(InnerNameValuePair param)
    {
        require_(param!=null,"addParamPlaceholder- nonzro param");
        if (m_nestedUnverifiedParams==null){
            m_nestedUnverifiedParams = AntXFixture.newList();
        }
        m_nestedUnverifiedParams.add(param);
    }



    /**
     * Tells this task whether it should suppress feedback chatter.
     * @param level feedback level ("normal", "loud", "quiet", etc.)
     **/
    public void setFeedback(String level)
    {
        m_fbLevel = FeedbackLevel.from(level);
    }



    /**
     * Returns this task's script-supplied quiet setting. Will
     * return {@linkplain FeedbackLevel#NORMAL NORMAL} if never
     * set explicitly.
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }



    /**
     * Executes the requested action passing any nested parameters
     * as options. If no action is named and this task's haltiferror
     * option is disabled, will issue a warning and do nothing.
     * @throws BuildException if no action named and haltiferror is true.
     **/
    public void execute()
    {
        verifyCanExecute_("exec");

        if (m_actionName!=null) {
            Map args = Collections.EMPTY_MAP;
            if (m_nestedParams!=null) {
                args = Collections.unmodifiableMap(m_nestedParams);
            }
            doAction(m_actionName,args,m_rqlink);
        }
    }



    /**
     * Verifies we have a specified action even if it's a default one.
     * @throws BuildException if no action named and haltiferror is true.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_actionName==null) {
            String message = Iteration.uistrs().get("task.needs.this.attr",
                getTaskName(), "action");
            if (isHaltIfError()) {
                log(message, Project.MSG_ERR);
                throw new BuildException(message,getLocation());
            } 
            //NB:*NOT* optional; always warn!
            log(message, Project.MSG_WARN);
        }

        if (m_nestedUnverifiedParams!=null) {
            Iterator itr = m_nestedUnverifiedParams.iterator();
            while (itr.hasNext()) {
                addConfiguredParameter((InnerNameValuePair)itr.next());
            }
            m_nestedUnverifiedParams = null;//incase rerun
        }
    }



    /**
     * Ensures we can execute as part of a simple antlib definition
     * with current action. Uses callback method
     * {@linkplain #isInvalidAntlibAction}.
     **/
    protected final void verifyNotInAntlib()
    {
        //If as Antlib definition, only install operations are permitted!
        if (getAntlibClassLoader()!=null) {//=>in Antlib always?
            if (isInvalidAntlibAction()) {
                String ermsg = uistrs().get
                    ("task.manager.err.illegal.operation",getActionName());
                log(ermsg,Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
        }
    }



    /**
     * Returns <i>true</i> if current action cannot be performed
     * if we are part of a simple antlib. Returns <i>false</i> by
     * default; subclasses should override to test our action name.
     **/
    protected boolean isInvalidAntlibAction()
    {
        return false;
    }



    /**
     * Collapses the visual spacers in an action's name. Spacers
     * are for the script author's benefit; they are omitted from
     * the action method's name.
     * @param s the incoming script action string
     * @return spacer-scrubed name (non-null)
     **/
    private String scrub(String s)
    {
        StringBuffer sb = new StringBuffer(s.length());
        int N= s.length();
        int i= 0;
        while (N!=0) {
            char c = s.charAt(i++);
            if (i==1) {
                if (Character.isJavaIdentifierStart(c))   {
                    sb.append(c);
                }
            } else if (Character.isJavaIdentifierPart(c)) {
                sb.append(c);
            }
            N--;
        }
        return sb!=null ? sb.substring(0) : s;
    }


    private String m_actionName;
    protected Requester m_rqlink = new Requester.ForComponent(this);
    private String m_uriParams;
    private FeedbackLevel m_fbLevel = FeedbackLevel.NORMAL;

    /** This task's script-nested parameters. **/
    protected Map m_nestedParams;
    private List m_nestedUnverifiedParams;
}


/* end-of-SimpleManagerTask.java */