/**
 * $Id: VendorInfoTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.net.URL;
import  java.util.List;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.apis.Buildstrs;
import  com.idaremedia.apis.EmptyBuildstrs;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;


/**
 * Task that converts a standard JWare
 * <span class="src">{@linkplain Buildstrs Buildstrs}</span> bean into a set of
 * project properties.
 * <p>
 * <b> Example Usage:</b><pre>
 *    &lt;vendorinfo/&gt; //implies AntXtras
 *    &lt;vendorinfo name="antxtras"/&gt;
 *    &lt;vendorinfo infoclass="com.idaremedia.antunit.BuildInfo" prefix="jwau."/&gt;
 *    &lt;vendorinfo name="antx" prefix="abc."/&gt;
 *    &lt;vendorinfo name="antx" prefix="abc." fields="longversion,longdate"/&gt;
 *    &lt;vendorinfo infoclass="mycompany.ProductInfo" fields="label,version"/&gt;
 *    &lt;vendorinfo name="pet" mustexist="no"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 **/

public class VendorInfoTask extends AssertableTask
    implements AntLibFriendly
{
    /** AntX's build information (which created/stored in
        <em>final</em> product archive). **/
    private static final String ANTX_BI= AntX.ANTX_PACKAGE_PREFIX+".BuildInfo";


    /**
     * Loads the default list of known name to info class mappings.
     **/
    private static final Properties DEFAULT_KNOWN_MAPPINGS;
    static {
        Properties list = null;
        URL url = VendorInfoTask.class.getResource("vendors.properties");
        if (url!=null) {
            try {
                list = InputFileLoader.loadProperties(url,null);
            } catch(Exception iox) {/*burp*/}
        }
        if (list==null) {
            System.err.println(AntX.uistrs().get("vendorinfo.noini"));
            list = new Properties();
            list.setProperty("none",EmptyBuildstrs.class.getName());
            list.setProperty("antxtras",ANTX_BI);
            list.setProperty("antx",ANTX_BI);
        }
        DEFAULT_KNOWN_MAPPINGS= list;
        list= null;
    }



    /**
     * Initializes a new vendorinfo task.
     **/
    public VendorInfoTask()
    {
        super(AntX.nopackage);
    }


    /**
     * Initializes a new, enclosed vendorinfo task.
     * @param iam CV-label (non-null)
     **/
    public VendorInfoTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets a default short hand name for this task. This name is
     * used if neither a name nor an info class is specified. This
     * name defaults to <span class="src">antxtras</span>" if not
     * set.
     * @param defaultName default (known) short hand name (non-null)
     **/
    protected final void setDefaultName(String defaultName)
    {
        require_(defaultName!=null,"setDfltName- nonzro name");
        m_defaultVendorName = defaultName;
    }

// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the vendor's name using a symbolic short hand name.
     * @param name the (known) short hand name (non-null)
     * @throws BuildException if the info class has already been set
     **/
    public void setName(String name)
    {
        require_(name!=null,"setName- nonzro vendor name");
        if (m_infoClass!=null) {
            String error = uistrs().get("task.one.or.other.attr",
                                        "infoclass","name");
            log(error, Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_vendorName = name;
    }


    /**
     * Returns the vendor's short hand name if known. Returns
     * <i>null</i> if never set.
     **/
    public String getName()
    {
        return m_vendorName;
    }


    /**
     * Sets the vendor's buildstrs class by FQ name. The class
     * must implement the standard JWare <span class="src">Buildstrs</span>
     * interface.
     * @param  infoClass the class to use (non-null)
     * @throws BuildException if the short hand name has already been set
     **/
    public void setInfoClass(Class infoClass)
    {
        require_(infoClass!=null,"setInfoClaz- nonzro claz");
        String error=null;
        if (!Buildstrs.class.isAssignableFrom(infoClass)) {
            error = uistrs().get("task.bad.custimpl.class1",infoClass.getName(),
                                 "Buildstrs");
        }
        if (m_vendorName!=null) {
            error = uistrs().get("task.one.or.other.attr","infoclass","name");
        }
        if (error!=null) {
            log(error, Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_infoClass = infoClass;
    }


    /**
     * Returns the vendor's buildstrs class if known. Will return
     * <i>null</i> if never set and this task has not been executed
     * at least once.
     **/
    public Class getInfoClass()
    {
        return m_infoClass;
    }


    /**
     * Tells this task that it should signal an error if it is
     * unable to either locate or create the vendor's build
     * information.
     * @param must <i>false</i> to let task ignore missing info
     **/
    public void setMustExist(boolean must)
    {
        m_mustExist = must;
    }


    /**
     * Returns <i>true</i> if this task will signal an error if it
     * cannot locate or create the vendor's information. Defaults
     * to <i>true</i> if never set explicitly.
     **/
    public boolean getMustExist()
    {
        return m_mustExist;
    }


    /**
     * Sets a custom property prefix for all of the interned vendor
     * info properties. If undefined each property is prefixed with
     * a standard marker like "<span class="src">vendor.build.</span>".
     * @param prefix the prefix string (non-null)
     **/
    public void setPrefix(String prefix)
    {
        require_(prefix!=null,"setPrefix- nonzro prefix");
        m_customPrefix = prefix;
    }


    /**
     * Returns the custom string this task will use with all of
     * the interned info properties. Will return <i>null</i> if
     * never set explicitly.
     **/
    public String getPrefix()
    {
        return m_customPrefix;
    }


    /**
     * Returns the string that <em>will</em> be used to prefix all
     * of the interned vendor info properties. This method returns
     * a standard prefix if you have not set the prefix explicitly.
     **/
    public String getFinalPrefix()
    {
        if (m_customPrefix!=null) {
            String prefix = m_customPrefix;
            if (!prefix.endsWith(".")) {
                prefix += ".";
            }
            return prefix;
        }
        if (m_vendorName!=null) {
            return m_vendorName+".build.";
        }
        return "vendor.build.";
    }


    /**
     * Tells this task to intern only the info fields described by
     * the comma-delimited list. See this class's static fields
     * for the set of field name choices.
     * @param fieldnames comma-delimited list of names (non-null)
     **/
    public void setFields(String fieldnames)
    {
        require_(fieldnames!=null,"setFields- nonzro list");
        m_fieldsList = Tk.lowercaseFrom(fieldnames);
    }


    /**
     * Returns the list of fields this task will intern. Returns
     * <i>null</i> if this list never set explicitly.
     **/
    public String getFieldsList()
    {
        return m_fieldsList;
    }

// ---------------------------------------------------------------------------------------
// Extracting information from (Arbitrary) Buildstrs:
// ---------------------------------------------------------------------------------------

    public static final String LABEL="label";
    public static final String VERSION="version";
    public static final String LONGVERSION="longversion";
    public static final String DATE="date";
    public static final String LONGDATE="longdate";
    public static final String PLATFORM="platform";
    public static final String HOST="host";


    /**
     * Returns the collection of known short hand names to FQ
     * class names. Never returns <i>null</i>; subclasses can add to
     * this default list.
     **/
    protected Properties getKnownVendors()
    {
        return m_knownVendors;
    }


    /**
     * Determines which <span class="src">Buildstrs</em> implementation
     * class this task will use. The class is determined once; subsequent
     * calls reuse this class reference.
     * @.sideeffect Updates this task's underlying info class reference
     *              if a symbolic vendor-name was specified.
     * @see #getKnownVendors
     **/
    private void determineBuildInfoClass()
    {
        if (m_infoClass==null) {
            if (m_vendorName==null && m_infoClass==null) {
                m_vendorName = m_defaultVendorName;
            }

            if (m_vendorName!=null) {
                String classname = getKnownVendors().getProperty(m_vendorName);
                if (classname==null) {
                    String error = uistrs().get("vendorinfo.nomatch.name",
                                                m_vendorName);
                    if (getMustExist()) {
                        log(error, Project.MSG_ERR);
                        throw new BuildException(error,getLocation());
                    }
                    log(error, Project.MSG_WARN);
                    m_infoClass = EmptyBuildstrs.class;
                } else {
                    try {
                        m_infoClass = Class.forName(classname);
                    } catch(Exception anyX) {
                        String error = uistrs().get("vendorinfo.cant.makeinfo",
                                                    classname);
                        if (getMustExist()) {
                            log(error, Project.MSG_ERR);
                            throw new BuildException(error,getLocation());
                        }
                        log(error, Project.MSG_WARN);
                        m_infoClass = EmptyBuildstrs.class;
                    }
                }
            }//nam!=nul
        }
    }


    /**
     * Returns the <span class="src">Buildstrs</span> this task uses
     * to extract vendor information. Never returns <i>null</i> but
     * can return a placeholder if unable to load/create the named
     * vendor's information (and {@linkplain #getMustExist mustExist}
     * option is <i>false</i>).
     * @.impl Expects this task's infoClass already determined.
     **/
    private Buildstrs getBuildInfo()
    {
        if (m_buildinfo==null) {
            determineBuildInfoClass();
            try {
                m_buildinfo = (Buildstrs)m_infoClass.newInstance();
            } catch(Exception anyX) {
                String error = uistrs().get("vendorinfo.cant.makeinfo",
                                            m_infoClass.getName());
                if (getMustExist()) {
                    log(error, Project.MSG_ERR);
                    throw new BuildException(error,anyX,getLocation());
                }
                log(error, Project.MSG_WARN);
                m_buildinfo = EmptyBuildstrs.INSTANCE;
            }
        }
        return m_buildinfo;
    }


    /**
     * Sets the appropriate field property in this task's project.
     **/
    private void setInfoProperty(String name, String value)
    {
        name = getFinalPrefix()+name;
        checkIfProperty_(name,true);
        getProject().setNewProperty(name,value);
    }


    /**
     * Copies all build information fields to properties.
     **/
    private void putAllProperties(Buildstrs info)
    {
        setInfoProperty(LABEL,info.getDisplayName());
        setInfoProperty(VERSION,info.getVersion());
        setInfoProperty(LONGVERSION,info.getBuildVersion());
        setInfoProperty(DATE,info.getAbbrDate());
        setInfoProperty(LONGDATE,info.getLongDate());
        setInfoProperty(PLATFORM,info.getOS());
        setInfoProperty(HOST,info.getHostID());
    }


    /**
     * Extracts the symbolically named field's value. Returns
     * <i>null</i> if field name not recognized.
     **/
    private String getFieldValue(Buildstrs info, String f)
    {
        String value = null;
        switch (f.charAt(0)) {
            case 'v': {
                if (VERSION.equals(f)) {
                    value = info.getVersion();
                } break;
            }
            case 'd': {
                if (DATE.equals(f)) {
                    value = info.getAbbrDate();
                } break;
            }
            case 'l': {
                if (LABEL.equals(f)) {
                    value = info.getDisplayName();
                } else if (LONGVERSION.equals(f)) {
                    value = info.getBuildVersion();
                } else if (LONGDATE.equals(f)) {
                    value = info.getLongDate();
                } break;
            }
            case 'p': {
                if (PLATFORM.equals(f)) {
                    value = info.getOS();
                } break;
            }
            case 'h': {
                if (HOST.equals(f)) {
                    value = info.getHostID();
                } break;
            }
        }
        return value;
    }


    /**
     * Copies a subset of build information to properties.
     **/
    private void putTheseProperties(Buildstrs info, String filterlist)
    {
        List l= Tk.splitList(filterlist);
        if (!l.isEmpty()) {
            for (int i=0,N=l.size();i<N;i++) {
                String f = l.get(i).toString();
                String value = getFieldValue(info,f);
                if (value!=null) {
                    setInfoProperty(f,value);
                } else {
                    String warning = uistrs().get("vendorinfo.bad.field",f);
                    log(warning,Project.MSG_WARN);
                }
            }
        }
        l=null;
    }


    /**
     * Copies the named vendor's build information into a set of
     * project properties.
     **/
    public void execute()
    {
        verifyCanExecute_("exec");

        Buildstrs info = getBuildInfo();

        if (m_fieldsList==null) {
            putAllProperties(info);
        } else {
            putTheseProperties(info,m_fieldsList);
        }
    }


    private boolean m_mustExist=true;
    private String m_vendorName;
    private Class m_infoClass;
    private String m_customPrefix, m_fieldsList;
    private Buildstrs m_buildinfo;
    private String m_defaultVendorName="antxtras";
    private Properties m_knownVendors= new Properties(DEFAULT_KNOWN_MAPPINGS);
}

/* end-of-VendorInfoTask.java */
