/**
 * $Id: PrintTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.BufferedOutputStream;
import  java.io.IOException;
import  java.io.OutputStream;
import  java.util.Hashtable;
import  java.util.Iterator;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Display arbitrary bits of build information; usually reference objects. Usually
 * defined &lt;print&gt;. PrintTask extends the standard &lt;printenv&gt; task; so it
 * can also be used to print project properties and AntX variables. If no display strategy
 * is specified, a &lt;print&gt; will simply display the string returned by the
 * displayed thing's <i>toString</i> method. If no default printer registry is installed
 * &lt;print&gt; behaves exactly like &lt;printenv&gt;
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;print msgid="banner.msgsbundle" reference="default.msgs"/&gt;
 *   &lt;print msgid="msg.error.default" reference="last.error" filter="..."/&gt;
 *   &lt;print properties="ant.file,ant.version,basedir"/&gt; &lt;-- <i>like printenv</i> --&gt;
 *   &lt;print if="things.debug" reference="mything.id" with="mythings.printer"/&gt;
 *   &lt;print message="Howdy World!" level="verbose"/&gt; &lt;-- <i>like echo</i> --&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      PrinterMapping
 **/

public class PrintTask extends EchoItemsTask
{
    /**
     * Initializes a new PrintTask instance.
     **/
    public PrintTask()
    {
        super(AntX.print);
    }


    /**
     * Initializes a new CV-labeled PrintTask instance.
     * @param iam CV-label (non-null)
     **/
    protected PrintTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets the printer or printer registry this task will use to
     * display project references.
     **/
    public void setWith(String with)
    {
        require_(with!=null,"setwith- nonzro refid");
        m_with = with;
    }


    /**
     * Returns the printer or printer registry this task will
     * use to display items.  Will return <i>null</i> if with
     * parameter never set.
     **/
    public final String getWith()
    {
        return m_with;
    }



    /**
     * Gives this printer a list of printer-specific filter options.
     * Only really useful for single item printing or for common
     * printer options (like "<span class="src">resolve</span>").
     * @param filterstring the printer-specific filter string (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setFilter(String filterstring)
    {
        m_filterstring = filterstring;
    }


    /**
     * Returns the custom printer filter string for this task. Will
     * return <i>null</i> if never set explicitly.
     * @since JWare/AntX 0.5
     **/
    public final String getFilterString()
    {
        return m_filterstring;
    }



    /**
     * Returns the default printer registry used by this task.
     * Never returns <i>null</i> but can return an empty registry.
     * @see PrinterRegistryContext#getPrinterRegistryNoNull
     **/
    private PrinterRegistry getDefaultPrinterRegistry()
    {
        return PrinterRegistryContext.getPrinterRegistryNoNull();
    }


    /**
     * Displays the references list with the strategy specified
     * by the given mapping.
     **/
    protected void printReferencesWith(PrinterMapping mapping,
                                       final Project P, String list,
                                       OutputStream os)
        throws IOException
    {
        require_(mapping!=null,"print- nonzro mappin");

        PrinterRegistry registry = getDefaultPrinterRegistry();
        DisplayStrategy printer = mapping.getPrinter();

        Class filterClass = getKindOfFilterClass();
        String cantprint;

        Iterator keysitr;
        if (Strings.ALL.equals(Tk.lowercaseFrom(list))) {
            Hashtable copy = (Hashtable)P.getReferences().clone();
            keysitr= copy.keySet().iterator();
        } else {
            keysitr= Tk.splitList(list).iterator();
        }

        BufferedOutputStream bos = new BufferedOutputStream(os,1024);
        DisplayStrategy defaultPrinter= registry.getDefaultPrinter();

        while (keysitr.hasNext()) {
            String key = (String)keysitr.next();

            Object object = FixtureExaminer.trueReference(P,key);//@since AntX 0.4
            if (object==FixtureExaminer.IGNORED_REFERENCE) {
                if (!willIncludeUnknowns()) {
                    continue;
                }
                object = unresolvedString(P,key);
            }

            if (filterClass==null/*allowAll*/ || object==null/*passthru*/ ||
                (filterClass.isInstance(object))) {

                DisplayRequest dr = new DisplayRequest(P,key,object);
                dr.setFilter(getFilterString());

                if (!mapping.isMatch(object)) {
                    cantprint = uistrs().get("printer.cant.print.item",mapping.getId(),key);
                    log(cantprint,Project.MSG_WARN);
                    defaultPrinter.print(dr,bos);
                } else {
                    printer.print(dr,bos);
                }
                dr = null;
            }//do-print
        }//while

        keysitr=null;
    }



    /**
     * Displays the references list with the printer registry
     * specified.
     **/
    protected void printReferencesWith(PrinterRegistry registry,
                                       final Project P, String list,
                                       OutputStream os)
        throws IOException
    {
        require_(registry!=null,"print- nonzro registry");

        Class filterClass = getKindOfFilterClass();

        Iterator keysitr;
        if (Strings.ALL.equals(Tk.lowercaseFrom(list))) {
            Hashtable copy = (Hashtable)P.getReferences().clone();
            keysitr= copy.keySet().iterator();
        } else {
            keysitr= Tk.splitList(list).iterator();
        }

        BufferedOutputStream bos = new BufferedOutputStream(os,1024);
        PrinterMapping prevHandler=null;
        DisplayStrategy printer=null;
        DisplayStrategy defaultPrinter= registry.getDefaultPrinter();

        while (keysitr.hasNext()) {
            String key = (String)keysitr.next();

            Object object = FixtureExaminer.trueReference(P,key);//@since AntX 0.4
            if (object==FixtureExaminer.IGNORED_REFERENCE) {
                if (!willIncludeUnknowns()) {
                    continue;
                }
                object = unresolvedString(P,key);
            }

            if (filterClass==null/*allowAll*/ || object==null/*passthru*/ ||
                (filterClass.isInstance(object))) {

                if (prevHandler!=null && prevHandler.isMatch(object)) {
                    printer= prevHandler.getPrinter();
                } else {
                    PrinterMapping handler= registry.findMappingFor(object);
                    if (handler==null) {
                        printer = defaultPrinter;
                    } else {
                        printer = handler.getPrinter();
                        prevHandler = handler;
                    }
                }
                DisplayRequest dr = new DisplayRequest(P,key,object);
                dr.setFilter(getFilterString());
                printer.print(dr,bos);
                dr = null;
            }//do-print
        }//while

        keysitr=null;
    }


    /**
     * Returns <i>true</i> if this task should try to customize the
     * display references. By default will check if either the 'with'
     * parameter or the active registry is enabled. If yes, will try
     * to use custom display strategy.
     **/
    private boolean tryCustomDisplay()
    {
        PrinterRegistry dfltPR= getDefaultPrinterRegistry();
        if (dfltPR!=PrinterRegistryContext.getEmptyPrinterRegistry()) {
            return true;
        }
        return false;
    }


    /**
     * Displays references using the frontmost context printer registry.
     **/
    protected boolean echoReferences(final Project P, OutputStream os)
        throws IOException
    {
        String list= getReferencesNameList();
        if (list!=null) {
            String with = getWith();
            if (with!=null) {
                Object withObject = P.getReference(with);
                if (withObject instanceof PrinterRegistry) {
                    printReferencesWith((PrinterRegistry)withObject,P,list,os);
                } else if (withObject instanceof PrinterMapping) {
                    printReferencesWith((PrinterMapping)withObject,P,list,os);
                } else {
                    String ermsg = uistrs().get("print.bad.with",with);
                    log(ermsg, Project.MSG_WARN);
                    if (tryCustomDisplay()) {
                        printReferencesWith(getDefaultPrinterRegistry(),P,list,os);
                    } else {
                        return super.echoReferences(P,os);
                    }
                }
                return true;
            } else if (tryCustomDisplay()) {
                printReferencesWith(getDefaultPrinterRegistry(),P,list,os);
            } else {
                return super.echoReferences(P,os);
            }
        }
        return false;
    }



    private String m_with;//optional
    private String m_filterstring;//optional
}

/* end-of-PrintTask.java */
