/**
 * $Id: ErrorPrinter.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.IOException;
import  java.io.OutputStream;
import  java.io.PrintStream;
import  java.util.List;
import  java.util.Properties;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ErrorSnapshot;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.apis.UIStringManager;

/**
 * Default strategy for printing an {@linkplain ErrorSnapshot ErrorSnapshot} to an
 * output stream. Basically dumps a big ol' Properties object containing all snapshot's
 * information as fields.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @see      EchoErrorTask
 **/

public final class ErrorPrinter implements DisplayStrategy
{
    /**
     * Initializes a new ErrorPrinter.
     **/
    public ErrorPrinter()
    {
    }


    /**
     * Converts an error snapshot's header fields to several Properties
     * entries.
     * @param id the display name of the request (non-null)
     * @param es the error snapshot (non-null)
     **/
    private Properties propertiesFromHeader(String id, ErrorSnapshot es)
    {
        /** Include Header information as Property=Value entries **/
        Properties allP = new Properties();

        Object thrown = es.getThrown();
        if (thrown==null) { thrown = ""; }
        allP.setProperty(uistrs().get("snapshot.print.thrown",id),thrown.toString());

        allP.setProperty(uistrs().get("snapshot.print.location",id),es.getLocation().toString());
        allP.setProperty(uistrs().get("snapshot.print.task",id),es.getTaskName());
        allP.setProperty(uistrs().get("snapshot.print.target",id),es.getTargetName());
        allP.setProperty(uistrs().get("snapshot.print.comment",id),es.getComment());

        return allP;
    }


    /**
     * Produces the appropriate set of captured properties. Will return
     * <i>null</i> if no properties requested.
     * @param id the display name of the request (non-null)
     * @param es the error snapshot (non-null)
     * @param list list of properties to display (non-null)
     **/
    private Properties propertiesFromEnv(String id, ErrorSnapshot es, String list)
    {
        String ll= Tk.lowercaseFrom(list);
        if (ll.length()==0 || Strings.NONE.equals(ll)) {
            return null;
        }

        /** Start with the entire universe captured by the snapshot **/
        Properties allP = es.copyOfProperties();

        /** Filter to only the properties requested (can be all) **/
        if (!Strings.ALL.equals(ll)) {
            List wanted = Tk.splitList(list,",");
            allP.keySet().retainAll(wanted);

            if (wanted.size()>allP.size()) {//insert "null" strings
                for (int i=0,N=wanted.size();i<N;i++) {
                    String key = (String)wanted.get(i);
                    if (allP.getProperty(key)==null) {
                        allP.setProperty(key,Strings.NULL);
                    }
                }
            }
            wanted=null;
        }

        return allP;
    }


    /**
     * Generates an ErrorSnapshot as a big ol' Properties listing
     * to the given output stream.
     * @param info display information (non-null)
     * @param os the output stream (non-null)
     * @throws IOException if any I/O error occurs
     **/
    public void print(final DisplayRequest info, OutputStream os)
        throws IOException
    {
        Object target = info.getObjectToBeDisplayed();

        /** Ignore nulls (do nada), only do ErrorSnapshots **/
        if (target instanceof ErrorSnapshot) {

            ErrorSnapshot es = (ErrorSnapshot)target;
            String id = info.getName();
            if (id==null) {
                id = es.getName();
                if (id==null) {
                    id = Strings.NULL;
                }
            }

            Properties allP;

            /** Always ensure header information is first. **/
            new PrintStream(os).println("########");
            allP = propertiesFromHeader(id,es);
            allP.store(os,uistrs().get("snapshot.print.header",id));
            allP.clear();
            allP=null;

            /** Write the requested environment stuff next. **/
            String list = info.getFilter();
            if (list==null) {
                list = Strings.ALL;
            }
            allP = propertiesFromEnv(id,es,list);
            if (allP!=null) {
                allP.store(os,uistrs().get("snapshot.print.content",id));
                allP.clear();
                allP=null;
            }
        }
        /** Ignore nulls (do nada), only do Throwables **/
        else if (target instanceof Throwable ) {

            Throwable thrown = (Throwable)target;
            PrintStream ps = new PrintStream(os);

            ps.println("########");
            ps.print("#");
            ps.println(uistrs().get("thrown.print.header",Tk.leafNameFrom(thrown.getClass())));
            ps.print("#");
            ps.println(new java.util.Date());
            thrown.printStackTrace(ps);
            ps.flush();
            ps=null;
        }
    }


    private UIStringManager uistrs()
    {
        return AntX.uistrs();
    }
}

/* end-of-ErrorPrinter.java */
