/**
 * $Id: DisplayRequest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectComponent;

import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.apis.ProjectDependent;
import  com.idaremedia.antx.apis.Requester;

/**
 * Information to be displayed by a {@linkplain DisplayStrategy display strategy}.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      DisplayStrategy
 **/

public final class DisplayRequest implements Nameable, ProjectDependent
{
    /**
     * Initializes a new empty display request.
     **/
    public DisplayRequest()
    {
    }


    /**
     * Initializes a new display request for given object.
     * @param thing thing to be displayed
     **/
    public DisplayRequest(Object thing)
    {
        setObjectToBeDisplayed(thing);
    }


    /**
     * Initializes a new display request for given named
     * object in project.
     * @param p enclosing project
     * @param thing thing to be displayed
     * @param id the thing's identifier (in build script)
     **/
    public DisplayRequest(Project p, String id, Object thing)
    {
        setObjectToBeDisplayed(thing);

        setProject(p); //NB:do these *after* settin obj
        setName(id);
    }


    /**
     * Returns the thing-to-be-displayed's identifier.
     * Will return <i>null</i> if never set.
     **/
    public String getName()
    {
        return m_Id;
    }


    /**
     * Sets the thing-to-be-displayed's identifier. Use <i>null</i>
     * to reset name to default setting.
     * @param name thing's name
     **/
    public void setName(String name)
    {
        m_Id = name;
    }


    /**
     * Returns the thing-to-be-displayed. Will return <i>null</i>
     * if never set.
     **/
    public Object getObjectToBeDisplayed()
    {
        return m_thing;
    }


    /**
     * Sets the thing-to-be-displayed.
     * @param thing the ting
     * @.sideeffect (Re)Sets this request's project if thing is ProjectComponent
     * @.sideeffect (Re)Sets this request's name  if thing is Nameable
     **/
    public void setObjectToBeDisplayed(Object thing)
    {
        m_thing = thing;

        if (thing instanceof ProjectComponent) {
            setProject(((ProjectComponent)thing).getProject());
        }
        if (thing instanceof AssertableDataType) {//NB:ordering explicit
            setName(((AssertableDataType)thing).getId());
        } else if (thing instanceof Nameable) {
            setName(((Nameable)thing).getName());
        }
    }


    /**
     * Returns the filtering information for this request.
     * Will return <i>null</i> if never set or was reset.
     **/
    public String getFilter()
    {
        return m_filter;
    }


    /**
     * Sets the filtering information for this request. Exactly
     * how this information is used depends on the strategy being
     * used and the thing-to-be-displayed. Use <i>null</i> to
     * unset a filter for a reused request.
     * @param filter the opaque filter information
     **/
    public void setFilter(String filter)
    {
        m_filter = filter;
    }


    /**
     * Returns this request's enclosing project. Will return
     * <i>null</i> if never defined and no callback information
     * set.
     **/
    public Project getProject()
    {
        Project project = m_project;
        if (project==null && m_rqlink!=Requester.ANONYMOUS) {
            project = m_rqlink.getProject();
        }
        return project;
    }


    /**
     * Sets this request's enclosing project.
     * @param project the enclosing project
     **/
    public void setProject(Project project)
    {
        m_project = project;
        if (project!=null && m_rqlink==Requester.ANONYMOUS) {
            m_rqlink = new Requester.ForProject(project);
        }
    }



    /**
     * Returns caller information if available. If never set explicitly,
     * returns either an anonymous caller or a caller based on this
     * request's project. Never returns <i>null</i>.
     * @since JWare/AntX 0.5
     **/
    public Requester getCaller()
    {
        return m_rqlink;
    }



    /**
     * Sets this request's user's callback information.
     * @param clnt [optional] caller information.
     * @since JWare/AntX 0.5
     **/
    public void setCaller(Requester clnt)
    {
        if (clnt!=null) {
            m_rqlink = clnt;
        } else {
            m_rqlink = Requester.ANONYMOUS;
        }
    }


    private Project   m_project;
    private Object    m_thing;
    private String    m_filter;
    private String    m_Id;
    private Requester m_rqlink = Requester.ANONYMOUS;
}

/* end-of-DisplayRequest.java */
