/**
 * $Id: ExecutionMode.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the various ways a task can be run by the Ant
 * runtime (or some other execution controller). The following list explains what
 * the various settings mean:<ul>
 *    <li><span class="src">local</span>: The task should be executed within the
 *        current project's scope.</li>
 *    <li><span class="src">localthread</span>: The task should be executed
 *        within the current project's scope but from a different thread.</li>
 *    <li><span class="src">isolated</span>: The task should be executed within
 *        it's own (child) project's scope. Similar to how the
 *        <span class="src">&lt;antcall&gt;</span> task works; most modifications
 *        to the child project are not seen by the current (parent) project.</li>
 *    <li><span class="src">external</span>: The task should be executed in its
 *        own VM instance.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 **/

public final class ExecutionMode extends EnumSkeleton
{
    /** Index of {@linkplain #LOCAL LOCAL}. **/
    public static final int LOCAL_INDEX = 0;
    /** Index of {@linkplain #LOCALTHREAD LOCALTHREAD}. **/
    public static final int LOCALTHREAD_INDEX = LOCAL_INDEX+1;
    /** Index of {@linkplain #ISOLATED ISOLATED}. **/
    public static final int ISOLATED_INDEX = LOCALTHREAD_INDEX+1;
    /** Index of {@linkplain #EXTERNAL EXTERNAL}. **/
    public static final int EXTERNAL_INDEX = ISOLATED_INDEX+1;


    /** Singleton "<span class="src">local</span>" choice. **/
    public static final ExecutionMode LOCAL=
        new ExecutionMode("local",LOCAL_INDEX);

    /** Singleton "<span class="src">localthread</span>" choice. **/
    public static final ExecutionMode LOCALTHREAD=
        new ExecutionMode("localthread",LOCALTHREAD_INDEX);

    /** Singleton "<span class="src">isolated</span>" choice. **/
    public static final ExecutionMode ISOLATED=
        new ExecutionMode("isolated",ISOLATED_INDEX);

    /** Singleton "<span class="src">external</span>" choice. **/
    public static final ExecutionMode EXTERNAL=
        new ExecutionMode("external",EXTERNAL_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public ExecutionMode()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private ExecutionMode(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"local", "localthread",
                             "isolated", "external"};
    };



    /**
     * Helper that converts a scalar to a known ExecutionMode.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static ExecutionMode from(int i)
    {
        if (i==LOCAL.index)    { return LOCAL; }
        if (i==ISOLATED.index) { return ISOLATED; }
        if (i==EXTERNAL.index) { return EXTERNAL; }
        if (i==LOCALTHREAD.index) { return LOCALTHREAD; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * ExecutionMode's index.
     * @param i the index to be matched
     * @param dflt the default ExecutionMode if necessary
     **/
    public static ExecutionMode from(int i, ExecutionMode dflt)
    {
        ExecutionMode choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known ExecutionMode
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either ExecutionMode's symbolic name or its index.
     **/
    public static ExecutionMode from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (LOCAL.value.equals(s))      { return LOCAL; }
                if (ISOLATED.value.equals(s))   { return ISOLATED; }
                if (EXTERNAL.value.equals(s))   { return EXTERNAL; }
                if (Strings.DEFAULT.equals(s))  { return ISOLATED; }/*safest*/
                if (LOCALTHREAD.value.equals(s)){ return LOCALTHREAD; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * ExecutionMode's name.
     * @param s the symbolic name to be matched
     * @param dflt the default ExecutionMode if necessary
     **/
    public static ExecutionMode from(String s, ExecutionMode dflt)
    {
        ExecutionMode choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-ExecutionMode.java */
