/**
 * $Id: ErrOutHandle.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers.io;

import  java.io.OutputStream;
import  java.io.PrintStream;

import  com.idaremedia.antx.AntX;
import com.idaremedia.antx.apis.BuildError;

/**
 * Helper that lets you easily swap (temporarily) the standard err and out streams
 * with your own handlers.
 * <p>
 * <b>Example Usage:</b><pre>
 *    ErrOutHandle devnull = new ErrOutHandle();
 *    devnull.install();
 *      ...//any output generated by these bits will be ignored
 *    devnull.uninstall(false,true);
 * </pre>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      NullPrintStream
 * @see      BufferStream
 **/

public final class ErrOutHandle
{
    private static final String IAM_ = AntX.utilities+"ErrOutHandle";



    /**
     * Initializes a new helper.
     **/
    public ErrOutHandle()
    {
    }



    /**
     * Initializes a new helper and immediately swaps out the
     * two System streams.
     * @param out new system standard output stream (non-null)
     * @param err new system error output stream (non-null)
     * @throws SecurityException if no permission to swap both streams.
     **/
    public ErrOutHandle(OutputStream out, OutputStream err)
    {
        install(new PrintStream(out),new PrintStream(err));
        m_newOut0 = out;
        m_newErr0 = err;
    }



    /**
     * Installs this two streams in place of current
     * <span class="src">System&#46;out</span> and
     * <span class="src">System&#46;err</span>.
     * @param out new system standard output stream (non-null)
     * @param err new system error output stream (non-null)
     * @throws SecurityException if no permission to swap both streams.
     **/
    public void install(PrintStream out, PrintStream err)
    {
        AntX.require_(out!=null && err!=null,IAM_,"install- nonzro streams");
        try {
            PrintStream os;

            os = System.out;
            System.setOut(out);
            m_oldOut = os;
            m_newOut = out;

            os = System.err;
            System.setErr(err);
            m_oldErr = os;
            m_newErr = err;
        }
        catch(RuntimeException secX) {
            //NB: ?is it possible to have perm to do one but not other?
            if (m_oldErr!=null) {
                System.setErr(m_oldErr);
            }
            if (m_oldOut!=null) {
                System.setOut(m_oldOut);
            }
            clear();
            throw secX;
        }
    }



    /**
     * Installs two "<span class="src">dev/null</span>" dummies
     * for <span class="src">System&#46;out</span> and
     * <span class="src">System&#46;err</span>. All output is
     * thrown away. <em>Must be followed by a call to
     * {@linkplain #uninstall(boolean,boolean) uninstall}.</em>
     * @throws SecurityException if no permission to swap both streams.
     * @see NullPrintStream
     **/
    public final void install()
    {
        install(new NullPrintStream(), new NullPrintStream());
    }



    /**
     * Restores the two systems streams that where in place when
     * this object's {@linkplain #install(PrintStream,PrintStream)
     * install} method was entered.
     * @param flush <i>true</i> if current streams should be flushed
     *    before being replaced.
     * @param check <i>true</i> if current streams should be checked
     *    to insure they're the ones this handle installed.
     * @throws SecurityException if no permission to swap both streams.
     * @throws BuildError if current streams aren't ours.
     * @throws IllegalStateException if not installed.
     **/
    public void uninstall(boolean flush, boolean check)
    {
        AntX.verify_(wasInstalled(),IAM_,"uninstall- installed");
        if (flush) {
            try {
                System.out.flush();
                System.err.flush();
            } catch(Exception iox) {/*burp*/}
        }
        if (check) {
            if (System.out!=m_newOut || System.err!=m_newErr) {
                String error = AntX.uistrs().get("fixture.stdio.outof.sync");
                throw new BuildError(error);
            }
        }
        System.setErr(m_oldErr);
        System.setOut(m_oldOut);
        clear();
    }



    /**
     * Like {@linkplain #uninstall(boolean,boolean)
     * uninstall(&#8230;)} but with flushing and checking
     * turned on.
     **/
    public final void uninstall()
    {
        uninstall(true,true);
    }



    /**
     * Returns <i>true</i> if this handle has been installed.
     **/
    public final boolean wasInstalled()
    {
        return m_oldOut!=null && m_oldErr!=null;
    }




    /**
     * Returns our underlying <span class="src">System&#46;out</span>
     * output stream. Returns <i>null</i> if not known.
     **/
    public final OutputStream getOurSystemOut()
    {
        return m_newOut0;
    }


    /**
     * Returns our underlying <span class="src">System&#46;err</span>
     * output stream. Returns <i>null</i> if not known.
     **/
    public final OutputStream getOurSystemErr()
    {
        return m_newErr0;
    }



    /**
     * Returns replaced <span class="src">System&#46;out</span>
     * stream reference. Returns <i>null</i> if not installed.
     **/
    public final PrintStream getReplacedSystemOut()
    {
        return m_oldOut;
    }


    /**
     * Returns replaced <span class="src">System&#46;err</span>
     * stream reference. Returns <i>null</i> if not installed.
     **/
    public final PrintStream getReplacedSystemErr()
    {
        return m_oldErr;
    }




    /**
     * Remove all our references to any streams.
     **/
    private void clear()
    {
        m_newOut  = m_newErr  = null;
        m_oldOut  = m_oldErr  = null;
        m_newOut0 = m_newErr0 = null;
    }


    private PrintStream m_newOut, m_newErr;
    private OutputStream m_newOut0, m_newErr0;//underlying ps
    private PrintStream m_oldOut, m_oldErr;
}

/* end-of-ErrOutHandle.java */
