/**
 * $Id: PropertyExaminer.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Map;
import  java.util.Vector;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.NameValuePair;

/**
 * Little helper that can evaluate a project property definition for unresolved
 * variable references. The helper can be setup to sniff for "simple" all-in-one
 * unresolved references (the default), or for complex, nested references (more
 * memory/time costly).
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple, after examiner configured
 * @.group   impl,helper
 * @see      PropertiesIterator
 **/

public class PropertyExaminer extends ProjectDependentSkeleton
{
    private static final String IAM_ = AntX.utilities+"PropertyExaminer:";


    /** Simple unresolved property check.  **/
    private static final int DONTCHECK= 0;
    /** Simple unresolved property check.  **/
    private static final int SIMPLE= DONTCHECK+100;
    /** Anal unresolved property check.    **/
    private static final int THOROUGH= SIMPLE+1;


    /**
     * Initializes new property examiner instance. The target
     * project must be suppled on a call-by-call basis or
     * <span class="src">setProject</span> must be called.
     **/
    public PropertyExaminer()
    {
    }


    /**
     * Initializes new project-specific property examiner instance.
     * @param P target project for property checks (non-null)
     **/
    public PropertyExaminer(Project P)
    {
        setProject(P);
    }


    /**
     * Tells this examiner to check the property for an all-in-one
     * failed variable substitution. If called, this examiner will
     * either skip bad properties or return a proxy if one defined.
     * @see #setBrokenSubstitutionProxy setBrokenSubstitutionProxy(String)
     **/
    public final void setCheckSimpleBrokenSubstitution()
    {
        m_checkBadSubst = SIMPLE;
    }


    /**
     * Tells this examiner to thoroughly check the property for any
     * bad (failed) variable substitution. If called, this examiner
     * will either skip bad properties or return a proxy if one
     * defined.
     * @see #setBrokenSubstitutionProxy setBrokenSubstitutionProxy(String)
     **/
    public final void setCheckBrokenSubstitution()
    {
        m_checkBadSubst = THOROUGH;
    }


    /**
     * Tells this examiner to return property candidate as-is.
     * Effectively makes this examiner a no-op.
     **/
    public final void clearCheckBrokenSubstitution()
    {
        m_checkBadSubst = DONTCHECK;
    }


    /**
     * Returns <i>true</i> if this examiner will check for
     * unresolved property definitions (simple or complex). Is
     * <i>true</i> by default.
     * @see #clearCheckBrokenSubstitution
     **/
    public final boolean willCheckBrokenSubstitution()
    {
        return m_checkBadSubst!=DONTCHECK;
    }


    /**
     * Define a proxy value string for failed substitution
     * properties. This proxy has no affect unless this examiner
     * is setup to check for bad property substitution.
     * @param valueProxy string to use for property value (non-null)
     * @see UnresolvedProperty#VALUE
     **/
    public final void setBrokenSubstitutionProxy(String valueProxy)
    {
        AntX.require_(valueProxy!=null,IAM_,"setProxy- nonzro valu");
        m_valueProxy = valueProxy;
    }


    /**
     * Common work that filters the given value through this examiner's
     * bad-substitution check. Will return <i>false</i> if entire
     * value is unresolved or this is a complex examiner and
     * part of value is unresolved.
     **/
    private boolean isBroken(String value, Project P, String name)
    {
        boolean broken = value==null;
        if (!broken && m_checkBadSubst!=DONTCHECK) {
            value  = value.trim();
            broken = value.startsWith("${") && value.endsWith("}");

            if (!broken && m_checkBadSubst!=SIMPLE) {//tst nested broken
                int i = value.indexOf("${");
                if (i>=0) {
                    PropertyHelper Ph= PropertyHelper.getPropertyHelper(P);
                    Vector ignore = new Vector(); /*icky-API*/
                    Vector misses = new Vector(); /* "   " */
                    try {
                        Ph.parsePropertyString(value,ignore,misses);
                        if (!misses.isEmpty()) {
                            broken= true;
                        }
                    } catch(BuildException malformX) {
                        String warning = AntX.uistrs().get
                            ("fixture.bad.propdef.found", name,value);
                        P.log(warning,Project.MSG_WARN);
                        broken= true;
                    }
                }//might-be
            }
        }
        return broken;
    }


    /**
     * Filters the incoming value through this examiner's
     * bad-substitution check. Will return <i>false</i> if could
     * not verify value.
     * @param value value to check (non-null)
     * @param P target project (non-null)
     **/
    public boolean verifiedLiteral(String value, Project P)
    {
        AntX.require_(value!=null,IAM_,"verifyLiteral- nonzro value");
        AntX.require_(P!=null,IAM_,"verifyLiteral- nonzro project");
        return !isBroken(value,P,"");
    }


    /**
     * Shortcut for {@linkplain #verifiedLiteral(String,Project)
     * verifiedLiteral(String,Project)} that uses this examiner's
     * own project.
     **/
    public final boolean verifiedLiteral(String value)
    {
        return verifiedLiteral(value,getProject());
    }


    /**
     * Filters the incoming property through this examiner's
     * bad-substitution check. Will return a replacement entry if
     * incoming entry fails check and a value-proxy was defined.
     * Otherwise will return <i>null</i> if the entry fails check.
     * @param entry property key-value pair (non-null)
     * @param P target project (non-null)
     * @see #setCheckBrokenSubstitution
     **/
    public Map.Entry verifiedPropertyValue(Map.Entry entry, Project P)
    {
        AntX.require_(entry!=null && entry.getKey()!=null,IAM_,
                      "verifyProp- nonzro property");
        AntX.require_(P!=null,IAM_,"verifyProp- nonzro project");

        Map.Entry result = entry;
        boolean broken = isBroken((String)entry.getValue(),P,
                                  entry.getKey().toString());
        if (broken) {
            //?replace
            if (m_valueProxy!=null) {
                NameValuePair nvp = new NameValuePair();
                result = nvp;
                nvp.setName(entry.getKey().toString());
                nvp.setValue(m_valueProxy);
            }
            //?or skip
            else {
                result = null;
            }
        }
        return result;
    }


    /**
     * Shortcut for {@linkplain #verifiedPropertyValue(Map.Entry,Project)
     * verifiedPropertyValue(Entry,Project)} that uses this examiner's
     * own project.
     **/
    public final Map.Entry verifiedPropertyValue(Map.Entry entry)
    {
        return verifiedPropertyValue(entry,getProject());
    }


    private int m_checkBadSubst=SIMPLE;//NB:=> return as-is
    private String m_valueProxy;//NB:=> skip-bad
}


/* end-of-PropertyExaminer.java */
