/**
 * $Id: InnerProperties.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.PropertySource;

/**
 * Custom &lt;propertyset&gt; that simplifies syntax for succinct inclusion in
 * assertions and other AntX rules. Also supports the AntX nomenclature for a
 * property's {@linkplain PropertySource source domain}.
 * <p>
 * Note that an <span class="src">InnerProperties</span>'s default domain is
 * "<span class="src">script--</span>" <em>not</em> "<span class="src">all</span>".
 * <p>
 * For Example:<pre>
 *   &lt;allset malformed="reject"&gt;
 *      &lt;properties prefix="jware.antx." domain="all--"/&gt;
 *      &lt;properties like="^build\..*$"/&gt;
 *      &lt;properties prefix="build." ignorecase="yes"/&gt;
 *   &lt;/allset&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      PropertiesIterator
 **/

public class InnerProperties extends AssertableProjectComponent
    implements IgnoreCaseEnabled, Cloneable
{
    private static final String PATTERN="like";
    private static final String PREFIX="prefix";
    private static final String NOTLIKE="notlike";


    /**
     * Initializes a new properties declaration.
     **/
    public InnerProperties()
    {
        super(AntX.nopackage);
    }


    /**
     * Initializes a new utility properties
     * declaration.
     **/
    public InnerProperties(String iam)
    {
        super(iam);
    }



    /**
     * Returns a clone of this properties declaration.
     * @since JWare/AntX 0.5
     */
    public Object clone()
    {
        try {
            return super.clone();
        } catch(CloneNotSupportedException clnX) {
            throw new CloneBrokenError();
        }
    }



    /**
     * Defines a regular expression filter for properties.
     **/
    public void setLike(String pattern)
    {
        require_(!Tk.isWhitespace(pattern),"setLik- nonzro pattern");
        setNewFilter(PATTERN,pattern);
    }


    /**
     * Returns this declaration's regular expression filter.
     * Returns <i>null</i> if never set.
     **/
    public String getLikeFilterPattern()
    {
        return getFilterParameter(PATTERN);
    }


    /**
     * Defines a prefix filter for properties.
     **/
    public void setPrefix(String prefix)
    {
        require_(prefix!=null,"setPrefx- nonzro prefix");
        setNewFilter(PREFIX,prefix);
    }


    /**
     * Returns this declaration's prefix filter.
     * Returns <i>null</i> if never set.
     **/
    public String getPrefixFilterPattern()
    {
        return getFilterParameter(PREFIX);
    }


    /**
     * Defines a regular expression filter for excluded properties.
     **/
    public void setNotLike(String pattern)
    {
        require_(!Tk.isWhitespace(pattern),"setNotLik- nonzro pattern");
        setNewFilter(NOTLIKE,pattern);
    }


    /**
     * Returns this declaration's exclusion regular expression filter.
     * Returns <i>null</i> if never set.
     **/
    public String getNotLikeFilterPattern()
    {
        return getFilterParameter(NOTLIKE);
    }


    /**
     * Sets the source of this declaration's properties. This setter
     * allows all the synonyms for the AntX property sources that
     * are compatible with Ant's own &lt;propertyset&gt;'s
     * <span class="src">builtin</span> parameter.
     * @param domainname symbolic name of domain (synonyms allowed)
     **/
    public void setDomain(String domainname)
    {
        PropertySource domain = PropertySource.from(domainname);
        require_(domain!=null,"setDomain- nonzro domain");
        m_domain = domain;
    }


    /**
     * Returns the source of this declaration's properties.
     * Returns {@linkplain PropertySource#SCRIPTlite} if
     * never set explicitly.
     **/
    public PropertySource getDomain()
    {
        return m_domain;
    }


    /**
     * Tells this declaration to ignore case (or not)
     * in filters.
     * @param ignore <i>true</i> to ignore
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_ignoreCase = ignore;
    }


    /**
     * Returns <i>true</i> if this declaration wants
     * case-insensitive filtering. Is <i>false</i> by default.
     **/
    public boolean isIgnoreCase()
    {
        return m_ignoreCase;
    }


    /**
     * Called whenever a filter is defined; ensures at most one
     * filter is defined.
     **/
    private void setNewFilter(String choice, String param)
    {
        if (m_filterType!=null) {
            String error = uistrs().get
                ("task.one.or.other.attr","like","prefix");
            log(error, Project.MSG_ERR);
            throw new BuildException(error);
        }
        m_filterType = choice;
        m_filterParameter = param;
    }


    /**
     * Returns the filter parameter string if if the caller's
     * type matches what was set. Otherwise, returns <i>null</i>.
     **/
    private String getFilterParameter(String choice)
    {
        return (m_filterType==choice) ? m_filterParameter : null;
    }


    private String m_filterType;
    private String m_filterParameter;
    private PropertySource m_domain = PropertySource.SCRIPTlite;
    private boolean m_ignoreCase;
}

/* end-of-InnerProperties.java */
