/**
 * $Id: TruncateFileTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;
import  java.io.FileOutputStream;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.FileSet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.ownhelpers.FileSetsIterator;
import  com.idaremedia.antx.parameters.FeedbackLevel;
import  com.idaremedia.antx.parameters.RecoveryEnabled;

/**
 * Task that sets the size of a file to zero making it empty. A faster way to do the
 * next snippet in tests.
 * <pre>
 *   &lt;replaceregexp flags="s"&gt;
 *      &lt;regexp pattern=".*"/&gt;
 *      &lt;substitution expression=""/&gt;
 *      ...
 *   &lt;/replaceregexp&gt;
 * </pre>
 * <b>Example Usage:</b><pre>
 *   &lt;truncatefile file="${dashboard}/build.status" mustexist="yes"/&gt;
 *   &lt;truncatefile projectfile="subbuild.xml"/&gt;
 *   &lt;truncatefile file="subbuild.xml" createIfMissing="yes"/&gt;
 *   &lt;truncatefile projectfile="vars.in" haltIfError="yes"/&gt;
 *
 *   &lt;truncatefiles haltiferror="yes" feedback="verbose"&gt;
 *      &lt;files dir="${basedir}/etc"&gt;
 *         &lt;include name="*.log"/&gt;
 *         &lt;include name="*.last"/&gt;
 *      &lt;/files&gt;
 *      &lt;file name="build.last"/&gt;
 *      &lt;files dir="${filterfiles}/dynamic"&gt;
 *         &lt;include name="**&#47;*.xml,vars"/&gt;
 *         &lt;include name="*.properties,vars"/&gt;
 *      &lt;/files&gt;
 *   &lt;/truncatefiles&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.5
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public class TruncateFileTask extends AssertableTask
    implements RecoveryEnabled
{

    /**
     * Initializes a new TruncateFileTask instance.
     **/
    public TruncateFileTask()
    {
        super(AntX.mktemp+"TruncateFileTask:");
    }


    /**
     * Initializes a new truncate file subclass instance.
     * @param iam CV-label (non-null)
     **/
    protected TruncateFileTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes this task instance's field helpers.
     */
    public void init()
    {
        super.init();
        m_filerefs = AntXFixture.newList();
    }



    /**
     * Tells this task how much non-diagnostic feedback to generate.
     * Really only has "loud" vs. "quiet-ish" interpretation. If
     * set quiet, this task will not issue a warning if it hits a
     * hard limit.
     * @param level feedback level (non-null)
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fbl = FeedbackLevel.from(level);
        if (fbl==null) {
            String e = getAntXMsg("task.illegal.param.value",
                           getTaskName(), level,"feedback");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fbl;
    }



    /**
     * Returns this task's assigned feedback level. Will return
     * <i>null</i> by default.
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }



    /**
     * Tells this task whether it should stop the build
     * process if unable to empty and existing file or create
     * a new empty file.
     * @param halt <i>true</i> if should stop
     **/
    public void setHaltIfError(boolean halt)
    {
        m_haltIfError= halt;
    }


    /**
     * Returns <i>true</i> if this task will generate an build
     * error if it is unable to empty an existing file or create
     * a new empty file.
     **/
    public boolean isHaltIfError()
    {
        return m_haltIfError;
    }


    /**
     * Tells this task whether the file to be emptied must
     * already exist (and be writable).
     * @param must <i>true</i> if must exist
     **/
    public void setMustExist(boolean must)
    {
        m_mustExist = must;
    }


    /**
     * Returns <i>true</i> if the file to be emptied must
     * already exist.
     **/
    public boolean getMustExist()
    {
        return m_mustExist;
    }


    /**
     * Tells this task whether it's ok to create a new empty
     * file if file does not exist.
     * @param create <i>true</i> to create new empty file
     **/
    public void setCreateIfMissing(boolean create)
    {
        m_createIfMissing = create;
    }



    /**
     * Returns <i>true</i> if will create an empty new file if
     * it does not already exist.
     **/
    public final boolean willCreateIfMissing()
    {
        return m_createIfMissing;
    }



    /**
     * Tells this task the path of file to be emptied.
     * @param filepath the path of file to be emptied (non-null)
     **/
    public void setFile(String filepath)
    {
        require_(filepath!=null,"setFile- nonzro path");
        m_filerefs.add(getProject().resolveFile(filepath));
    }



    /**
     * Tells this task the file to be emptied.
     * @param file the file to be emptied (non-null)
     **/
    public void setProjectFile(File file)
    {
        require_(file!=null,"setFile- nonzro file");
        m_filerefs.add(file);
    }



    /**
     * Returns a new fileset for (independent) configuration.
     * @return new fileset for configuration
     **/
    public void addConfiguredFiles(FileSet fs)
    {
        require_(fs!=null,"addFiles- nonzro set");
        m_filerefs.add(fs);
    }



    /**
     * Adds a single file path element to be truncated.
     * @param path file name (non-null)
     **/
    public void addConfiguredFile(InnerString path)
    {
        require_(path!=null,"addFile- nonzro path");
        m_filerefs.add(getProject().resolveFile(path.toString(getProject())));
    }


    /**
     * Returns an iterator for all of the files to be emptied.
     * This is an expensive operation potentially because it must
     * determine matched files from nested filesets. Will return
     * <i>null</i> if no file has been specified explicitly. The
     * returned iterator returns <em>file paths</em>.
     **/
    public final Iterator getPaths()
    {
        if (m_filerefs.isEmpty()) {
            return null;
        }
        return new FileSetsIterator(m_filerefs,getProject());
    }



    /**
     * Ensures we've been given at least one file to truncate.
     * Whether or not the file is valid is determined by the
     * execute method.
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);

        if (m_filerefs.isEmpty()) {
            String error = getAntXMsg("task.needs.oneof.these.nested",
                                      getTaskName(),"files|file");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
    }



    /**
     * Ensures the named file is empty; will create a new empty file
     * if necessary.
     * @throws BuildException if unable to create/empty file and the
     *   'haltiferror' option is turned on
     **/
    public void execute()
    {
        verifyCanExecute_("exec");

        Iterator itr = getPaths();

        while (itr.hasNext()) {
            File f= new File(itr.next().toString());
            String problem = truncateFile(f);
            if (problem!=null) {
                throw new BuildException(problem,getLocation());
            }
        }
    }



    /**
     * Ensures the named file is empty; will create a new empty file
     * if necessary.
     * @return Error message if unable to create/empty file and the
     *   'haltiferror' option is turned on; otherwise returns <i>null</i>
     **/
    protected String truncateFile(File f)
    {
        boolean notThere= !f.exists();

        if (notThere) {
            if (getMustExist()) {
                String error = getAntXMsg("task.err.filenotfound",f.getPath());
                log(error,Project.MSG_ERR);
                return error;
            }
            else if (!willCreateIfMissing()) {
                return null;//=>done!
            }
        }
        try {
            if (!FeedbackLevel.isQuietish(m_fbLevel,true)) {
                log("Trying: "+f.getPath(), Project.MSG_VERBOSE);
            }
            if (notThere) {
                f.createNewFile();
            } else {
                if (f.length()!=0) {
                    FileOutputStream fos = new FileOutputStream(f);
                    fos.write(new byte[0]);//NB:truncate!
                    fos.close();
                }
            }
        }
        catch(Exception anyX) {
            String problem = getAntXMsg("mktemp.truncfile.cant",
                                        f.getPath(), anyX.getMessage());
            if (isHaltIfError()) {
                log(problem,Project.MSG_ERR);
                return problem;
            }
            log(problem, Project.MSG_WARN);
        }
        return null;
    }


    private FeedbackLevel m_fbLevel=FeedbackLevel.NORMAL;
    private boolean m_haltIfError;//no
    private boolean m_mustExist;//no
    private boolean m_createIfMissing;//no
    private List    m_filerefs;
}

/* end-of-TruncateFileTask.java */
