/**
 * $Id: TempLocator.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.parameters.RecoveryEnabled;

/**
 * Task that locates suitable scratch or temporary directories into which scratch files can
 * be safely dumped. Helper for other mktemp package classes.
 * <p>
 * <b>Examples:</b><pre>
 * &lt;<b>tempdir</b> haltiferror="no" subdirectory="&#46;antx-build"
          pathproperty="scratch&#46;root"/&gt;
 *
 * &lt;<b>tempdir</b> subdirectory="&#46;antx-builds" pathproperty="scratch&#46;root"/&gt;
 *   &lt;tstamp&gt;
 *      &lt;format property="itid" pattern="..."/&gt;
 *   &lt;/tstamp&gt;
 *   &lt;property name="debug&#46;root" value="${scratch&#46;root}/debug_${itid}"/&gt;
 *   &lt;property name="opt&#46;root" value="${scratch&#46;root}/opt_${itid}"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  guarded
 * @.group   impl,helper
 * @see      MkTempObject
 **/

public final class TempLocator extends AssertableTask 
    implements RecoveryEnabled, AntLibFriendly
{
    /**
     * Tries to determine the OS-specific temporary directory. Not as nice
     * as the JWare TempLocator but fixed to work with Ant's own tempfile
     * factory as a fallback.
     **/
    private static final File TMPDIR;
    static {
        File f=null;
        try {
            f = File.createTempFile("qat",null);
            f.deleteOnExit();
            f = f.getParentFile();
            f = f.getCanonicalFile();
        } catch(Exception anyX) {//hmm...this is bad...ok...make ant do it...
            FileUtils fu= FileUtils.getFileUtils();
            f = fu.createTempFile("qat",".tmp",null);
            try {
                f = f.getParentFile().getCanonicalFile();
            } catch (Exception ioX) {
                f = new File("_will_die_on_first_use");
            }
        }
        TMPDIR= f;
    }

    /**
     * Returns the OS-specific temp directory. Never returns <i>null</i>.
     **/
    public final static File getSystemTempDir()
    {
        return TMPDIR;//?expose
    }


    /**
     * Initializes a new TempLocator task.
     **/
    public TempLocator()
    {
        super(AntX.mktemp);

    }


    /**
     * Sets the property into which the temp directory's path is copied.
     * @param property the property's name (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setPathProperty(String property)
    {
        require_(property!=null,"setProp- nonzro name");
        m_updateProperty = property;
    }


    /**
     * Returns the name of the property into which the temp directory's
     * path will be stored. Returns <i>null</i> if never set.
     **/
    public final String getPathProperty()
    {
        return m_updateProperty;
    }



    /**
     * Sets the property into which a URL representation of the 
     * temp directory's path is copied.
     * @param urlproperty the property's name (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setUrlProperty(String urlproperty)
    {
        require_(urlproperty!=null,"setUrlProp- nonzro name");
        m_updateUrlProperty = urlproperty;
    }



    /**
     * Returns the name of the property into which a URL representation
     * of the temp directory's path will be stored. Returns <i>null</i>
     * if never set explicitly.
     * @since JWare/AntX 0.5
     **/
    public final String getUrlPathProperty()
    {
        return m_updateUrlProperty;
    }



    /**
     * Sets name of preferrred sub-directory within the OS-specific temp
     * directory. When executed, this templocator will try to create a
     * writable directory with this name in the temp directory.
     * @param directory the sub-directory's name (non-null)
     **/
    public void setSubdirectory(String directory)
    {
        require_(directory!=null, "setroot- nonzro dirname");
        m_unverifiedTmpDir = new File(getSystemTempDir(),directory);
    }


    /**
     * Returns the preferred sub-directory as a file. Whether this
     * sub-directory actually exists depends on whether this task has
     * been executed at least once. Returns <i>null</i> if a
     * preferred subdirectory has never been set, <em>or</em> if
     * the preferred subdirectory has been created and verified.
     * @see #getVerifiedRoot
     **/
    public File getUnverifiedRoot()
    {
        return m_unverifiedTmpDir;
    }


    /**
     * Returns the temporary directory created by this task. If
     * non-null then this directory has been verified by this task
     * (exists, is-a directory, and is writable).
     * @see #getUnverifiedRoot
     **/
    public File getVerifiedRoot()
    {
        return m_verifiedTmpDir;
    }


    /**
     * Sets whether this task will throw a build exception if it's
     * unable to create a preferred sub-directory. Defaults to
     * <i>true</i>.
     **/
    public void setHaltIfError(boolean haltIfError)
    {
        m_haltIfError = haltIfError;
    }


    /**
     * Returns <i>true</i> if this task will fail if it's unable to
     * create a preferred temp sub-directory. Defaults to <i>true</i>.
     * @see #setHaltIfError
     **/
    public boolean isHaltIfError()
    {
        return m_haltIfError;
    }


    /**
     * Locates the runtime platform's temporary directory and stores its
     * location in the specified {@linkplain #setPathProperty property}. If
     * a preferred temp sub-directory has been requested, this method will
     * try to ensure this directory exists and is read/writable (creating
     * it if it has to)-- the property is set to the appropriate
     * subdirectory's path.
     * @throws BuildException if fail-on-error and unable to create
     *         preferred sub-directory
     **/
    public void execute() throws BuildException
    {
        File rootDir = getSystemTempDir();

        if (getUnverifiedRoot()!=null) {
            File preferredRootDir=getUnverifiedRoot();
            try {
                preferredRootDir.mkdirs();
                if (!preferredRootDir.isDirectory() || !preferredRootDir.canWrite()) {
                    String ermsg = uistrs().get("mktemp.cant.make.tmpdir",
                            rootDir.getPath(),  preferredRootDir.getName());
                    log(ermsg, Project.MSG_ERR);
                    throw new BuildException(ermsg);
                }
                m_verifiedTmpDir = preferredRootDir;
                m_unverifiedTmpDir = null;
                rootDir = preferredRootDir;

            } catch(RuntimeException rtX) {
                if (isHaltIfError()) {
                    throw rtX;
                }//NB:ELSE-otherwise defaults back to systemTmpDir...
            }
        }
        else if (getVerifiedRoot()!=null) {
            rootDir = getVerifiedRoot();
        }

        if (getPathProperty()!=null) {
            getProject().setNewProperty(getPathProperty(),rootDir.getPath());
        }
        
        if (getUrlPathProperty()!=null) {
            String urlPath = AntXFixture.fileUtils().toURI(rootDir.getPath());
            getProject().setNewProperty(getUrlPathProperty(),urlPath);
        }
    }


    private String m_updateProperty;
    private String m_updateUrlProperty;
    private File m_verifiedTmpDir, m_unverifiedTmpDir;
    private boolean m_haltIfError=true;//NB:default barfs if cant make what user asked for
}

/* end-of-TempLocator.java */
