/**
 * $Id: MkTempObject.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;
import  java.io.IOException;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.helpers.Tk;

/**
 * A factory task for temporary files and directories. Used to aid with testing other
 * task implementations but is generally useful if your build process creates temporary
 * or intermediate files.
 * <p>
 * <b> Examples:</b><pre>
 *   &lt;mktmpdir prefix=".sql-" pathproperty="sql.root"/&gt;
 *
 *   &lt;mktmpfile prefix="sot" copyresource="antx/testfiles/broken-build.xml"
 *                 pathproperty="scratch.file0"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   api,helper
 * @see      TempLocator
 **/

public abstract class MkTempObject extends MkNewObject
{
    /** The default temp-object file prefix. **/
    protected static final String DEFAULT_PREFIX= "qat";


    /**
     * Initializes a new MkTempObject task.
     **/
    protected MkTempObject()
    {
        super(AntX.mktemp);
    }


    /**
     * Initializes a new CV-labeled MkTempObject task.
     **/
    protected MkTempObject(String iam)
    {
        super(iam);
    }


    /**
     * Returns the preferred (prefix) filler character. Used
     * when a prefix doesn't met minimum length requirements.
     * Should return a simple ASCII character to be safe.
     **/
    protected char getFillerChar()
    {
        return '_';
    }


    /**
     * Returns this class's default prefix if none specified.
     * Looks for build-supplied default in appropriate project
     * property; if none defined, uses {@linkplain #DEFAULT_PREFIX}.
     * @param forFile <i>true</i> if query is for a file
     **/
    protected final String getDefaultPrefix(boolean forFile)
    {
        String pfx = Tk.getTheProperty
            (getProject(), AntX.DEFAULT_TEMPOBJECT_PREFIX_PROP);

        return (pfx!=null) ? pfx : MkTempObject.DEFAULT_PREFIX;
    }


    /**
     * Sets a user-defined prefix for new temporary items.
     * This prefix might be adjusted to platform-required
     * rules if necessary (like minimum length).
     **/
    public void setPrefix(String prefix)
    {
        require_(prefix!=null,"setPfx- nonzro pfx");
        m_prefix = prefix;
    }


    /**
     * Returns the user-defined prefix; returns <i>null</i>
     * if never set.
     **/
    public String getPrefix()
    {
        return m_prefix;
    }


    /**
     * Returns the calculated prefix to-be used with a new
     * temporary item. The returned prefix mets all Java API
     * prerequisites for length, form, etc.
     * @param forFile <i>true</i> if query is for a file
     **/
    public final String getEffectivePrefix(boolean forFile)
    {
        String pfx = getPrefix();
        if (pfx!=null && pfx.length()<3) {
            char filler = getFillerChar();
            for (int i=3-pfx.length();i>0;i--) {
                pfx += filler;
            }
        }
        return (pfx!=null) ? pfx : getDefaultPrefix(forFile);
    }


    /**
     * Sets a user-defined suffix for new temporary items.
     * This suffix might be adjusted to platform-required
     * rules if necessary (like minimum length).
     **/
    public void setSuffix(String suffix)
    {
        require_(suffix!=null,"setSfx- nonzro sfx");
        m_suffix = suffix;
    }


    /**
     * Returns the user-defined suffix; returns <i>null</i>
     * if never set.
     **/
    public String getSuffix()
    {
        return m_suffix;
    }


    /**
     * Returns the calculated suffix to-be used with a new
     * temporary item. The returned suffix mets all Java API
     * prerequisites for length, form, etc.
     * @param forFile <i>true</i> if query is for a file
     **/
    public final String getEffectiveSuffix(boolean forFile)
    {
        String sfx = getSuffix();
        if (sfx!=null && forFile) {
            if (sfx.length()>0 && sfx.charAt(0)!='.') {
                sfx = "."+sfx;
            }
        }
        return (sfx!=null) ? sfx : (forFile ? ".tmp" : "");
    }


    /**
     * Sets the property to be created with the temporary
     * object's full path.
     * @param property property to be created (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setPathProperty(String property)
    {
        require_(property!=null,"setProp- nonzro nam");
        m_updateProperty = property;
    }


    /**
     * Returns the property created with the temporary item's
     * full path. Returns <i>null</i> if never set.
     **/
    public final String getPathProperty()
    {
        return m_updateProperty;
    }


    /**
     * Sets the variable to be updated with the temporary
     * object's full path.
     * @param variable variable to be updated (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setPathVariable(String variable)
    {
        require_(variable!=null,"setVar- nonzro nam");
        m_updateVar = variable;
    }


    /**
     * Returns the variable updated with the temporary item's
     * full path. Returns <i>null</i> if never set.
     * @since JWare/AntX 0.5
     **/
    public final String getPathVariable()
    {
        return m_updateVar;
    }


    /**
     * Sets the parent directory of any created temporary object. The
     * specified directory must be an existing, writable directory.
     * @see TempLocator
     **/
    public void setIn(File inDir)
    {
        require_(inDir!=null,"setIn- nonzro dir");
        require_(inDir.canWrite(), "setIn- existing writable directory");
        m_inDir= inDir;
    }


    /**
     * Returns the root directory into which all other temporary
     * objects (files and directories) created. Never returns
     * <i>null</i>; defaults the OS's default temporary directory.
     **/
    public File getInDir()
    {
        return m_inDir;
    }


    /**
     * Defines whether this task will mark all temporary objects
     * as delete-on-exit.
     * @param leaveOnExit <i>true</i> if items should not be deleted
     **/
    public void setPersist(boolean leaveOnExit)
    {
        m_autoDelete = !leaveOnExit;
    }


    /**
     * Returns <i>true</i> if this task will try to mark all
     * temporary objects (files and directories) as delete-on-exit.
     **/
    public boolean isAutoDelete()
    {
        return m_autoDelete;
    }


    /**
     * Grunt work of creating a new scratch file in specified location.
     * If a prototype source file/resource is defined, its contents
     * are copied to new file.
     * @param inDir new scratch file's parent directory (non-null)
     * @throws BuildException if unable to create parent directory or new file
     **/
    protected final File createFile(File inDir) throws BuildException
    {
        File newFile=null;
        FileUtils fsu= getFileUtils();

        try {
            //NB: Use Ant's temp file facilities which don't create actual files
            //    since would be wasteful to create a file if it was just going
            //    to be stomped on with a copy. Also is safer to use a single
            //    source of tempfile names to make sure chance for name-collisions
            //    is minimized (ssmc).
            newFile = fsu.createTempFile
                (getEffectivePrefix(true),getEffectiveSuffix(true),inDir);

            if (getPrototypePlainFile()!=null) {
                fsu.copyFile(getPrototypePlainFile(),newFile,
                             getCopyFilters(),false);
            }
            else if (getPrototypeResourceFile()!=null) {
                fsu.copyFile(getPrototypeResourceFile(),newFile,
                             getCopyFilters(),false);
            } else {
                newFile.createNewFile();
                copyPrototypeLines(newFile,false);
            }
        } catch(IOException iox) {
            throw new BuildException(iox, getLocation());
        }

        if (isAutoDelete()) {
            newFile.deleteOnExit();
        }
        return newFile;
    }


    /**
     * Grunt work of creating a directory (plus prototype entities)
     * in specified location.
     * @param inDir temp directory's location (non-null)
     * @return new directory's full path
     * @throws BuildException if unable to create or verify directory
     **/
    protected final File createDirectory(File inDir) throws BuildException
    {
        File newDir=null;
        FileUtils fsu= getFileUtils();

        try {
            newDir = fsu.createTempFile
                (getEffectivePrefix(false),getEffectiveSuffix(false),inDir);

            newDir.mkdir();

            if (!newDir.isDirectory() || !newDir.canWrite()) {
                String ermsg = uistrs().get
                    ("mktemp.cant.make.tmpdir",
                     inDir.getPath(), newDir.getName());
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }

            if (getPrototypePlainFile()!=null) {
                copyPrototypeFile(getPrototypePlainFile(), newDir);
            }
            else if (getPrototypeResourceFile()!=null) {
                copyPrototypeFile(getPrototypeResourceFile(), newDir);
            }

        } catch(IOException iox) {
            throw new BuildException(iox, getLocation());
        }
        return newDir;
    }



    /**
     * Ensure any update properties are set to new object's path information.
     * @since JWare/AntX 0.5
     */
    protected void saveFinalPath(File finalObject, boolean strict)
    {
        String finalPath = finalObject.getPath(); 
        if (getPathProperty()!=null) {
            checkIfProperty_(getPathProperty(),!strict);
            getProject().setNewProperty(getPathProperty(),finalPath);
        }
        if (getPathVariable()!=null) {
            ExportedProperties.set(getPathVariable(),finalPath);
        }

        super.saveFinalPath(finalObject,strict);
    }


    private String m_prefix = null;//NB: required for files
    private String m_suffix = null;//NB: will be made '.tmp'
    private String m_updateProperty;
    private String m_updateVar;
    private File m_inDir= TempLocator.getSystemTempDir();
    private boolean m_autoDelete=true;
}

/* end-of-MkTempObject.java */
