/**
 * $Id: MkTempFile.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.types.PropertySet;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.starters.StringItemListHandle;

/**
 * Instance of MkTempObject that only produces actual temporary files. Note temporary
 * files <em>are</em>  marked for automatic deletion when the VM exits. Unlike the
 * builtin Ant '<i>tempfile</i>' task, this command produces a real file-system entity.
 * Prototype lines (as specified with the &lt;line&gt; or &lt;lines&gt; elements) are
 * only applied to new empty files. If a prototype copy file (or resource) has been
 * defined, these lines are <em>not</em> included to the new file.
 * <p>
 * <b> Examples:</b><pre>
 *   &lt;<b>newtempfile</b> prefix="sot" copyresource="antx/testfiles/broken-build.xml"
 *                 pathproperty="scratch.file0"/&gt;
 *
 *   &lt;<b>newtempfile</b> prefix="corn-" suffix=".sql" copyresource="build/getcorn.sql"
 *                 pathproperty="corn.sql.file"&gt;
 *     &lt;filterset refid="corn.sql.copyfilters"/&gt;
 *   &lt;/newtempfile&gt;
 *
 *   &lt;<b>newtempfile</b> pathproperty="sillyfile"&gt;
 *     &lt;line value="hello"/&gt;
 *     &lt;line value="there"/&gt;
 *     &lt;line value="world"/&gt;
 *   &lt;/newtempfile&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 **/

public final class MkTempFile extends MkTempObject
{
    /**
     * Create a new mktempfile task.
     **/
    public MkTempFile()
    {
        super();
    }


    /**
     * Adds a new prototype line to include in new file.
     * @param line the line information (non-null)
     * @since JWare/AntX 0.3
     **/
    public void addLine(InnerString line)
    {
        require_(line!=null,"addLn- nonzro ln");
        getPrototypeLinesNoNull().add(line);
    }


    /**
     * Adds a reference to prototype lines to include in new file.
     * @param lines the line information (non-null)
     * @since JWare/AntX 0.3
     **/
    public void addLines(StringItemListHandle lines)
    {
        require_(lines!=null,"addLns- nonzro lns");
        getPrototypeLinesNoNull().add(lines);
    }


    /**
     * Shortcut for {@linkplain #addLines addLines()}. The
     * refered-to list is <em>added</em> to this task's prototype
     * lines.
     * @param ref the itemlist's reference (non-null)
     * @since JWare/AntX 0.3
     **/
    public final void setLines(Reference ref)
    {
        StringItemListHandle lines = new StringItemListHandle(ref);
        getPrototypeLinesNoNull().add(lines);
    }


    /**
     * Adds a new property set to include in new temporary file.
     * @param propertyset new propertyset line (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addConfiguredPropertySet(PropertySet propertyset)
    {
        require_(propertyset!=null,"addPropSet- nonzro pset");
        getPrototypeLinesNoNull().add(propertyset);
    }


    /**
     * Creates a new temporary file. If executed multiple times, will not
     * produce the same named file within a single VM instance. If either a prototype
     * source or  prototype lines were defined, the new file's contents will
     * match the prototype information.
     **/
    public void execute() throws BuildException
    {
        verifyInProject_("exec");

        File newFile = createFile(getInDir());

        saveFinalPath(newFile,true);
    }
}

/* end-of-MkTempFile.java */
