/**
 * $Id: ListDirTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DirectoryScanner;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.FileSet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TransformHelper;
import  com.idaremedia.antx.parameters.ValueTransform;

/**
 * Utility task that lets you copy a directory's listing into the current project
 * as a string list. You can list either the directory's immediate contents (the
 * default) or all of its contents including sub-directories (requires a filter).
 * The ListDirTask is useful when you need to iteratively process (identically) the
 * contents of a directory, but do not know before hand the contents of the directory.
 * <p/>
 * The ListDirTask supports a single <span class="src">FileSet</span> that describes
 * what it will return in its list. You can define this file set (under the
 * <span class="src">&lt;filter&gt;</span> element) like you would any other Ant
 * file set. Note that the shortcut filter parameters (like
 * <span class="src">onlyFiles</span>) are applied to the output of the filter if
 * any.
 * <p/>
 * Once you've loaded the listing into the project, you must use the AntX item list
 * value URIs to read the list or the AntX flow control tasks to iterate the list.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;<b>listdir</b> path="${basedir}" torefid="basedir.files" onlyfiles="yes"/&gt;
 *
 *    &lt;<b>listdir</b> path="file:///builds/${user.name}/.ini" torefid="ini.ls" &gt;
 *       &lt;filter defaultexcludes="no"&gt;
 *           &lt;include name="*.xml,in"/&gt;
 *        &lt;/filter&gt;
 *    &lt;/listdir&gt;
 *    &lt;callforeach i="file" items="ini.ls" functions="..."/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class ListDirTask extends AssertableTask
{
    /**
     * Initializes a new listdir task instance.
     **/
    public ListDirTask()
    {
        super(AntX.mktemp+"ListDirTask");
    }



    /**
     * Sets the directory being listed. If additional filtering
     * information is specified, this path is used as the base
     * directory for the associated <span class="src">FileSet</span>.
     * @param pathOrUrl the directory information (non-null)
     **/
    public void setPath(String pathOrUrl)
    {
        require_(pathOrUrl!=null,"setDir- nonzro path");
        m_path = new File(TransformHelper.apply(ValueTransform.OSPATH,
                          pathOrUrl, getProject()));
    }



    /**
     * Returns the directory being listed. Will return <i>null</i>
     * if path was not set explicitly.
     **/
    public final File getPath()
    {
        return m_path;
    }




    /**
     * Returns the directory that will be processed taking the
     * nested filter (FileSet) into account. This method looks
     * at the script supplied {@linkplain #getPath path}; if that
     * was not set, it looks at the basedir option for any nested
     * {@linkplain #createFilter filter}. Will return <i>null</i>
     * if neither of these options have been set.
     **/
    public final File getListDirectory()
    {
        if (m_path!=null) {
            return m_path;
        }
        if (m_fspec!=null) {
            return m_fspec.getDir(getProject());
        }
        return null;
    }



    /**
     * Sets the name of the string item list to which the listing
     * should be written. This reference must not exist.
     * @param refid the name of the (new) string list (non-null)
     **/
    public void setToRefId(String refid)
    {
        require_(!Tk.isWhitespace(refid), "setToRefId- nonzro name");
        m_outlistRefId = refid;
    }



    /**
     * Returns the name of the string item list created by this
     * task. Will return <i>null</i> if not set explicitly. This
     * attribute must be set before this task is executed.
     **/
    public final String getOutListRefId()
    {
        return m_outlistRefId;
    }


    /**
     * Tells this task to only include non-directories in this list.
     * Exactly what a "non-directory" includes depends on the Ant
     * <span class="src">FileSet</span> implementation.
     * @param onlyFiles <i>true</i> if only non-directories should be
     *          included.
     **/
    public void setOnlyFiles(boolean onlyFiles)
    {
        verifyNotDefined(m_onlyDirs);
        m_onlyFiles = onlyFiles;
    }



    /**
     * Tells this task to only include directories in this list.
     * @param onlyDirs <i>true</i> if only directories should be included.
     **/
    public void setOnlyDirs(boolean onlyDirs)
    {
        verifyNotDefined(m_onlyFiles);
        m_onlyDirs = onlyDirs;
    }



    /**
     * Tells this task to include only the item's base names in the
     * listing.
     * @param onlyBase <i>true</i> to only include base names.
     **/
    public void setBaseNamesOnly(boolean onlyBase)
    {
        m_onlyBasename = onlyBase;
    }



    /**
     * Returns <i>true</i> if only the items' base names will be
     * included in the saved list. Defaults <i>off</i>; the full,
     * absolute path is usually saved.
     **/
    public final boolean isBaseNamesOnly()
    {
        return m_onlyBasename;
    }



    /**
     * Returns the filter (<span class="src">FileSet</span>) used by
     * this task. Only one filter can be nested per list task instance.
     * @return the new file set (never <i>null</i>)
     * @throws BuildException if a filter file set already specified.
     **/
    public FileSet createFilter()
    {
        if (m_fspec!=null) {
            String error = getAntXMsg("task.one.specialtask",
                                getTaskName(), "filter");
            log(error, Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
        m_fspec = new FileSet();
        if (m_path!=null) {
            m_fspec.setDir(m_path);
        }
        return m_fspec;
    }


    /**
     * Ensures all of this listing's required attributes have been
     * properly defined. Also ensure the named directory exists and
     * is a readable directory.
     * @throws BuildException if unable to proceed in current state.
     */
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        final Project P = getProject();
        String refid = getOutListRefId();
        String error = null;

        // 1. No storage specified
        if (refid==null) {
            error = getAntXMsg("task.needs.this.attr",getTaskName(),"outlist");
        } else {
            // 2. Existing storage specified
            FixtureExaminer.checkIfReference(P,m_rqlink,refid,true);

            // 3. Directory not specified at all
            if (m_fspec==null && m_path==null) {
                error = getAntXMsg("mktemp.ls.needs.path",getTaskName());
            }
            else if (m_fspec!=null) {
                // 4. Directory not specified as part of filter fileset.
                File specDir = m_fspec.getDir(P);
                if (specDir==null) {
                    if (m_path==null) {
                        error = getAntXMsg("mktemp.ls.needs.path",getTaskName());
                    } else {
                        m_fspec.setDir(m_path);
                    }
                    // 5. Two different directories specified (?!)
                } else if (m_path!=null &&
                           !AntXFixture.fileUtils().fileNameEquals(m_path,specDir)) {
                    error = getAntXMsg("mktemp.ls.diff.paths",m_path.getPath(),
                                       specDir.getPath());
                }
            } else {
                if (m_fspec==null) {
                    createFilter().setDir(m_path);
                }
            }
        }
        // 6. Non-existing directory (or non-directory) specified.
        if (error==null) {
            File lsDir = getListDirectory();
            if (!lsDir.isDirectory()) {
                error = getAntXMsg("mktemp.ls.bad.dirpath",lsDir.getPath());
            }
        }
        if (error!=null) {
            log(error, Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
    }



    /**
     * Creates the string list containing the matched contents of the named
     * directory. Will create an empty list if no items match filter criteria.
     * The main directory and its parent directory ("." and "..") are
     * <em>not</em> included.
     * @throws BuildException if unable to execute in current state.
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        //NB: Limit to just the directory iff no other filter given!!
        if (!m_fspec.hasPatterns() && !m_fspec.hasSelectors()) {
            m_fspec.createInclude().setName("*");
        }

        DirectoryListing list = new DirectoryListing(m_onlyBasename);
        DirectoryScanner ds = m_fspec.getDirectoryScanner(getProject());

        if (m_onlyFiles || !m_onlyDirs) {
            list.addIncludes(ds.getIncludedFiles(),ds);
        }
        if (m_onlyDirs || !m_onlyFiles) {
            list.addIncludes(ds.getIncludedDirectories(),ds);
        }

        getProject().addReference(getOutListRefId(),list);
    }




    /**
     * Verifies not trying to list onlyFile and onlyDirs at the same time.
     **/
    private void verifyNotDefined(boolean other)
    {
        if (other) {
            String error = getAntXMsg("task.one.or.other.attr",
                "onlyfiles", "onlydirs");
            log(error, Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
    }


    private Requester m_rqlink = new Requester.ForComponent(this);
    private String m_outlistRefId;//NB:required
    private File m_path;//NB:required
    private FileSet m_fspec;
    private boolean m_onlyDirs, m_onlyFiles, m_onlyBasename;
}

/* end-of-ListDirTask.java */