/**
 * $Id: UISMBundle.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.init;

import  java.io.ByteArrayInputStream;
import  java.io.File;
import  java.io.IOException;
import  java.net.MalformedURLException;
import  java.net.URL;
import  java.util.Properties;
import  java.util.PropertyResourceBundle;

import  org.apache.tools.ant.AntClassLoader;
import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.util.ClasspathUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.CustomLoaderFactoryMethod;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.parameters.CustomLoaderEnabled;
import  com.idaremedia.antx.parameters.FlexExistenceEnabled;

/**
 * Maintains the definition of a properties-based resource bundle for a UIStringManager.
 * Usually defined &lt;msgsbundle&gt;. UISMBundles are also used by all tasks
 * that generate UIStringManagers from a UISMSources.
 * <p>
 * UISMBundles are usually defined as part of a build-iterations's fixture and
 * referred-to by target-based configuration tasks.
 * <p>
 * <b>Examples:</b><pre>
 *      &lt;msgsbundle id="compile.msgs"
 *             resource="resources/msgs/compile-msgs.properties"/&gt;
 *
 *      &lt;msgsbundle id="subbuild.msgs" mustexist="yes"
 *             file="${build.root}/${module.id}/build-msgs.properties"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @.caveat  For URL-based data files, assumes the Ant runtime is authorized on
 *           the targeted server(s).
 **/

public class UISMBundle extends AssertableDataType
    implements FixtureComponent, CustomLoaderEnabled, FlexExistenceEnabled,
               UISMSource, Cloneable, AntLibFriendly
{
    /**
     * Initializes a new UISMBundle instance. The new instance is equivalent
     * to an empty properties file.
     **/
    public UISMBundle()
    {
        super(AntX.uism);
    }


    /**
     * Initializes a new UISMBundle instance with its 'mustExist' option
     * predefined. Useful constructor when this item is embedded in another
     * object.
     * @see #setMustExist setMustExist(&#8230;)
     **/
    public UISMBundle(boolean mustExist)
    {
        super(AntX.uism);
        m_failIfMissing = mustExist;
    }


    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Tries to return an unique identifier for this bundle. Looks
     * for a build-script identifier({@linkplain #setId setId}); if none
     * found, creates a unique string based on type's class name and
     * instance reference.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getBundleRef().getId();
        }
        return super.getId();
    }


    /**
     * Tells this bundle it is part of another object's implementation.
     * This bundle will defer resource loading to the controlling object's
     * class loader if it exists.
     * @param cLFactory bundle controller (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setController(CustomLoaderFactoryMethod cLFactory)
    {
        m_cLFactory = cLFactory;
    }


    /**
     * Sets whether this bundle will fail if it cannot locate and
     * load its strings from the specified location.
     * @param mustExist <i>true</i> if resource bundle must be loaded
     **/
    public void setMustExist(boolean mustExist)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_failIfMissing = mustExist;
        edited();
    }


    /**
     * Returns <i>true</i> if the resource bundle specified for this item
     * must exist (or a default must be specified in system properties).
     * Defaults to <i>false</i>.
     **/
    public boolean getMustExist()
    {
        if (isReference()) {
            return getBundleRef().getMustExist();
        }
        return m_failIfMissing;
    }



    /**
     * Returns a new Properties object based on this bundle's properties
     * contents. Never returns <i>null</i>.
     * @param errH [optional] callback if load problem occurs
     * @throws BuildException if unable to load resource-bundle information.
     **/
    public synchronized Properties newProperties(ProblemHandler errH)
    {
        if  (isReference()) {
            return getBundleRef().newProperties(errH);
        }
        if (!triedLoadOnce()) {
            tryLoadSource(errH);
        }
        try {
            Properties p= new Properties();
            p.load(new ByteArrayInputStream(getBytes()));
            return p;
        } catch(IOException iox) {
            throw new BuildError(iox.getMessage());//NB:never
        }
    }




    /**
     * Returns a new PropertyResourceBundle based on this bundle's properties
     * contents. Never returns <i>null</i>.
     * @param errH [optional] callback if load problem occurs
     * @throws BuildException if unable to load resource-bundle information.
     **/
    public synchronized PropertyResourceBundle newPropertyBundle(ProblemHandler errH)
    {
        if  (isReference()) {
            PropertyResourceBundle rb = getBundleRef().newPropertyBundle(errH);

            if (!m_isInited) {//NB:mark as installed if 1st time and OK
                m_isInited=true;
            }
            return rb;
        }

        if (!triedLoadOnce()) {
            tryLoadSource(errH);
        }
        try {
            return new PropertyResourceBundle(new ByteArrayInputStream(getBytes()));
        } catch(IOException iox) {
            throw new BuildError(iox.getMessage());//NB:never
        }
    }


    /**
     * Returns a clone of this UISMBundle. Supports passing of UISMBundles between
     * project and sub-projects. If this bundle is a reference to another object,
     * that object's clone method is invoked.
     * @since JWare/AntX 0.3
     **/
    public Object clone()
    {
        if (isReference()) {
            return getBundleRef().clone();
        }
        try {
            UISMBundle copy = (UISMBundle)super.clone();
            if (m_file!=null) {
                copy.m_file = new File(m_file.getPath());
            }
            if (m_URL!=null) {
                copy.m_URL= new URL(m_URL.toString());
            }
            if (m_bytes.length>0) {
                byte[] newbytes = new byte[m_bytes.length];
                System.arraycopy(m_bytes,0,newbytes,0,m_bytes.length);
                copy.setBytes(newbytes);
            }
            return copy;
        } catch (MalformedURLException mfx) {
            throw new BuildException(mfx);
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }

// ---------------------------------------------------------------------------------------
// Datasources as URLs
// ---------------------------------------------------------------------------------------

    /**
     * Sets this holder's resource bundle's location as a URL.
     * @param urlstr URL string representation (non-null)
     * @throws BuildException if this bundle is reference or another
     *         bundle or if another source has already been specified.
     **/
    public void setURL(String urlstr)
    {
        require_(urlstr!=null,"setURL- nonzro url");
        if (isReference()) {
            throw tooManyAttributes();
        }
        checkNewDatasource();
        m_URLstr= urlstr;
        m_URL= null;
        edited(UISMTk.BY_URL);
    }



    /**
     * Returns this holder's resource bundle's URL location. Returns
     * <i>null</i> if never set or was set to an invalid URL string.
     **/
    public final URL getURL()
    {
        if  (isReference()) {
            return getBundleRef().getURL();
        }
        if (m_URL!=null) {
            return m_URL;
        }
        if (m_URLstr!=null) {
            try {
                m_URL = new URL(m_URLstr);
            } catch(MalformedURLException mfx) {
                log(uistrs().get("task.uism.err.bad.url",m_URLstr), Project.MSG_ERR);
            }
        }
        return m_URL;
    }


    /**
     * Opens URL connection, copies returned output to this bundle as raw
     * properties information. This method assumes URL's contents are a
     * standard Properties file.
     * @param url the url to be read (non-null)
     * @return <i>true</i> if read and stored successfully
     * @see InputFileLoader
     **/
    protected final boolean copyURLStreamToSelf(URL url)
    {
        boolean copied=false;
        try {
            setBytes(new InputFileLoader().loadURL(url));//NB:assumin iso8851-1
            log("Copied UISM URL("+url+") to bundle",Project.MSG_DEBUG);
            copied=true;
        } catch(IOException iox) {
            log(uistrs().get("task.uism.err.loading.url",url.toString()),
                Project.MSG_ERR);
        }
        return copied;
    }


    /**
     * Shared grunt work for specify-by-url or specifiy-by-resource.
     * @param url the url to be checked and/or loaded (non-null)
     **/
    protected final boolean handleThisURL(URL url)
    {
        File f = Tk.toFile(url);
        if (f==null) {
            return copyURLStreamToSelf(url);
        } else if (f.canRead()) {
            boolean OK = copyURLStreamToSelf(url);
            if (OK) {
                m_file = f;
            }
            return OK;
        } else {
            log(uistrs().get("task.uism.err.bad.file",f.getPath()),
                Project.MSG_ERR);
        }
        return false;
    }


    /**
     * Checks if this UISM bundle has been specified as a URL. If it has
     * tries to load the URL's raw contents to this bundle holder.
     * @return <i>true</i> if url was defined and was loaded.
     **/
    protected final boolean handleURL()
    {
        URL url = getURL();

        return (url!=null) ? handleThisURL(url) : false;
    }

// ---------------------------------------------------------------------------------------
// Datasources as Files
// ---------------------------------------------------------------------------------------

    /**
     * Sets this holder's resource bundle's location as a file.
     * @param fp the readable file path (non-null)
     * @throws BuildException if this bundle is reference or another
     *         bundle or if another source has already been specified.
     **/
    public void setFile(String fp)
    {
        require_(fp!=null,"setFil- nonzro filpath");
        if (isReference()) {
            throw tooManyAttributes();
        }
        checkNewDatasource();
        m_file= getProject().resolveFile(fp);
        edited(UISMTk.BY_FILE);
    }


    /**
     * Returns this holder's resource bundle's file location. Returns
     * <i>null</i> if never set (or determined from other datasource type).
     **/
    public final File getFile()
    {
        if  (isReference()) {
            return getBundleRef().getFile();
        }
        return m_file;
    }


    /**
     * Checks if this holder's resource bundle has been specified as a file.
     * If it has, tries to load the file's raw contents to this bundle holder.
     * @return <i>true</i> if file was defined and was loaded.
     **/
    protected final boolean handleFile()
    {
        File f = getFile();
        if (f!=null) {
            if (f.canRead()) {
                try {
                    URL furl = AntXFixture.fileUtils().getFileURL(f);
                    boolean OK = copyURLStreamToSelf(furl);
                    if (OK) {
                        m_URL = furl;
                    }
                    return OK;
                } catch(MalformedURLException mfx) {
                    log(uistrs().get("task.uism.err.bad.file",f.getPath()),
                        Project.MSG_ERR);
                }
            } else {
                log(uistrs().get("task.uism.err.bad.file",f.getPath()),
                    Project.MSG_ERR);
            }
        }
        return false;
    }

// ---------------------------------------------------------------------------------------
// Datasources as Resources
// ---------------------------------------------------------------------------------------

    /**
     * Creates and returns a new classpath element from this
     * bundle's custom resource search path. Returned path must be
     * configured by caller.
     **/
    public Path createClassPath()
    {
        return getCLSpi(true).createClasspath();
    }


    /**
     * Returns this bundle's custom classpath. Returns <i>null</i> if
     * never created.
     **/
    public final Path getClassPath()
    {
        if  (isReference()) {
            return getBundleRef().getClassPath();
        }
        if (m_CLspi!=null) {
            return m_CLspi.getClasspath();
        }
        return null;
    }


    /**
     * Adds a new classpath element to this bundle's custom classpath.
     * @param classpath additional classpath element (non-null)
     */
    public void setClassPath(Path classpath)
    {
        require_(classpath!=null,"setClzpath- nonzro path");
        getCLSpi(false).setClasspath(classpath);
    }


    /**
     * Adds a new classpath by-reference to this bundle's custom
     * classpath.
     * @param r reference to existing classpath item (non-null)
     */
    public void setClassPathRef(Reference r)
    {
        require_(r!=null,"setClzpathRef- nonzro ref");
        getCLSpi(false).setClasspathref(r);
    }


    /**
     * Tells this bundle to use an existing classloader to
     * search for and load resources. If loader does not exist,
     * and this bundle has a custom class path, a new class loader
     * will be stored under this reference's id.
     * @param r reference to an existing ClassLoader (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setLoaderRef(Reference r)
    {
        require_(r!=null,"setLodrRef- nonzro ref");
        getCLSpi(false).setLoaderRef(r);
    }


    /**
     * Returns this bundle's own loader identifier based on
     * its <em>current</em> configuration. Returns <i>null</i>
     * if never defined (directly or indirectly through a classpath
     * reference).
     * @since JWare/AntX 0.4
     **/
    public String getLoaderRefId()
    {
        if  (isReference()) {
            return getBundleRef().getLoaderRefId();
        }
        if (m_CLspi!=null) {
            return m_CLspi.getClassLoadId();
        }
        return null;
    }


    /**
     * Sets this holder's resource bundle's as a classpath-based
     * resource base name.
     **/
    public void setResource(String rsrc)
    {
        require_(rsrc!=null,"setSysRsrc- nonzro rsrc");
        if (isReference()) {
            throw tooManyAttributes();
        }
        checkNewDatasource();
        m_resource = rsrc;
        edited(UISMTk.BY_REZ);
    }


    /**
     * Returns this holder's resource bundle's as a classpath-based
     * resource base name. Returns <i>null</i> if never set.
     **/
    public final String getResource()
    {
        if  (isReference()) {
            return getBundleRef().getResource();
        }
        return m_resource;
    }


    /**
     * Shared grunt work for specify-by-resource.
     * @param rsrc the resource to be checked and/or loaded (non-null)
     **/
    protected final boolean handleThisResource(String rsrc)
    {
        boolean OK=false;

        AntClassLoader AcL=null;
        ClassLoader cL=null;
        URL url;
        Project P= getProject();

        if (m_cLFactory!=null) {
            cL = m_cLFactory.getClassLoader();
        }
        if (cL==null) {
            if (P!=null && m_CLspi!=null) {
                AcL = (AntClassLoader)m_CLspi.getClassLoader();
                cL = AcL;
            } else {
                cL = this.getClass().getClassLoader();
            }
        }
        if (cL==null) {
            url = LocalTk.getSystemResource(rsrc, P);//?ever?
        } else {
            url = cL.getResource(rsrc);
        }

        if (url!=null) {
            OK = handleThisURL(url);
            if (OK) {
                m_URL = url;
            }
        } else {
            log(uistrs().get("task.uism.err.bad.resource",rsrc),
                Project.MSG_ERR);
        }

        if (AcL!=null) {
            if (m_CLspi.getClassLoadId()==null) {
                AcL.cleanup();
            }
            AcL=null;
        }
        return OK;
    }


    /**
     * Checks if the UISM bundle has been specified as a class or system resource.
     * If it has, tries to load the resource's raw contents to this bundle holder.
     * @return <i>true</i> if resource defined, read, and loaded successfully.
     **/
    protected final boolean handleResource()
    {
        boolean OK=false;
        String rsrc = getResource();

        if (rsrc!=null) {
            OK = handleThisResource(rsrc);
        }

        return OK;
    }

// ---------------------------------------------------------------------------------------
// Loading Properties-Bundle to this in-memory holder:
// ---------------------------------------------------------------------------------------

    /**
     * Returns the UISMBundle referenced by this instance.
     * @throws BuildException is this object is not a reference.
     **/
    protected final UISMBundle getBundleRef()
    {
        return (UISMBundle)getCheckedRef(UISMBundle.class,"msgsbundle");
    }


    /**
     * Returns <i>true</i> if we've tried to load this bundle's raw
     * properties information at least once.
     **/
    protected final boolean triedLoadOnce()
    {
        return m_isInited;
    }


    /**
     * Returns this bundles custom class path helper. Never returns
     * <i>null</i>.
     * @.safety single
     * @since JWare/AntX 0.4
     **/
    private final ClasspathUtils.Delegate getCLSpi(boolean subel)
    {
        if (isReference()) {
            if (subel) {
                throw noChildrenAllowed();
            }
            throw tooManyAttributes();
        }
        if (m_CLspi==null) {
            m_CLspi= ClasspathUtils.getDelegate(this);
            edited();
        }
        return m_CLspi;
    }


    /**
     * Tries to load the raw properties resource bundle into this memory
     * holder. Loading order has following precedence:<ol>
     *  <li>File attribute</li>
     *  <li>Resource attribute</li>
     *  <li>URL attribute</li>
     *  <li>Resource named in AntX.DEFAUL_UISTRS_BUNDLE_PROP property</li>
     * </ol>
     **/
    private void tryLoadSource(ProblemHandler errH) throws BuildException
    {
        if (!m_isInited) {
            if (!handleFile() &&
                !handleResource() &&
                !handleURL()) {

                String defaultRsrc = null;
                Project P= getProject();
                if (P!=null) {
                    defaultRsrc = P.getProperty(AntX.DEFAULT_UISTRS_BUNDLE_PROP);
                }
                if (defaultRsrc==null) {
                    defaultRsrc = System.getProperty(AntX.DEFAULT_UISTRS_BUNDLE_PROP);
                }
                if (defaultRsrc!=null) {
                    m_isInited = handleThisResource(defaultRsrc);
                }
                if (!m_isInited) {
                    String ermsg = uistrs().get("task.uism.err.invalid.setup");
                    log(ermsg, Project.MSG_ERR);
                    if (errH!=null) {
                        errH.problem(ermsg, Project.MSG_ERR);
                    }
                    if (getMustExist()) {
                        throw new BuildException(ermsg);
                    }
                }
            }
            m_isInited=true;
        }
    }


    /**
     * Called whenever a source-attribute is defined; ensure at most
     * one source is defined.
     **/
    private void checkNewDatasource()
    {
        if (m_Nsources>0) {
            throw UISMTk.tooManySources();
        }
        m_Nsources++;
    }


    /**
     * Returns this bundle's raw properties definition information. Never
     * returns <i>null</i> but can be empty byte array if never loaded
     * or was unable to loade.
     **/
    private byte[] getBytes()
    {
        return m_bytes;
    }


    /**
     * Changes this bundle's raw properties definition information.
     * @.safety single
     **/
    protected final void setBytes(byte[] bytes)
    {
        require_(bytes!=null,"setBytes- nonzro byts");
        m_bytes = bytes;
    }


    private File m_file;
    private String m_URLstr;
    private URL m_URL;
    private String m_resource;
    private ClasspathUtils.Delegate m_CLspi;//NB:lazy-inited
    private boolean m_failIfMissing;//NB:=> empty bundle
    private CustomLoaderFactoryMethod m_cLFactory;//NB:=>standalone-type

    private boolean m_isInited;
    private byte[] m_bytes= new byte[0];//NB:=> like empty bundle!
    private int m_Nsources=0;//NB:number of sources defined (0..1)
    private String m_Id;
}

/* end-of-UISMBundle.java */
