/**
 * $Id: InitUISMTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.init;

import  java.io.File;
import  java.net.URL;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.AntlibDefinition;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.apis.BundleStringManager;
import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureInitializer;
import  com.idaremedia.antx.UISMContext;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.DefinitionLoader;
import  com.idaremedia.antx.parameters.CustomLoaderEnabled;

/**
 * Helper task that initializes a thread's default and root iteration UIStringManagers.
 * Usually used in a build's initialization task (outside any target) or with test scripts.
 * If undefined, this tasks will try to install a <em>default</em> string manager.
 * Below are some examples using InitUISMTask (assume '<i>managebundles</i>' is the
 * declared task name of the InitUISMTask class):<pre>
 *  &lt;managebundles url="http://buildmach.mycompany.com:9090/builds/antx/strings.properties"/&gt;
 *  &lt;managebundles resource="my/msgs.properties" classpathref="my.resources.cp"/&gt;
 *  &lt;managebundles action="install-default" bundleid="my.msgs"/&gt;
 *  &lt;managebundles resource="mycompany/tools/strings/AntX.strs" classpath="${my.classpath}"/&gt;
 *  &lt;managebundles action="install" file="/builds/conf/antx/subbuild-msgs.properties"/&gt;
 *  &lt;managebundles action="uninstall"/&gt;
 *  &lt;managebundles resource="my/msgs.properties" loaderref="my.resources.cpl"/&gt;
 * </pre>
 * <p>
 * Implementation note: This class is intentionally subclassed from the standard <i>Task</i>
 * and not AntX's <i>AssertableTask</i> to ensure UISM-based functionality doesn't apply to
 * instances of this class (publicly). You can also use this task in your own antlib files
 * to install a <em>default</em> message bundle like:<pre>
 *   &lt;antlib&gt;
 *     &lt;msgsdef resource="my/msgs.properties"/&gt;
 *     &#46;&#46;&#46;
 *   &lt;/antlib&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public final class InitUISMTask extends AntlibDefinition/*intentional*/
    implements UISMSource, CustomLoaderEnabled, AntLibFriendly, FixtureInitializer
{
    /**
     * {@linkplain #setAction Action} name to install the
     * <em>default</em> bundle.
     * @since JWare/AntX 0.4
     **/
    public static final String OP_INSTALL_DEFAULT   = "install-default";

    /**
     * {@linkplain #setAction Action} name to uninstall the
     * <em>default</em> bundle.
     * @since JWare/AntX 0.4
     **/
    public static final String OP_UNINSTALL_DEFAULT = "uninstall-default";

    /**
     * {@linkplain #setAction Action} name to install the
     * <em>root</em> bundle.
     * @since JWare/AntX 0.4
     **/
    public static final String OP_INSTALL_ROOT      = "install-root";

    /**
     * {@linkplain #setAction Action} name to uninstall the
     * <em>root</em> bundle.
     * @since JWare/AntX 0.4
     **/
    public static final String OP_UNINSTALL_ROOT    = "uninstall-root";



    /**
     * Initializes a new init UISM configuration task.
     **/
    public InitUISMTask()
    {
    }


    /**
     * Initializes this task's default problem notify callback and
     * ties its bundle impl to any antlib bits.
     * @since JWare/AntX 0.4
     **/
    public void init()
    {
        super.init();
        m_boobooResponse = new Requester.ForComponent(this);
        m_bundle.setController(new DefinitionLoader(this));
    }


    /**
     * Sets this task's project; updates other underlying
     * project-components.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_bundle.setProject(P);
    }



    /**
     * Sets whether this task will fail if it cannot locate and
     * load its strings from the specified location.
     * @param mustExist <i>true</i> if resource bundle must be loaded
     * @since JWare/AntX 0.4
     **/
    public void setMustExist(boolean mustExist)
    {
        m_bundle.setMustExist(mustExist);
    }



    /**
     * Sets this task's bundle source information as a reference
     * to an existing UISMBundle.
     **/
    public void setBundleId(Reference reference)
    {
        m_bundle.setRefid(reference);
    }


    /**
     * Sets this task's resource bundle's location as a URL. This
     * URL setting is checked after both the file and resource options
     * are checked.
     * @param url URL string representation (non-null)
     **/
    public void setURL(String url)
    {
        m_bundle.setURL(url);
    }


    /**
     * Returns this task's resource bundle's URL location. Returns
     * <i>null</i> if never set or was set to an invalid URL string.
     **/

    public URL getURL()
    {
        return m_bundle.getURL();
    }


    /**
     * Sets this task's resource bundle's location. This file setting
     * is checked first when this task tries to locate its UISM
     * bundle strings.
     **/
    public void setFile(String filepath)
    {
        m_bundle.setFile(filepath);
    }


    /**
     * Returns this task's resource bundle's file location. Returns
     * <i>null</i> if never set.
     **/
    public File getFile()
    {
        return m_bundle.getFile();
    }


    /**
     * Sets this task's resource bundle's base name. Resources are
     * checked <em>after</em> the {@linkplain #setFile file} property.
     **/
    public void setResource(String rsrc)
    {
        m_bundle.setResource(rsrc);
    }


    /**
     * Returns this task's resource bundle's base name. Returns
     * <i>null</i> if never set.
     **/
    public String getResource()
    {
        return m_bundle.getResource();
    }


    /**
     * Returns this task's custom classpath used when loading a class
     * or system resource. Returns <i>null</i> if never set.
     **/
    public Path getClassPath()
    {
        return m_bundle.getClassPath();
    }


    /**
     * Returns new (added) empty (nested) Path for use when searching
     * for resources.
     */
    public Path createClassPath()
    {
        return m_bundle.createClassPath();
    }


    /**
     * Adds new classpath element for use when searching for resources.
     * @param classpath the search path (non-null)
     */
    public void setClassPath(Path classpath)
    {
        m_bundle.setClassPath(classpath);
    }


    /**
     * Set the classpath by-reference for use when searching for
     * resources.
     * @param reference a Reference to a Path instance (non-null)
     */
    public void setClassPathRef(Reference reference)
    {
        m_bundle.setClassPathRef(reference);
    }


    /**
     * Tells this task to use an existing classloader to search
     * and/or load resources.
     * @param r reference to an existing ClassLoader (non-null)
     **/
    public void setLoaderRef(Reference r)
    {
        m_bundle.setLoaderRef(r);
    }


    /**
     * Returns this task's custom class loader identifier. Will
     * return <i>null</i> if never set.
     **/
    public String getLoaderRefId()
    {
        return m_bundle.getLoaderRefId();
    }


    /**
     * Instructs this task how to manage the iteration's UISM
     * context.
     * @param op the operation
     **/
    public void setAction(String op)
    {
        m_Op = Tk.lowercaseFrom(op);
    }


    /**
     * Factory method for our UIStringManager.
     **/
    private UIStringManager getUISM()
    {
        return new BundleStringManager
            (m_bundle.newPropertyBundle(m_boobooResponse),null);
    }


    /**
     * Installs a root UIStringManager (based on this bundle) as the catch-all
     * for UISMContexts, AssertableTasks, etc. within this thread . Should be done
     * at most once from an 'init' type task.
     **/
    private void installRootSM() throws BuildException
    {
        UIStringManager alreadyInstalled = UISMContext.installStringManager(getUISM(),null);
        if (alreadyInstalled!=null) {//Whups!
            UISMContext.unwindStringManager(null);
            String ermsg = AntX.uistrs().get("task.uism.err.too.many.root.bundles");
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        log("Installed root UIStringManager from URL="+getURL(),Project.MSG_DEBUG);
    }


    /**
     * Installs a <em>default</em> UIStringManager (based on this bundle)
     * as the catch-all for UISMContexts, AssertableTasks, etc.
     * @since JWare/AntX 0.4
     **/
    private void installDefaultSM()
    {
        boolean alreadyInstalled = !UISMContext.isDefaultUndefined();
        UISMContext.setDefaultStringManager(getUISM());
        if (alreadyInstalled) {
            String ermsg = AntX.uistrs().get("task.uism.warn.repl.dflt.bundle");
            log(ermsg, Project.MSG_WARN);
        } else {
            log("Installed default UIStringManager from URL="+getURL(),Project.MSG_DEBUG);
        }
    }



    /**
     * Executes this UISM manager's instruction; by default installs a default
     * thread-specific UIStringManager.
     * @throws BuildException if a root UIStringManager already installed for
     *         current thread or unable to load strings
     **/
    public void execute()
        throws BuildException
    {
        //If as Antlib definition, only install operations are permitted!
        if (getAntlibClassLoader()!=null) {//=>in Antlib always?
            if (m_Op!=null && m_Op.startsWith("uninstall")) {
                String ermsg = AntX.uistrs().get
                    ("task.manager.err.illegal.operation",m_Op);
                log(ermsg,Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
        }

        //Install|Uninstall UI string managers
        synchronized(m_bundle) {
            if (m_Op==null || OP_INSTALL_DEFAULT.equals(m_Op)) {
                installDefaultSM();
            }
            else if (OP_INSTALL_ROOT.equals(m_Op)   ||
                     Strings.INSTALL.equals(m_Op)/*backward-compat*/) {
                installRootSM();
            }
            else if (OP_UNINSTALL_ROOT.equals(m_Op) ||
                     Strings.UNINSTALL.equals(m_Op)/*backward-compat*/) {
                log("Trying to uninstall the active root UIStringManager",
                    Project.MSG_DEBUG);
                UISMContext.unwindStringManager(m_boobooResponse);
            }
            else if (OP_UNINSTALL_DEFAULT.equals(m_Op)) {
                if (!UISMContext.isDefaultUndefined()) {
                    log("Trying to uninstall the default UIStringManager",
                        Project.MSG_DEBUG);
                    AntXFixture.reset(UISMContext.FXID,Strings.DEFAULT,m_boobooResponse);
                }
            }
            else {
                String ermsg = AntX.uistrs().get
                    ("task.manager.err.unknown.operation",m_Op);
                log(ermsg,Project.MSG_WARN);
            }
        }
    }

    private UISMBundle m_bundle= new UISMBundle(true);//NB:delegatee for configuration
    private String m_Op;//NB:==> install-default
    private ProblemHandler m_boobooResponse = Responses.ERROR;
}

/* end-of-InitUISMTask.java */
