/**
 * $Id: DateTimeFormat.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2002,2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.text.FieldPosition;
import  java.text.Format;
import  java.text.SimpleDateFormat;
import  java.util.Date;
import  java.util.TimeZone;

/**
 * Helper for formatting time/dates in standard package format. Generated strings are
 * locale-specific.
 *
 * @since    JWare/core 0.5
 * @author   ssmc, &copy;1997-2002,2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 **/

public final class DateTimeFormat
{
    private DateTimeFormat() {/*prevent*/}

    /** Abbrev date/time format (not safe for concurrent use). **/
    public static final SimpleDateFormat ABBREV = new SimpleDateFormat("dd-MMM h:mm a");

    /** Abbrev date format (not safe for concurrent use). **/
    public static final SimpleDateFormat ABBREV_DATE = new SimpleDateFormat("dd-MMM");

    /** Abbrev time format (not safe for concurrent use).
     *  @since JWare/AntX 0.5
     **/
    public static final SimpleDateFormat ABBREV_TIME = new SimpleDateFormat("h:mm a");

    /** Standard date/time format (not safe for concurrent use). **/
    public static final SimpleDateFormat STANDARD = new SimpleDateFormat("h:mm:ss a dd-MMM-yyyy");

    /** Standard date format (not safe for concurrent use). **/
    public static final SimpleDateFormat STANDARD_DATE = new SimpleDateFormat("dd-MMM-yyyy");

    /** Standard time format (not safe for concurrent use). 
     *  @since JWare/AntX 0.5
     **/
    public static final SimpleDateFormat STANDARD_TIME = new SimpleDateFormat("h:mm:ss a");

    /** Standard date format for CVS changelog.
     *  @since JWare/AntX 0.5 
     **/
    public static final SimpleDateFormat CHANGELOG = new SimpleDateFormat("dd MMM yyyy");

    /** GMT-based date/time format (not safe for concurrent use). **/
    public static final SimpleDateFormat GMT = new SimpleDateFormat("k:mm:ss dd-MMM-yyyy 'GMT'");

    /** GMT-based date format (not safe for concurrent use). **/
    public static final SimpleDateFormat GMT_DATE = new SimpleDateFormat("dd-MMM-yyyy 'GMT'");

    /** GMT-based time format (not safe for concurrent use). **/
    public static final SimpleDateFormat GMT_TIME = new SimpleDateFormat("k:mm:ss 'GMT'");

    static {
        TimeZone gmtz = TimeZone.getTimeZone("GMT");
        GMT.setTimeZone(gmtz);
        GMT_DATE.setTimeZone(gmtz);
        GMT_TIME.setTimeZone(gmtz);
    }


    /** Converts timestamp into new formatted string; protected against
        concurrent use.
        @param tm the timestamp to be formatted
        @param dfmt the date, time, duration formatter to use (non-null)
        @throws IllegalArgumentException if formatter is null.
        @since JWare/AntX 0.5
     **/
    public static String format(long tm, Format dfmt) {
        if (dfmt==null) {
            throw new IllegalArgumentException();
        }
        synchronized(dfmt) {
            sm_adhoc.setTime(tm);
            return dfmt.format(sm_adhoc);
        }
    }


    /** Converts timestamp into new standard formatted string; protected against
        concurrent use. **/
    public static String format(long tm) {
        synchronized(STANDARD) {
            sm_standard.setTime(tm);
            return STANDARD.format(sm_standard);
        }
    }

    /** Updates existing buffer with timestamp as standard formatted string;
        protected against concurrent use. **/
    public static StringBuffer format(long tm, StringBuffer sb) {
        synchronized(STANDARD) {
            sm_standard.setTime(tm);
            return STANDARD.format(sm_standard,sb,new FieldPosition(0));
        }
    }

    /** Converts timestamp into new abbreviated formatted string; protected
        against concurrent use. **/
    public static String shortformat(long tm) {
        synchronized(ABBREV) {
            sm_abbrev.setTime(tm);
            return ABBREV.format(sm_abbrev);
        }
    }

    /** Updates existing buffer with timestamp as abbreviated formatted string;
        protected against concurrent use. **/
    public static StringBuffer shortformat(long tm, StringBuffer sb) {
        synchronized(ABBREV) {
            sm_abbrev.setTime(tm);
            return ABBREV.format(sm_abbrev,sb,new FieldPosition(0));
        }
    }

    /** Converts timestamp into new GMT formatted string; protected
        against concurrent use. **/
    public static String GMTformat(long tm) {
        synchronized(GMT) {
            sm_GMT.setTime(tm);
            return GMT.format(sm_GMT);
        }
    }

    /** Updates existing buffer with timestamp as GMT formatted string;
        protected against concurrent use. **/
    public static StringBuffer GMTformat(long tm, StringBuffer sb) {
        synchronized(GMT) {
            sm_GMT.setTime(tm);
            return GMT.format(sm_GMT,sb,new FieldPosition(0));
        }
    }

    /** Updates existing buffer with date as abbreviated formatted string;
        protected against concurrent use. 
        @since JWare/AntX 0.4
     **/
    public static StringBuffer shortdate(long tm, StringBuffer sb) {
        synchronized(ABBREV_DATE) {
            Date dt = new Date(tm);
            return ABBREV_DATE.format(dt,sb,new FieldPosition(0));
        }
    }

    /** Converts timestamp into new short date formatted string; protected
        against concurrent use.
        @since JWare/AntX 0.4
     **/
    public static String shortdate(long tm) {
        synchronized(ABBREV_DATE) {
            Date dt = new Date(tm);
            return ABBREV_DATE.format(dt);
        }
    }



    private static final Date sm_abbrev   = new Date();
    private static final Date sm_standard = new Date();
    private static final Date sm_GMT      = new Date();
    private static final Date sm_adhoc    = new Date();
}

/* end-of-DateTimeFormat.java */
