/**
 * $Id: Iff.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.go;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Implementation of the general <em>if</em> and <em>ifTrue</em> test for all
 * conditional components.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @see      Go
 * @see      IffAll
 **/

public final class Iff
{
    /**
     * Tests whether or not the given property name resolves to
     * an existing project property. The empty string always
     * returns <i>true</i>.
     * @param property the property name (non-null)
     * @param P the project to check against (non-null)
     * @return <i>true</i> if the property is defined
     **/
    public static final boolean allowed(String property, Project P)
    {
        return allowed(property,P,false);
    }



    /**
     * Tests whether or not the given property name resolves to
     * a project property with a positive boolean value. The empty
     * string always returns <i>true</i>.
     * @param property the property to test (non-null)
     * @param P the project to check against (non-null)
     * @param onlyIfTrue <i>true</i> if property must be defined
     *        to a positive boolean string (like "yes")
     * @return <i>true</i> if the property is defined to a
     *         positive boolean string
     **/
    public static final boolean allowed(String property, Project P,
                                        boolean onlyIfTrue)
    {
        if (property==null || "".equals(property)) {
            return true;
        }
        String test = Tk.resolveString(P,property);
        if (onlyIfTrue) {
            return Tk.booleanFrom(P.getProperty(test));//allow on,yes,true
        }
        return P.getProperty(test)!=null;
    }



    /**
     * Execute test for a simple "if" conditional parameter. Null
     * property names are not allowed.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     **/
    public static final class Exists extends Go.TestSkeleton {
        public Exists() {
        }
        public Exists(String property) {
            super(property);
        }
        public boolean pass(Project P) {
            verifyInited();
            return Iff.allowed(getParameter(),P,false);
        }
        public String getParameterName() {
            return "if";
        }
    }


    /**
     * Execute test for a simple "if-true" conditional parameter.
     * Null property names are not allowed.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     **/
    public static final class IsTrue extends Go.TestSkeleton {
        public IsTrue() {
        }
        public IsTrue(String list) {
            super(list);
        }
        public boolean pass(Project P) {
            verifyInited();
            return Iff.allowed(getParameter(),P,true);
        }
        public String getParameterName() {
            return "ifTrue";
        }
    }



    /**
     * Execute test for a simple "unless" conditional parameter.
     * Null property names are not allowed.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     * @see      Unless
     **/
    public static final class NotExists extends Go.TestSkeleton {
        public NotExists() {
        }
        public NotExists(String property) {
            super(property);
        }
        public boolean pass(Project P) {
            verifyInited();
            return Unless.allowed(getParameter(),P,false);
        }
        public String getParameterName() {
            return "unless";
        }
    }


    /**
     * Execute test for a simple "unless-true" conditional
     * parameter. Null property names are not allowed.
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     * @.group   impl,helper
     * @see      Unless
     **/
    public static final class IsNotTrue extends Go.TestSkeleton {
        public IsNotTrue() {
        }
        public IsNotTrue(String list) {
            super(list);
        }
        public boolean pass(Project P) {
            verifyInited();
            return Unless.allowed(getParameter(),P,true);
        }
        public String getParameterName() {
            return "unlessTrue";
        }
    }


    /** Don't allow **/
    private Iff()
    {
    }
}

/* end-of-Iff.java */
