/**
 * $Id: Go.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.go;

import  java.util.Collections;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.Project;
import  com.idaremedia.apis.FixtureStarter;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Namespace for the collection of simple tests used by conditionally executed
 * AntX components.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @.pattern GoF.Strategy
 * @see      com.idaremedia.antx.parameters.FlexConditional FlexConditional
 **/

public final class Go
{
    /**
     * Single execution criterion test for a conditionally executed
     * Ant component. "Go" tests typically customize a generic, criteria
     * decision strategy to a specific Ant task or application request.
     *
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  n/a
     * @.group   impl,infra
     * @see      Go.TestList
     **/
    public interface Test
    {
        /**
         * Returns <i>true</i> if test is passed and component can run.
         * @param P project from which fixture information should
         *          be read (non-null).
         **/
        boolean pass(Project P);

        /**
         * Returns this test's parameter string. Will return
         * <i>null</i> if never defined.
         **/
        String getParameter();

        /**
         * Returns this test's parameter's public-facing name.
         * Will return the empty string if never defined.
         **/
        String getParameterName();
    }


    /**
     * Starter implementation of an instance-configurable
     * execute ("go") test.
     *
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  multiple
     * @.pattern GoF.Strategy
     * @.group   impl,helper
     **/
    public static abstract class TestSkeleton implements Test
    {
        /**
         * Initializes new test. Test's parameter must
         * be specified.
         * @see #setParameter setParameter
         **/
        protected TestSkeleton() {
        }

        /**
         * Initializes new test with parameter.
         * @param param test's parameter (non-null)
         **/
        protected TestSkeleton(String param) {
            setParameter(param);
        }

        /**
         * Initializes this test's parameter string. Should
         * be done before first call to {@linkplain #pass pass()}.
         **/
        public final void setParameter(String parameter) {
            m_param = parameter;
        }

        /**
         * Returns this test's parameter string. Will return
         * <i>null</i> if never defined.
         **/
        public final String getParameter() {
            return m_param;
        }

        /* @no.javadoc
         * Returns leaf name of this class. Never <i>null</i>.
         */
        public String getParameterName() {
            return Tk.lowercaseFrom(Tk.leafNameFrom(getClass()));
        }

        /**
         * Ensures this test's parameter string is defined.
         * @throws IllegalStateException if not fully defined.
         **/
        protected void verifyInited() {
            if (getParameter()==null) {
                throw new IllegalStateException();
            }
        }

        /**
         * Returns a simple diagnostics string like=
         * "<span class="src">parametername=parameter</span>" or
         * (for example) "<span class="src">ifAll='p0,p2,p3'</span>".
         **/
        public String toString() {
            return getParameterName()+"='"+getParameter()+"'";
        }

        private String m_param;
    }


    /**
     * A simple sequence of execution ("go") tests.
     *
     * @since    JWare/AntX 0.4
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  special (depends on factory method used)
     * @.group   impl,helper
     **/
    public static final class TestList
    {
        /**
         * Factory method for new unsynchronized list of
         * default capacity (4 slots).
         **/
        public static TestList newList() {
            return new TestList(false,4);
        }

        /**
         * Factory method for new unsynchronized list of
         * custom capacity.
         **/
        public static TestList newList(int iniCapacity) {
            return new TestList(false,iniCapacity);
        }

        /**
         * Factory method for new synchronized list of
         * default capacity (4 slots).
         **/
        public static TestList newSynchronizedList() {
            return new TestList(true,4);
        }

        /**
         * Factory method for new synchronized list of
         * custom capacity.
         **/
        public static TestList newSynchronizedList(int iniCapacity) {
            return new TestList(true,iniCapacity);
        }

        /**
         * Hidden constructor. Available through factory methods.
         **/
        TestList(boolean mtsafe, int iniCapacity) {
            if (mtsafe) {
                m_tests = FixtureStarter.newSynchronizedList(iniCapacity);
            } else {
                m_tests = FixtureStarter.newList(iniCapacity);
            }
        }

        /**
         * Returns <i>true</i> if this list is empty.
         **/
        public boolean isEmpty() {
            return m_tests.isEmpty();
        }

        /**
         * Appends a new go test to this list. New test is
         * not checked for duplication.
         * @param t new test (non-null)
         **/
        public void add(Go.Test t) {
            if (t==null) {
                throw new IllegalArgumentException();
            }
            m_tests.add(t);
        }

        /**
         * Appends an existing list of tests to this list.
         * New tests are not checked for duplication.
         * @param l list of tests (non-null)
         **/
        public void add(TestList l) {
            if (l==null) {
                throw new IllegalArgumentException();
            }
            if (!l.isEmpty()) {
                m_tests.addAll(l.m_tests);
            }
        }

        /**
         * Removes all tests from this list. On return this
         * list is {@linkplain #isEmpty empty}.
         **/
        public void clear() {
            m_tests.clear();
        }

        /**
         * Returns the number of tests in this list.
         **/
        public int size() {
            return m_tests.size();
        }

        /**
         * Returns the index'th test.
         **/
        public Go.Test get(int i) {
            return (Go.Test)m_tests.get(i);
        }

        /**
         * Returns the first test of specified class in
         * this list. Returns <i>null</i> if no match.
         * @param c class to test for (non-null)
         **/
        public Go.Test get(Class c)
        {
            if (c==null) {
                throw new IllegalArgumentException();
            }
            synchronized(m_tests) {
                for (int i=0,N=size();i<N;i++) {
                    Go.Test t= (Go.Test)m_tests.get(i);
                    if (c.isInstance(t)) {
                        return t;
                    }
                }
            }
            return null;
        }


        /**
         * Returns <i>true</i> if this list contains at
         * least one test of the given class.
         * @param c class to test for (non-null)
         **/
        public boolean contains(Class c)
        {
            return get(c)!=null;
        }

        /**
         * Returns a <em>readonly</em> ordered iterator for
         * this list. Items are returned in the order they
         * were added.
         **/
        public Iterator iterator()
        {
            return Collections.unmodifiableList(m_tests).iterator();
        }

        /**
         * Returns <i>true</i> if this list is equivalent
         * to another list.
         * @param o object to check
         **/
        public boolean equals(Object o) {
            if (o==this) {
                return true;
            }
            if (o==null) {
                return false;
            }
            if (o.getClass()==TestList.class) {
                return m_tests.equals(((TestList)o).m_tests);
            }
            return false;
        }

        /**
         * Returns a hash value for this list. Returned
         * value depends on this list's current tests.
         **/
        public int hashCode()
        {
            return m_tests.hashCode();
        }

        private final List m_tests;
    }


    /**
     * Prevent; only strategies public.
     **/
    private Go()
    {
    }
}


/* end-of-Go.java */
