/**
 * $Id: TolerantTaskSet.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2003-2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.wrap;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.FixtureOverlay;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.parameters.RecoveryEnabled;
import  com.idaremedia.antx.starters.StrictOuterTask;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Taskset that captures any build exception thrown by nested tasks and gives the build
 * script an opportunity to cleanup after process-aborting build errors. Like Java-style
 * try-catch-finally construct.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class TolerantTaskSet extends TaskSet
    implements StrictOuterTask, RecoveryEnabled, FixtureOverlay
{
    /**
     * Initializes new tolerant taskset instance.
     **/
    protected TolerantTaskSet(String iam)
    {
        super(iam);
    }


    /**
     * Initializes new CV-labeled tolerant taskset subclass instance.
     **/
    protected TolerantTaskSet(String iam, boolean delayConfiguration)
    {
        super(iam, delayConfiguration);
    }


    /**
     * Sets whether this taskset will continue to unwind the call
     * stack and eventually fail with a top-level build exception.
     * Defaults to <i>true</i>.
     **/
    public void setHaltIfError(boolean balk)
    {
        m_haltIfError = balk;
    }


    /**
     * Returns <i>true</i> if this tolerant taskset will propagate
     * any build exceptions unconditionally. <i>true</i> by default.
     **/
    public final boolean isHaltIfError()
    {
        return m_haltIfError;
    }


    /**
     * Tells this task whether <em>all</em> types of runtime
     * exceptions should be captured. Normally on build exceptions
     * are captured.
     * @param allRTX <i>true</i> for all types of runtime exceptions.
     * @since JWare/AntX 0.5
     **/
    public void setAgainstAll(boolean allRTX)
    {
        m_allRTX = allRTX;
    }



    /**
     * Returns <i>true</i> if all types of <em>runtime</em>
     * exceptions will be captured and processed. Defaults to
     * <i>false</i>.
     * @since JWare/AntX 0.5
     **/
    public final boolean willCaptureAllRuntimeExceptions()
    {
        return m_allRTX;
    }



    /**
     * Ensure the iferror and always helpers tasks are configured
     * at same time as this task.
     **/
    protected void maybeConfigureSpecialTasks()
    {
        if (m_iferrorTask!=null) {
            m_iferrorTask.maybeConfigure();
        }
        if (m_alwaysTask!=null) {
            m_alwaysTask.maybeConfigure();
        }
    }


    /**
     * Adds a task to this tolerant taskset. If the task is one
     * of the special helper tasks (<i>iferror</i> or <i>always</i>)
     * it is handled specially. A tolerant taskset can have at most
     * one of each special task. A template method that first
     * extracts the actual task (stripping away any placeholders),
     * calls {@linkplain #addSpecialTask addSpecialTask}, and
     * finally calls the inherited add method if task was not special.
     * @param task new task to be added (non-null)
     **/
    public void addTask(Task task)
    {
        require_(task!=null,"addTsk- nonzro task");

        task = TaskExaminer.trueTask(task,getCOI(),this);//zap proxies!

        if (!addSpecialTask(task)) {
            super.addTask(task);
        } else {
            addedTask();
        }
    }


    /**
     * This taskset's special task inclusion filter method. Subclasses
     * should extend this method to add custom processing for additional
     * special tasks.
     * @param task the actual task to be added (already examined)
     * @return <i>true</i> if the task was processed as a special task
     * @since JWare/AntX 0.4
     **/
    protected boolean addSpecialTask(Task task)
    {
        boolean wasSpecial=true;

        if (task instanceof IfErrorTask) {
            addTheIfErrorTaskset((IfErrorTask)task);
        } else if (task instanceof AlwaysTask) {
            addTheAlwaysTaskset((AlwaysTask)task);
        } else {
            wasSpecial = false;
        }
        return wasSpecial;
    }


    /**
     * Ensures this taskset's nested tasks are executed with proper
     * try-catch-finally protection. If neither an <i>iferror</i>
     * nor a <i>always</i> task has been defined, this performs basically
     * as if the tolerant wrapper didn't exist-- build errors will
     * unwind the call stack until Ant exits the build process.
     **/
    protected void performNestedTasks()
    {
        try {
            //perform nested tasks that can barf...
            performTheProtectedTasksList();

        } catch(RuntimeException rtX) {
            //AntX 1.6+ is just wrong to do this! (ssmc)
            if (rtX instanceof BuildException && (rtX.getCause() instanceof Error)) {
                throw (Error)rtX.getCause();
            }

            //iff allow all types thru do we do generic processing
            if (!willCaptureAllRuntimeExceptions() && 
                !(rtX instanceof BuildException)) {
                    throw rtX;
            }

            RuntimeException rethrowX;

            //let iferror handler do its work!
            if (m_iferrorTask!=null) {
                rethrowX = m_iferrorTask.throwOccured(rtX);
                m_iferrorTask.perform();
            } else {
                //ensure something is recorded
                rethrowX = recordThrowOccured(rtX);
            }
            //continue the unwind by default, else breakout
            if (isHaltIfError()) {
                if (rethrowX!=rtX) {
                    rethrowX.fillInStackTrace();
                }
                throw rethrowX;
            }
            log("'"+getTaskName()+"' consuming build exception",
                Project.MSG_VERBOSE);
        } finally {
            //do things that must always be done failure|ok
            if (m_alwaysTask!=null) {
                m_alwaysTask.perform();
            }
        }
    }


    /**
     * Called to execute this taskset's nested tasks from the protected
     * try/catch/finally {@linkplain #performNestedTasks performNestedTasks}
     * method. By default just calls the inherited 'performTheTasksList'.
     * Subclasses should override this method to customize what tasks
     * are performed. The taskset's execute conditions have already been
     * evaluated and determined to be <i>true</i>.
     **/
    protected void performTheProtectedTasksList()
        throws BuildException
    {
        performTheTasksList();
    }



    /**
     * Returns this taskset's error handling helper task. Can return
     * <i>null</i> if never defined.
     **/
    protected final IfErrorTask getIfErrorTaskset()
    {
        return m_iferrorTask;
    }


    /**
     * Returns this taskset's always helper task. Can return
     * <i>null</i> if never defined.
     **/
    protected final AlwaysTask getAlwaysTaskset()
    {
        return m_alwaysTask;
    }


    /**
     * Initializes this taskset's <i>iferror</i> special task.
     **/
    protected final void addTheIfErrorTaskset(IfErrorTask task)
    {
        verifyIsUndefined_(m_iferrorTask, task);
        task.setEnclosingTask(this);
        m_iferrorTask = task;
        if (task.getProject()==null) {
            task.setProject(getProject());
        }
    }


    /**
     * Initializes this taskset's <i>always</i> special task.
     **/
    protected final void addTheAlwaysTaskset(AlwaysTask task)
    {
        verifyIsUndefined_(m_alwaysTask, task);
        task.setEnclosingTask(this);
        m_alwaysTask = task;
        if (task.getProject()==null) {
            task.setProject(getProject());
        }
    }


    /**
     * Writes a log message recording barfage.
     * @param rtX barfage-in-question
     * @return incoming (no substitution done)
     **/
    private RuntimeException recordThrowOccured(RuntimeException rtX)
    {
        int loglevel = Project.MSG_ERR;
        if (!isHaltIfError()) {
            loglevel = Project.MSG_INFO;
        }
        log(uistrs().get("flow.caught.failure",
                         getOwningTarget().getName(), rtX.getMessage()),
            loglevel);
        return rtX;
    }



    private boolean m_haltIfError= true; //NB: unrolls and barfs by default
    private boolean m_allRTX;//NB: only BuildExceptions by default
    private IfErrorTask m_iferrorTask;
    private AlwaysTask  m_alwaysTask;


    /**
     * Controls the amount of peek-under for UnknownElement placeholders
     * nested inside switch tasks.
     * @since JWare/AntX 0.4
     **/
    private static final Class[] COI_= {
        IfErrorTask.class, AlwaysTask.class
    };


    /**
     * Returns a list of possible special task class names for this
     * taskset class.
     * @since JWare/AntX 0.4
     **/
    protected Class[] getCOI()
    {
        return COI_;
    }
}

/* end-of-TolerantTaskSet.java */
