/**
 * $Id: ChoiceType.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.match;

import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * Helper enum that represents the known AntX choice types. 
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    impl,helper
 **/

final class ChoiceType extends EnumSkeleton
{
    /** Index of {@linkplain #NONE NONE}. **/
    static final int NONE_INDEX = 0;
    /** Index of {@linkplain #SAME SAME}. **/
    static final int SAME_INDEX = NONE_INDEX+1;
    /** Index of {@linkplain #LIKE LIKE}. **/
    static final int LIKE_INDEX = SAME_INDEX+1;
    /** Index of {@linkplain #MEETS MEETS}. **/
    static final int MEETS_INDEX = LIKE_INDEX+1;


    /** Singleton "<span class="src">none</span>" choice. **/
    static final ChoiceType NONE=
        new ChoiceType("none",NONE_INDEX,false);

    /** Singleton "<span class="src">same</span>" choice. **/
    static final ChoiceType SAME=
        new ChoiceType("equals",SAME_INDEX,true);

    /** Singleton "<span class="src">like</span>" choice. **/
    static final ChoiceType LIKE=
        new ChoiceType("like",LIKE_INDEX,true);

    /** Singleton "<span class="src">meets</span>" choice. **/
    static final ChoiceType MEETS=
        new ChoiceType("meets",MEETS_INDEX,false);


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private ChoiceType(String v, int i, boolean landr)
    {
        super(v);
        m_landr = landr;
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"none", "equals", "like", "meets"};
    };



    /**
     * Returns whether this choice type needs both a known and
     * unknown value for evaluation.
     **/
    boolean needsLeftAndRight()
    {
        return m_landr;
    }


    /**
     * Helper that converts a string to a known ChoiceType
     * singleton. Returns <i>null</i> if string unrecognized.
     **/
    static ChoiceType from(Class c)
    {
        if (c!=null) {
            if (MatchEquals.class.isAssignableFrom(c)) {
                return SAME;
            }
            if (MatchLike.class.isAssignableFrom(c)) {
                return LIKE;
            }
            if (MatchCondition.class.isAssignableFrom(c)) {
                return MEETS;
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(Class) from(Class)} but with a
     * default value if supplied value does not match any known
     * ChoiceTask implementation.
     * @param s the symbolic name to be matched
     * @param dflt the default ChoiceType if necessary
     **/
    static ChoiceType from(Class c, ChoiceType dflt)
    {
        ChoiceType choice= from(c);
        return (choice==null) ? dflt : choice;
    }


    private final boolean m_landr;//default for most types
}

/* end-of-ChoiceType.java */