/**
 * $Id: ForEachTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.io.BufferedReader;
import  java.io.File;
import  java.util.List;
import  java.util.Iterator;
import  java.util.NoSuchElementException;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DirectoryScanner;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.DirSet;
import  org.apache.tools.ant.types.FileSet;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.flowcontrol.HardLimitEnabled;
import  com.idaremedia.antx.helpers.ArrayIterator;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.starters.ListFriendly;

/**
 * A caller extension that adds a looping mechanism for executing a set of steps or
 * targets. This task is the internal implementation; only the common loop parameters
 * are defined. Additional script-facing options must be provided by a subclass like
 * {@linkplain CallForEachTask}. A special cursor property is passed to each step
 * or target; this property's value contains the current iteration identifier (which
 * is based on the kind-of loop control defined). This task is an extension of the
 * simple {@linkplain OnceTask} that adds a looping decorator around
 * the call-in-sequence functionality of the OnceTask.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 **/

public abstract class ForEachTask extends OnceTask implements TrimEnabled, HardLimitEnabled
{
    /**
     * Initialzies a new ForEachTask instance.
     **/
    protected ForEachTask()
    {
        super(AntX.flow+"foreach");
    }


    /**
     * Initializes a new CV-labeled ForEachTask instance.
     * @param iam CV-label (non-null)
     **/
    protected ForEachTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets this foreach task's cursor's property name. This cursor property is
     * automatically passed to each step or target as a readonly system property
     * (it's refreshed every iteration).
     * @param cursorName cursor's property name (non-null)
     **/
    public void setCursor(String cursorName)
    {
        require_(!Tk.isWhitespace(cursorName),"setCursor- nonwspc cursor");
        m_cursorName = cursorName;
    }


    /**
     * Synonym for {@linkplain #setCursor setCursor}; convenient short-hand.
     **/
    public final void setI(String cursorName)
    {
        setCursor(cursorName);
    }


    /**
     * Returns this task's cursor's property name. Will return <i>null</i>
     * if never set.
     **/
    public String getCursorName()
    {
        return m_cursorName;
    }


//  ---------------------------------------------------------------------------------------------------------|
//   Optional Hard Loop Stopper (Script-Facing)
//  ---------------------------------------------------------------------------------------------------------|

    /**
     * Gives this loop task a hard limit to the number of times
     * it can execute. This attribute serves mostly as an error
     * handling mechanism to prevent infinite loops.
     * @since JWare/AntX 0.5
     **/
    public void setMaxLoops(int maxLoops)
    {
        m_maxLoops = maxLoops>=0 ? maxLoops : Integer.MAX_VALUE;
    }



    /**
     * Returns <i>true</i> if this loop tasks has a hard limit.
     * @since JWare/AntX 0.5
     **/
    public final boolean hasMax()
    {
        return m_maxLoops!=Integer.MAX_VALUE;
    }



    /**
     * Returns this loop tasks hard limit or <span class="src">-1</span>
     * if no such limit exists.
     * @since JWare/AntX 0.5
     **/
    public final int getMaxLoops()
    {
        return hasMax() ? m_maxLoops : -1;
    }



    /**
     * Tells this tasks whether to fail if the hard limit is
     * reached. Ignored if this loop task does not have a hard limit.
     * @see #setMaxLoops setMaxLoops(LIMIT)
     * @since JWare/AntX 0.5
     **/
    public void setHaltIfMax(boolean halt)
    {
        m_haltIfMax = halt ? Boolean.TRUE : Boolean.FALSE;
    }



    /**
     * Returns this task's fail-if-max option setting. Will
     * return <i>null</i> if never set explicitly. By default this
     * task will <em>not</em> fail if its hard limit is hit; it
     * just stops.
     * @since JWare/AntX 0.5
     **/
    public final Boolean getHaltIfMax()
    {
        return m_haltIfMax;
    }

//---------------------------------------------------------------------------------------------------------|
// Possible Loop-Controls (Script-Facing)
//---------------------------------------------------------------------------------------------------------|

    /**
     * Adds a list-based loop control declaration to this task.
     * @param list the delimited-list of names (non-null)
     * @see #setDelim
     **/
    public void setList(String list)
    {
        require_(list!=null,"setList- nonzro list");
        m_byList = list;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's list-based loop control declaration. Returns
     * <i>null</i> if never set.
     **/
    public final String getListLoopControl()
    {
        return m_byList;
    }


    /**
     * Sets a custom list delimiter character. Used in conjunction
     * with a list-based loop control declaration. Use the special
     * <span class="src">[:space:]</span> delimiter to use the standard
     * whitespace set of tokens (space, tab, newline, carriage-return).
     * @param listDelim the delimiter string (non-null)
     **/
    public void setDelim(String listDelim)
    {
        require_(listDelim!=null && listDelim.length()>0,"setDelim- nonzro delim");
        if ("[:space:]".equals(listDelim)) {
            m_byListDelim = " \t\n\r";//NB:jdk standard-formfeed(AntX0.5)
        } else {
            m_byListDelim = listDelim;
        }
    }


    /**
     * Returns the list delimiter this task uses to parse list-based
     * declarations. Never returns <i>null</i>; defaults to a comma (,).
     **/
    public final String getListDelim()
    {
        return m_byListDelim;
    }


    /**
     * Adds a fileset-based loop control declaration to this task. The
     * FileSet is not determined until this task is actually executed.
     * @param filesetRef the refid of an existing fileset (non-null)
     **/
    public void setFiles(Reference filesetRef)
    {
        require_(filesetRef!=null && filesetRef.getRefId()!=null,"setfils- nonzro ref");
        m_byFilesRef= filesetRef;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's fileset-based loop control declaration.
     * Returns <i>null</i> if never set.
     **/
    public final Reference getFilesetLoopControl()
    {
        return m_byFilesRef;
    }


    /**
     * Adds a dirset-based loop control declaration to this task.
     * The DirSet is not determined until this task is actually executed.
     * @param dirsetRef the refid of an existing dirset (non-null)
     **/
    public void setDirs(Reference dirsetRef)
    {
        require_(dirsetRef!=null && dirsetRef.getRefId()!=null,"setdirs- nonzro ref");
        m_byDirsRef= dirsetRef;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's dirset-based loop control declaration.
     * Returns <i>null</i> if never set.
     **/
    public final Reference getDirsetLoopControl()
    {
        return m_byDirsRef;
    }


    /**
     * Adds a basename-only modifier to fileset based loops.
     * @param only <i>true</i> if basename only
     * @since JWare/AntX 0.4
     **/
    public void setBaseNameOnly(boolean only)
    {
        m_basenameOnly = only;
    }


    /**
     * Adds a file-based loop control declaration to this task. The
     * line-by-line contents of the file are interpreted as the items
     * in a list to be iterated. Note unless explicitly changed,
     * the entire line is passed as the cursor-- this task will do no
     * automatic trimming or special interpretation of the line before
     * passing it to the target.
     * @param file path to an existing file (non-null)
     * @see #setTrim
     * @see #setIgnorePrefix
     **/
    public void setInFile(File file)
    {
        require_(file!=null,"setInFil- nonzro file");
        m_byListFile = file;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's file-based loop control declaration.
     * Returns <i>null</i> if never set.
     **/
    public final File getFileContentsLoopControl()
    {
        return m_byListFile;
    }


    /**
     * Tells this task to trim lines returned from the file loop
     * control.
     * @since JWare/AntX 0.3
     **/
    public void setTrim(boolean trim)
    {
        m_byfileModifiers.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if this task will trim every line
     * returned by the file loop control. Only used if a 'infile'
     * loop-control defined.
     * @since JWare/AntX 0.3
     **/
    public final boolean willTrim()
    {
        return m_byfileModifiers.isTrimmed();
    }


    /**
     * Tells this task to ignore lines that begin with a particular
     * comment character.
     * @param prefix the prefix string
     * @since JWare/AntX 0.3
     **/
    public void setIgnorePrefix(String prefix)
    {
        m_byfileModifiers.setIgnorePrefix(prefix);
    }


    /**
     * Returns the prefix used to filter valid input lines for the
     * 'infile' looping control. Returns <i>null</i> if never set
     * or there is not special filter prefix. Only used if a 'infile'
     * loop-control defined.
     * @since JWare/AntX 0.3
     **/
    public final String getFileContentsIgnorePrefix()
    {
        return m_byfileModifiers.getPrefixFilter();
    }


    /**
     * Adds a C-style scalar-range loop control declaration to
     * this task. The string is a comma-delimited triple: <i>start</i>,
     * <i>end+1</i>[,<i>increment</i>]. If unspecified the increment
     * is set to one. The loop executes end-start/increment times; i.e
     * it is  exclusive of the <i>end</i>.
     * @param loopDefn the defintion of the start,end,and increment (non-null)
     **/
    public void setIn(String loopDefn)
    {
        require_(loopDefn!=null,"setIn- nonzro lopvars");
        m_byIn = loopDefn;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's scalar range loop control. Returns <i>null</i>
     * if never set.
     **/
    public final String getForRangeLoopControl()
    {
        return m_byIn;
    }



    /**
     * Adds a path-based loop control declaration to this task. The
     * path's individual elements are passed as the cursor's value during
     * the iteration.
     * @param path the path list (non-null)
     **/
    public void setPathList(Path path)
    {
        require_(path!=null,"setpathlst- nonzro path");
        m_byPath = path;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's path-based loop control. Returns <i>null</i>
     * if never set.
     **/
    public final Path getPathListLoopControl()
    {
        return m_byPath;
    }


    /**
     * Adds a pathref-based loop control declaration to this task. Exactly
     * like {@linkplain #setPathList setPathList} but using a reference to
     * an existing path declaration (instead of inlined like setPathList).
     * @param pathRef the refid of an existing path (non-null)
     **/
    public void setPath(Reference pathRef)
    {
        require_(pathRef!=null && pathRef.getRefId()!=null,"setpathRef- nonzro ref");
        m_byPathRef = pathRef;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's pathref-based loop control. Returns <i>null</i>
     * if never set.
     **/
    public final Reference getPathLoopControl(Reference pathRef)
    {
        return m_byPathRef;
    }


    /**
     * Adds a stringlist-based loop control declaration to this task.
     * @param itemsRef reference to an existing list-friendly data object
     * @since JWare/AntX 0.3
     **/
    public void setItems(Reference itemsRef)
    {
        require_(itemsRef!=null && itemsRef.getRefId()!=null,"setItms- nonzro ref");
        m_byListRef = itemsRef;
        resetCachedLoopInfo();
    }


    /**
     * Returns this task's list-friendly loop control declaration.
     * Returns <i>null</i> if never set.
     * @since JWare/AntX 0.3
     **/
    public final Reference getItemsLoopControl()
    {
        return m_byListRef;
    }


    /**
     * Resets this task's internal loop-control cached information.
     **/
    protected final void resetCachedLoopInfo()
    {
        if (m_foreachList!=null || m_byInStart!=Tk.NO_INT) {
            m_foreachList= null;
            m_byInStart= Tk.NO_INT;
            m_byInEnd= Tk.NO_INT;
            m_byInIncr= Tk.NO_INT;
        }
    }

//---------------------------------------------------------------------------------------------------------|
// Loop-Control -> Iterator Handlers
//---------------------------------------------------------------------------------------------------------|

    private static final String BROKEN_CONTROL_MSG = "flow.foreach.err.bad.loopcontrol";
    private static final String BROKEN_SETUP_MSG = "flow.foreach.invalid.setup";


    /**
     * Converts the '<i>list</i>' and '<i>items</i>' loop controls to an
     * array of strings.
     * @return <i>true</i> if list specified and parsed successfully
     **/
    protected boolean handleStringLists()
    {
        if (m_foreachList==null) {
            if (m_byList!=null) {
                if (!Tk.isWhitespace(m_byList)) {
                    List l= Tk.splitList(m_byList,m_byListDelim);
                    if (!l.isEmpty()) {
                        m_foreachList= (String[])l.toArray(Empties.EMPTY_STRING_ARRAY);
                        log("Will do "+l.size()+" items for foreach(list)",Project.MSG_DEBUG);
                    } else {
                        String ermsg= uistrs().get(BROKEN_CONTROL_MSG,"list","");
                        log(ermsg,Project.MSG_ERR);
                    }
                } else {
                    String ermsg= uistrs().get(BROKEN_CONTROL_MSG,"list","");
                    log(ermsg,Project.MSG_WARN);
                    m_foreachList = new String[0];
                }
            }
            else if (m_byListRef!=null) {
                Object o = getProject().getReference(m_byListRef.getRefId());
                if (!(o instanceof ListFriendly)) {
                    String ermsg = getAntXMsg("task.bad.refid",m_byListRef.getRefId(),
                                               ListFriendly.class.getName(),
                                              (o==null ? "NULL" : o.getClass().getName()));
                    log(ermsg,Project.MSG_ERR);
                } else {
                    ListFriendly list = (ListFriendly)o;
                    String[] sl= new String[list.size()];
                    if (sl.length>0) {
                        int i=0;
                        Iterator itr= list.readonlyStringIterator(getProject());
                        while (itr.hasNext()) {
                            verify_(i<sl.length,"handlLst- no-concur-mods on roItr");
                            sl[i++] = itr.next().toString();
                        }
                    }
                    m_foreachList = sl;
                    log("Will do "+sl.length+" items for foreach(items)",Project.MSG_DEBUG);
                }
            }
        }
        return m_foreachList!=null;
    }


    /**
     * Converts the '<i>in</i>' loop controls to a set of numeric loop controls.
     * @return <i>true</i> if 'in' parameters parsed and converted successfully
     **/
    protected boolean handleIn()
    {
        if (m_byInStart==Tk.NO_INT) {
            if (!Tk.isWhitespace(m_byIn)) {
                List l= Tk.splitList(m_byIn,",");
                int istart=Tk.NO_INT,iend=Tk.NO_INT,iincr=Tk.NO_INT;
                if (l.size()>=2) {
                    istart= Tk.integerFrom(l.get(0),Tk.NO_INT);
                    iend= Tk.integerFrom(l.get(1),Tk.NO_INT);
                }
                if (l.size()>=3) {
                    iincr = Tk.integerFrom(l.get(2),Tk.NO_INT);
                    if (iincr==Tk.NO_INT) {
                        String ermsg= uistrs().get("flow.foreach.err.in.incr",l.get(2));
                        log(ermsg,Project.MSG_ERR);
                        return false;
                    }
                }
                if (istart!=Tk.NO_INT && iend!=Tk.NO_INT) {
                    m_byInStart = istart;
                    m_byInEnd = iend;
                    if (iincr!=Tk.NO_INT) {
                        m_byInIncr= iincr;
                    } else {
                        m_byInIncr= 1;
                    }
                    int N = (iend-istart)/iincr;
                    log("Will do "+N+" iterations for foreach(in)",Project.MSG_DEBUG);
                } else {
                    String ermsg= uistrs().get(BROKEN_CONTROL_MSG,"in","");
                    log(ermsg,Project.MSG_ERR);
                }
            }
        }
        return m_byInStart!=Tk.NO_INT;
    }


    /**
     * Converts the '<i>files</i>' loop controls to a set of string file names.
     * @return <i>true</i> if fileset specified and converted successfully
     **/
    protected boolean handleFiles()
    {
        if (m_foreachList==null) {
            if (m_byFilesRef!=null) {
                try {
                    FileSet fs = new FileSet();
                    fs.setProject(getProject());
                    fs.setRefid(m_byFilesRef);
                    DirectoryScanner dirscan = fs.getDirectoryScanner(getProject());
                    m_foreachList= dirscan.getIncludedFiles();
                    resolveAllFilesOrDirs(m_foreachList, dirscan);
                    log("Will do "+m_foreachList.length+" files for foreach(files)",Project.MSG_DEBUG);
                } catch(BuildException bx) {
                    String ermsg= uistrs().get(BROKEN_CONTROL_MSG,"files",bx.getMessage());
                    log(ermsg,Project.MSG_ERR);
                }
            }
        }
        return m_foreachList!=null;
    }


    /**
     * Converts the '<i>dirs</i>' loop controls to a set of string directory names.
     * @return <i>true</i> if dirset specified and converted successfully
     **/
    protected boolean handleDirs()
    {
        if (m_foreachList==null) {
            if (m_byDirsRef!=null) {
                try {
                    DirSet ds = new DirSet();
                    ds.setProject(getProject());
                    ds.setRefid(m_byDirsRef);
                    DirectoryScanner dirscan = ds.getDirectoryScanner(getProject());
                    m_foreachList= dirscan.getIncludedDirectories();
                    resolveAllFilesOrDirs(m_foreachList, dirscan);
                    log("Will do "+m_foreachList.length+" dirs for foreach(dirs)",Project.MSG_DEBUG);
                } catch(BuildException bx) {
                    String ermsg = uistrs().get(BROKEN_CONTROL_MSG,"dirs",bx.getMessage());
                    log(ermsg,Project.MSG_ERR);
                }
            }
        }
        return m_foreachList!=null;
    }


    /**
     * Converts the '<i>path</i>' and '<i>pathlist</i>' loop controls
     * to a set of strings.
     * @return <i>true</i> if path controls parsed and converted successfully
     **/
    protected boolean handlePaths()
    {
        if (m_foreachList==null) {
            if (m_byPath!=null) {
                try {
                    m_foreachList = m_byPath.list();
                    log("Will do "+m_foreachList.length+" elements for foreach(path)",Project.MSG_DEBUG);
                } catch(BuildException bx) {
                    String ermsg = uistrs().get(BROKEN_CONTROL_MSG,"path",bx.getMessage());
                    log(ermsg,Project.MSG_ERR);
                }
            }
            else if (m_byPathRef!=null) {
                try {
                    Path path = new Path(getProject());
                    path.setRefid(m_byPathRef);
                    m_foreachList = path.list();
                    log("Will do "+m_foreachList.length+" elements for foreach(pathref)",Project.MSG_DEBUG);
                } catch(BuildException bx) {
                    String ermsg = uistrs().get(BROKEN_CONTROL_MSG,"pathref",bx.getMessage());
                    log(ermsg,Project.MSG_ERR);
                }
            }
        }
        return m_foreachList!=null;
    }


    /**
     * Converts the '<i>infile</i>' loop control to a set of strings.
     * @return <i>true</i> if file specified and read successfully
     **/
    protected boolean handleFileContents()
    {
        if (m_foreachList==null) {
            if (m_byListFile!=null && m_byListFile.canRead()) {
                try {
                    BufferedReader fr = new BufferedReader(new java.io.FileReader(m_byListFile));
                    List propList = AntXFixture.newList(10);
                    Project P= getProject();
                    String next;
                    FlexString filter = (FlexString)m_byfileModifiers.clone();
                    while ((next=fr.readLine())!=null) {
                        filter.set(next);
                        next = filter.getValue();
                        if (next!=null) {
                            propList.add(Tk.resolveString(P,next));
                        }
                    }
                    m_foreachList = (String[])propList.toArray(Empties.EMPTY_STRING_ARRAY);
                    log("Will do "+m_foreachList.length+" items for foreach(infile)",Project.MSG_DEBUG);
                    propList = null;
                } catch(Exception ioxOrbuildX) {
                    String ermsg = uistrs().get(BROKEN_CONTROL_MSG,"infile",ioxOrbuildX.getMessage());
                    log(ermsg,Project.MSG_ERR);
                }
            }
        }
        return m_foreachList!=null;
    }


    /**
     * Ensures all file names from a directory scanner are
     * fully-resolved absolute path names.
     **/
    private void resolveAllFilesOrDirs(String[] filesOrDirs, DirectoryScanner dirscan)
    {
        FileUtils fileUtils = AntXFixture.fileUtils();
        File wrt = dirscan.getBasedir();
        for (int i=0;i<filesOrDirs.length;i++) {
            File rf= fileUtils.resolveFile(wrt,filesOrDirs[i]);
            if (m_basenameOnly) {
                filesOrDirs[i]= rf.getName();
            } else {
                filesOrDirs[i]= rf.getAbsolutePath();
            }
        }
    }


    /**
     * Returns the underlying array of strings to be iterated <em>if</em>
     * such a list is available. Will return <i>null</i> if this task
     * never executed or the loop control doesn't convert to a simple
     * string array.
     **/
    protected final String[] getForEachList()
    {
        return m_foreachList;
    }


    /**
     * Sets the (dominant) string array to be iterated. Usually called from
     * one of the 'handle' methods after the loop control has been converted
     * to a concrete list of strings.
     **/
    protected final void setForEachList(String[] newlist)
    {
        m_foreachList = newlist;
    }


//---------------------------------------------------------------------------------------------------------|
// Loop Execution
//---------------------------------------------------------------------------------------------------------|

    /**
     * Returns <i>true</i> if at least one loop control
     * is defined.
     **/
    protected final boolean anyLoopControlsSpecified()
    {
        return (m_byList!=null     || m_byListRef!=null ||
                m_byFilesRef!=null || m_byDirsRef!=null ||
                m_byListFile!=null || m_byPath!=null    ||
                m_byPathRef!=null  || m_byIn!=null
                );
    }


    /**
     * Returns the 'effective' number of loop controls defined
     * for this task. Currently only one kind is allowed.
     **/
    protected final int getNumberLoopControlsSpecified()
    {
        int N=0;
        if (m_byList!=null || m_byListRef!=null) { N++; }//both=1
        if (m_byIn!=null) { N++; }
        if (m_byFilesRef!=null) { N++; }
        if (m_byDirsRef!=null) { N++; }
        if (m_byListFile!=null) { N++; }
        if (m_byPath!=null || m_byPathRef!=null) { N++; }//both=1
        return N;
    }


    /**
     * Ensures some form of loop control and cursor has been defined
     * for this task. Also verifies that all the inherited requirements
     * have been met.
     * @throws BuildException if none/too-many loop controls or no
     *         cursor defined or other inherited prerequiste violated
     **/
    protected void verifyCanExecute_(String calr)
        throws BuildException
    {
        super.verifyCanExecute_(calr);

        int Nc= getNumberLoopControlsSpecified();

        if ((getCursorName()==null)  ||      //required for steps
            (Nc==0) || (Nc>1)) {             //only one loop-control

            String ermsg = uistrs().get(BROKEN_SETUP_MSG);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Executes this task's set of steps or targets for each element
     * returned by an iterator. The iterator's value is passed to
     * target using this task's cursor name setting-- the iterator's
     * value is always converted to a string using the standard
     * <span class="src">toString</span> method.
     * @param itr iterator (non-null)
     **/
    protected final void forEachRunTargetCallers(Iterator itr)
        throws BuildException
    {
        final String cursorName = getCursorName();
        int ith=0;
        final int Nth=m_maxLoops;

        while (itr.hasNext() && ith<Nth) {
            final String cursor = itr.next().toString();
            try {
                runTargetCallers(new TargetCaller.RunSelector() {
                        public void run(TargetCaller caller) {
                            caller.run(cursorName, cursor);
                        }
                    });
                ith++;
            } catch(RuntimeException rtX) {
                log(uistrs().get("flow.cursor.failure",cursor),
                    Project.MSG_ERR);
                throw rtX;
            }
        }

        if (ith==Nth) {
            String msg = getMsg();
            if (Tk.isWhitespace(msg)) {
                msg = getAntXMsg("flow.loop.overflow",String.valueOf(ith));
            }
            if (m_haltIfMax==Boolean.TRUE) {
                log(msg, Project.MSG_ERR);
                throw new BuildException(msg, getLocation());
            }
            log(msg, Project.MSG_INFO);
        }
    }


    /**
     * Determines what we're actually iterating over based on one of the
     * kinds-of loop controls then iterates calling the targetted steps
     * or targets for each iteration. All but the 'in' control produces a
     * list of strings for iteration. An iterator is used to mask the difference
     * between the 'in' control and all others. This iterator is passed to
     * {@linkplain #forEachRunTargetCallers(Iterator)
     * forEachRunTargetCallers(Iterator)}.
     * @throws BuildException if unable to create a list for iteration
     * @see IntRangeIterator
     **/
    protected void forEachRunTargetCallers() throws BuildException
    {
        if (!handleStringLists()  && /*ugly-but-works*/
            !handleIn()           &&
            !handleFileContents() &&
            !handleFiles()        &&
            !handlePaths()        &&
            !handleDirs()) {

            String ermsg = uistrs().get(BROKEN_SETUP_MSG);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }

        if (m_foreachList!=null) {
            forEachRunTargetCallers(new ArrayIterator(m_foreachList));
        }
        else if (m_byInStart!=Tk.NO_INT) {
            forEachRunTargetCallers
                (new IntRangeIterator(m_byInStart,m_byInEnd,m_byInIncr));
        }
        else {
            String ermsg = uistrs().get(BROKEN_SETUP_MSG);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * For each loop iteration generates a series of partitioned calls to
     * either a set of real top-level targets or a set of nested steps.
     * Both kinds of calls actually execute a set of &lt;antcall&gt;
     * helpers to do most of the work.
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");
        forEachRunTargetCallers();
    }


    /**
     * Iterator of a range of integers. Used with 'in' loop controls.
     * @since    JWare/AntX 0.1
     * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  single
     **/
    public static final class IntRangeIterator implements Iterator
    {
        private final int end, incr;
        private int i;
        public IntRangeIterator(int start, int end, int incr) {
            this.i = start;
            this.end= end;
            this.incr = incr;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
        public boolean hasNext() {
            return i<end;
        }
        public Object next() {
            if (i<end) {
                Object iObject= String.valueOf(i);
                i += incr;
                return iObject;
            }
            throw new NoSuchElementException();
        }
    }

//---------------------------------------------------------------------------------------------------------|

    //Required cursor name as property
    private String m_cursorName;

    //Required actual loop-controls is either
    private String[] m_foreachList;//this-or
    private int m_byInStart=Tk.NO_INT, m_byInEnd=Tk.NO_INT, m_byInIncr=Tk.NO_INT;//this

    //Useful ways to specify loop-controls
    private String m_byList, m_byListDelim= AntX.DEFAULT_DELIMITER;
    private String m_byIn;
    private Reference m_byFilesRef, m_byDirsRef, m_byPathRef;
    private File m_byListFile;
    private FlexString m_byfileModifiers = new FlexString();
    private Path m_byPath;
    private Reference m_byListRef;
    private boolean m_basenameOnly;
    
    //Optional hard-brake for runaway loops (AntX0.5)
    private int m_maxLoops = Integer.MAX_VALUE;
    private Boolean m_haltIfMax = null;//NB: => no
}

/* end-of-ForEachTask.java */
